<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2018 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace LifeStyle\ActivityLogBundle\Controller;

use JMS\SecurityExtraBundle\Annotation\Secure;
use JMS\DiExtraBundle\Annotation as DI;
use LifeStyle\ActivityLogBundle\Annotation\ActivityLog;
use LifeStyle\ActivityLogBundle\Exception\InvalidRequestException;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Route;
use Symfony\Bundle\FrameworkBundle\Controller\Controller;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Knp\Component\Pager\Paginator;
use LifeStyle\ActivityLogBundle\Database\UserActionLogManager;

/**
 * Class DefaultController
 *
 * @copyright  2018 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package    LifeStyle\ActivityLogBundle\Controller
 *
 * @ActivityLog(type="superadmin")
 */
class DefaultController extends Controller
{
    /**
     * @var UserActionLogManager
     *
     * @DI\Inject("life_style_activity_log.database.user_action_log_manager")
     */
    private $databaseManager;

    /**
     * @var Paginator
     *
     * @DI\Inject("knp_paginator")
     */
    private $knpPaginator;

    /**
     * @Secure(roles="ROLE_USER")
     *
     * @Route("/admin/activitylog", name="admin_activitylog_list")
     * @param Request $request
     * @return Response
     */
    public function listAction(Request $request)
    {
        $query = $this->databaseManager->getListQuery();
        $pagination = $this->knpPaginator->paginate($query, $request->query->get('page', 1), 10);

        return $this->render('LifeStyleActivityLogBundle:Default:list.html.twig', [
            'activityLogs' => $pagination
        ]);
    }

    /**
     * @Secure(roles="ROLE_USER")
     *
     * @Route("/admin/activitylog/delete", name="admin_activitylog_delete")
     * @return Response
     * @throws InvalidRequestException
     */
    public function deleteAllAction()
    {
        // For development only
        if ($this->get('kernel')->isDebug()) {
            $this->databaseManager->deleteAll();
            $this->addFlash('info', 'Activity log deleted successfully!');
        }

        return $this->redirectToRoute('admin_activitylog_list');
    }

    /**
     * @Secure(roles="ROLE_USER")
     *
     * @Route("/admin/activitylog/{userGuid}", name="admin_activitylog_list_by_user")
     * @param Request $request
     * @param string $userGuid
     * @return Response
     * @throws InvalidRequestException
     */
    public function listByUserAction(Request $request, $userGuid)
    {
        // This should only be used as sub-request
        if (null === $this->get('request_stack')->getParentRequest()) {
            throw new InvalidRequestException('This request should only used as a sub-request.');
        }

        $query = $this->databaseManager->getListByUserQuery($userGuid);
        $pagination = $this->knpPaginator->paginate($query, $request->query->get('page', 1), 10);

        return $this->render('LifeStyleActivityLogBundle:Default:listByUser.html.twig', [
            'activityLogs' => $pagination
        ]);
    }
}
