<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2018 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace LifeStyle\ActivityLogBundle\Entity;

use Doctrine\ORM\Mapping as ORM;
use LifeStyle\SamlAuthBundle\Entity\User;
use Ramsey\Uuid\Uuid;

/**
 * Class UserActionLog
 * @package LifeStyle\ActivityLogBundle\Entity
 *
 * @ORM\Entity
 * @ORM\Table(
 *     name="saml_user_action_log",
 *     indexes={
 *         @ORM\Index(name="idx_created", columns={"created_at", "local_user_id", "controller", "action", "type", "ip_address"}),
 *     })
 */
class UserActionLog
{
    /**
     * @ORM\Column(name="id", type="uuid_binary")
     * @ORM\Id
     * @ORM\GeneratedValue(strategy="NONE")
     * @var Uuid
     */
    private $keyId;

    /**
     * @ORM\ManyToOne(targetEntity="LifeStyle\SamlAuthBundle\Entity\User", inversedBy="userActionLog")
     * @ORM\JoinColumns({
     *   @ORM\JoinColumn(name="local_user_id", referencedColumnName="id")
     * })
     * @var User
     */
    private $user;

    /**
     * @ORM\Column(name="local_user_id", type="guid", nullable=false)
     * @var string
     */
    private $localUserId;

    /**
     * @ORM\Column(name="controller", type="string", nullable=false, length=127)
     * @var string
     */
    private $controller;

    /**
     * @ORM\Column(name="action", type="string", nullable=false, length=127)
     * @var string
     */
    private $action;

    /**
     * @ORM\Column(name="type", type="string", nullable=false, length=32)
     * @var string
     */
    private $type;

    /**
     * @ORM\Column(name="created_at", type="datetime", nullable=false)
     * @var \DateTime
     */
    private $createdAt;

    /**
     * @ORM\OneToOne(targetEntity="UserActionLogData", mappedBy="userActionLog")
     * @var UserActionLogData
     */
    private $userActionLogData;

    /**
     * @ORM\Column(name="ip_address", type="string", nullable=true, length=32)
     * @var string
     */
    private $ipAddress;

    /**
     * @ORM\Column(name="stats", type="array", nullable=true)
     * @var array
     */
    private $stats = [];

    /**
     * UserActionLog constructor.
     */
    public function __construct()
    {
        $this->keyId = Uuid::uuid4();
        $this->createdAt = new \DateTime();
    }

    /**
     * @return Uuid
     */
    public function getKeyId()
    {
        return $this->keyId;
    }

    /**
     * @return User
     */
    public function getUser()
    {
        return $this->user;
    }

    /**
     * @param User $user
     * @return UserActionLog
     */
    public function setUser($user)
    {
        $this->user = $user;
        return $this;
    }

    /**
     * @return string
     */
    public function getLocalUserId()
    {
        return $this->localUserId;
    }

    /**
     * @param string $localUserId
     * @return UserActionLog
     */
    public function setLocalUserId($localUserId)
    {
        $this->localUserId = $localUserId;
        return $this;
    }

    /**
     * @return string
     */
    public function getController()
    {
        return $this->controller;
    }

    /**
     * @param string $controller
     * @return UserActionLog
     */
    public function setController($controller)
    {
        $this->controller = $controller;
        return $this;
    }

    /**
     * @return string
     */
    public function getAction()
    {
        return $this->action;
    }

    /**
     * @param string $action
     * @return UserActionLog
     */
    public function setAction($action)
    {
        $this->action = $action;
        return $this;
    }

    /**
     * @return string
     */
    public function getType()
    {
        return $this->type;
    }

    /**
     * @param string $type
     * @return UserActionLog
     */
    public function setType($type)
    {
        $this->type = $type;
        return $this;
    }

    /**
     * @return \DateTime
     */
    public function getCreatedAt()
    {
        return $this->createdAt;
    }

    /**
     * @return UserActionLogData
     */
    public function getUserActionLogData()
    {
        return $this->userActionLogData;
    }

    /**
     * @param UserActionLogData $userActionLogData
     * @return UserActionLog
     */
    public function setUserActionLogData(UserActionLogData $userActionLogData)
    {
        $this->userActionLogData = $userActionLogData;
        $userActionLogData->setUserActionLog($this);
        return $this;
    }

    /**
     * @return string
     */
    public function getRequest()
    {
        return null !== $this->userActionLogData ? $this->userActionLogData->getRequest() : '';
    }

    /**
     * @return string
     */
    public function getResponse()
    {
        return null !== $this->userActionLogData ? $this->userActionLogData->getResponse() : '';
    }

    /**
     * @return string
     */
    public function getIpAddress()
    {
        return $this->ipAddress;
    }

    /**
     * @param string $ipAddress
     * @return UserActionLog
     */
    public function setIpAddress($ipAddress)
    {
        $this->ipAddress = $ipAddress;
        return $this;
    }

    /**
     * @return mixed
     */
    public function getStats()
    {
        return $this->stats;
    }

    /**
     * @param mixed $stats
     * @return UserActionLog
     */
    public function setStats($stats)
    {
        $this->stats = $stats;

        return $this;
    }
}
