<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2018 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace LifeStyle\SamlAuthBundle\Security\User;

use Symfony\Component\Security\Core\User\UserInterface;
use Symfony\Component\Security\Core\User\EquatableInterface;
use LifeStyle\SamlAuthBundle\Model\UserData\User as UserDto;

/**
 * Class SamlUser
 * @package LifeStyle\SamlAuthBundle\Security\User
 */
class SamlUser implements UserInterface, EquatableInterface
{
    /**
     * @var string
     */
    private $username;
    /**
     * @var null|string
     */
    private $password = null;

    /**
     * @var null|string
     */
    private $salt = null;

    /**
     * @var array
     */
    private $roles;

    /**
     * @var UserDto
     */
    private $userSamlData;

    /**
     * SamlUser constructor.
     * @param UserDto $userDto
     * @param array $roles
     */
    public function __construct(UserDto $userSamlData, array $roles)
    {
        $this->userSamlData = $userSamlData;
        $this->username = $this->userSamlData->getUsername();
        $this->roles = $roles;
    }

    /**
     * The equality comparison should neither be done by referential equality
     * nor by comparing identities (i.e. getId() === getId()).
     *
     * However, you do not need to compare every attribute, but only those that
     * are relevant for assessing whether re-authentication is required.
     *
     * Also implementation should consider that $user instance may implement
     * the extended user interface `AdvancedUserInterface`.
     *
     * @param UserInterface $user
     *
     * @return bool
     */
    public function isEqualTo(UserInterface $user)
    {
        return $this->getUsername() && $user->getUsername();
    }

    /**
     * Returns the roles granted to the user.
     *
     * <code>
     * public function getRoles()
     * {
     *     return array('ROLE_USER');
     * }
     * </code>
     *
     * Alternatively, the roles might be stored on a ``roles`` property,
     * and populated in any number of different ways when the user object
     * is created.
     *
     * @return (Role|string)[] The user roles
     */
    public function getRoles()
    {
        return $this->roles;
    }

    /**
     * @return string|null
     */
    public function getPassword()
    {
        return $this->password;
    }

    /**
     * @return string|null
     */
    public function getSalt()
    {
        return $this->salt;
    }

    /**
     * Returns the username used to authenticate the user.
     *
     * @return string The username
     */
    public function getUsername()
    {
        return $this->username;
    }

    /**
     * @return UserDto
     */
    public function getUserSamlData()
    {
        return $this->userSamlData;
    }

    /**
     * Removes sensitive data from the user.
     *
     * This is important if, at any given point, sensitive information like
     * the plain-text password is stored on this object.
     */
    public function eraseCredentials()
    {
        return false;
    }


    /**
     * @return string
     */
    public function __toString()
    {
        return $this->username;
    }
}
