<?php

/**
 * Api user controller
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2014 Lifestyle Webconsulting GmbH
 * @version    $Id:$
 * @link       http://www.life-style.de
 */

namespace Sso\ApiBundle\Api\Controller;

use Sso\ApiBundle\Api\Error\Type\External as ExternalError;
use Sso\ApiBundle\Api\Response\Builder as XmlResponse;

/**
 * Api user controller
 */
class LocationController extends AbstractController
{

    /**
     * Returns list of available actions
     * 
     * @return array
     */
    protected function _getActions()
    {
        return array(
            'Show',
        );
    }

    /**
     * Show user
     * 
     * @param \SimpleXMLElement $xmlRequest
     * @param XmlResponse $xmlResponse
     */
    protected function show(\SimpleXMLElement $xmlRequest, XmlResponse $xmlResponse)
    {

        $requestModel = $this->modelFactory->request()->location()->show($xmlRequest);
        $locationModel = $requestModel->getLocationModel();

        if (!$requestModel->hasErrors() && !($locations = $this->dbM->location()->getLocationsByLocationModel($locationModel))) {
            $requestModel->addError(new ExternalError('l011', 'LocationsNotFound'));
        }
        if (!$requestModel->hasErrors()) {
            // search location
            $requestModel->setLocationModels($locations);
        }
        $this->modelFactory->response($xmlResponse)->location()->show($requestModel);
    }

    /**
     * Add user
     * 
     * @param \SimpleXMLElement $xmlRequest
     * @param XmlResponse $xmlResponse
     */
    protected function add(\SimpleXMLElement $xmlRequest, XmlResponse $xmlResponse)
    {
        $requestModel = $this->modelFactory->request()->user()->add($xmlRequest);
        $userModel = $requestModel->getUserModel();

        if (!$requestModel->hasErrors()) {
            $userModel->generateGuid();
            if (!$userModel->getActive()) {
                $userModel->generateActivateRequestToken();
            }
            $this->dbM->user()->saveUser($userModel);
        }
        $this->modelFactory->response($xmlResponse)->user()->add($requestModel);
    }

    /**
     * Update user
     * 
     * @param \SimpleXMLElement $xmlRequest
     * @param XmlResponse $xmlResponse
     */
    protected function update(\SimpleXMLElement $xmlRequest, XmlResponse $xmlResponse)
    {
        $requestModel = $this->modelFactory->request()->user()->update($xmlRequest);
        $userModel = $requestModel->getUserModel();
        $userUpdateModel = $requestModel->getUserUpdateModel();

        if (!$requestModel->hasErrors() && !($user = $this->dbM->user()->getUserByUser($userUpdateModel))) {
            $requestModel->addError(new ExternalError('u021', 'UserNotFound'));
        }

        if (!$requestModel->hasErrors()) {
            $user->merge($userModel);
            $user->setValidator($this->apiM->validator());
            if (!$this->dbM->user()->saveUser($user)) {
                $requestModel->addErrors($user->errors()->getErrors());
            }
        }
        $this->modelFactory->response($xmlResponse)->user()->update($requestModel);
    }

    /**
     * Change Password
     * 
     * @param \SimpleXMLElement $xmlRequest
     * @param XmlResponse $xmlResponse
     */
    protected function changePassword(\SimpleXMLElement $xmlRequest, XmlResponse $xmlResponse)
    {
        $requestModel = $this->modelFactory->request()->user()->changePassword($xmlRequest);
        $userModel = $requestModel->getUserModel();
        $userUpdateModel = $requestModel->getUserUpdateModel();

        if (!$requestModel->hasErrors() && !($user = $this->dbM->user()->getUserByUser($userUpdateModel))) {
            $requestModel->addError(new ExternalError('u031', 'UserNotFound'));
        }
        if (!$requestModel->hasErrors()) {
            $user->merge($userModel);
            $user->setValidator($this->apiM->validator());
            if (!$this->dbM->user()->saveUser($user)) {
                $requestModel->addErrors($user->errors()->getErrors());
            }
        }
        $this->modelFactory->response($xmlResponse)->user()->changePassword($requestModel);
    }

    /**
     * Delete user
     * 
     * @param \SimpleXMLElement $xmlRequest
     * @param XmlResponse $xmlResponse
     */
    protected function delete(\SimpleXMLElement $xmlRequest, XmlResponse $xmlResponse)
    {
        $requestModel = $this->modelFactory->request()->user()->delete($xmlRequest);
        $userModel = $requestModel->getUserModel();

        if (!$requestModel->hasErrors() && !($user = $this->dbM->user()->getUserByUser($userModel))) {
            $requestModel->addError(new ExternalError('u041', 'UserNotFound'));
        }
        if (!$requestModel->hasErrors() && !$this->dbM->user()->deleteUser($user)) {
            $requestModel->addErrors($user->errors()->getErrors());
        }
        $this->modelFactory->response($xmlResponse)->user()->delete($requestModel);
    }

}
