<?php

/**
 * @class TokenController
 * @author sb
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2017 Lifestyle Webconsulting GmbH
 * @version    $Id:$
 * @link       http://www.life-style.de
 */

namespace Sso\BackendBundle\Controller;

use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Route;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Template;
use Sso\BackendBundle\Form;
use LifeStyle\Tools\CredentialsBundle\Entity;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;
use JMS\DiExtraBundle\Annotation\Inject;
use Symfony\Component\Form\Extension\Core\Type as FormType;

/**
 * Class TokenController
 * @package Sso\BackendBundle\Controller
 */
class TokenController extends WsAbstractController
{
    /**
     * @Inject("form.factory")
     * @var \Symfony\Component\Form\FormFactory
     */
    private $formFactory;

    /**
     * @Inject("lexik_form_filter.query_builder_updater")
     * @var \Lexik\Bundle\FormFilterBundle\Filter\FilterBuilderUpdater
     */
    private $lexikFilterUpdater;

    /**
     * @Inject("knp_paginator")
     * @var \Knp\Component\Pager\Paginator
     */
    private $knpPaginator;

    /**
     * @Route("/list", name="_admin_backend_token_list")
     *
     * @param Request $request
     * @return Response
     */
    public function listAction(Request $request)
    {
        $filterBuilder = $this->databaseManager()->serviceProvider()->token()->getRepository()->createQueryBuilder('a');
        $form = $this->formFactory->create(Form\TokenFilterType::class);

        $reset = false;
        if ($request->query->has('filter_action') && $request->query->get('filter_action') == 'reset') {
            $reset = true;
        }
        if ($request->query->has($form->getName()) && !$reset) {
            $form->submit($request->query->get($form->getName()));
            $this->lexikFilterUpdater->addFilterConditions($form, $filterBuilder);
        }
        $query = $filterBuilder->getQuery();
        $paginator = $this->knpPaginator;
        $pagination = $paginator->paginate($query, $request->query->get('page', 1), 10);

        return $this->render(
            '@SsoBackend/token/list.html.twig',
            ['form' => $form->createView(), 'pagination' => $pagination]
        );
    }

    /**
     * @Route("/edit/{tokenId}", name="_admin_backend_token_edit")
     *
     * @param string $tokenId
     * @return Response
     * @throws NotFoundHttpException
     */
    public function editAction($tokenId)
    {
        $entity = $this->tokenHandler()->getTokenByToken($tokenId);

        if (!$entity) {
            throw $this->createNotFoundException('Token not found!');
        }

        $editForm = $this->createEditForm($entity);
        $deleteForm = $this->createDeleteForm($tokenId);

        return $this->render(
            '@SsoBackend/token/edit.html.twig',
            [
                'entity' => $entity,
                'edit_form' => $editForm->createView(),
                'delete_form' => $deleteForm->createView(),
            ]
        );
    }

    /**
     * @Route("/create", name="_admin_backend_token_create")
     *
     * @param Request $request
     * @return Response
     */
    public function createAction(Request $request)
    {
        $tokenModel = new Entity\Token();

        $form = $this->createCreateForm($tokenModel);
        $form->handleRequest($request);

        if ($form->isValid()) {
            $this->tokenHandler()->saveToken($tokenModel);
            $this->get('session')->getFlashBag()->add('success', 'Token created success');
            return $this->redirect($this->generateUrl('_admin_backend_token_show',
                array('tokenId' => $tokenModel->getServiceToken())));
        }

        return $this->render(
            '@SsoBackend/token/new.html.twig',
            [
                'entity' => $tokenModel,
                'form' => $form->createView(),
            ]
        );
    }

    /**
     * @Route("/update/{tokenId}", name="_admin_backend_token_update")
     *
     * @param \Symfony\Component\HttpFoundation\Request $request
     * @param string $tokenId
     * @return Response
     */
    public function updateAction(Request $request, $tokenId)
    {
        $entity = $this->tokenHandler()->getTokenByToken($tokenId);
        if (!$entity) {
            throw $this->createNotFoundException('Token not found!');
        }

        $editForm = $this->createEditForm($entity);
        $deleteForm = $this->createDeleteForm($tokenId);
        $editForm->handleRequest($request);

        if ($editForm->isValid()) {
            $this->tokenHandler()->saveToken($entity);
            $this->get('session')->getFlashBag()->add('info', 'Token Saved Sucess');
            return $this->redirect($this->generateUrl('_admin_backend_token_edit', array('tokenId' => $tokenId)));
        } else {
            $this->get('session')->getFlashBag()->add('info', 'Token Saved Sucess');
        }

        return $this->render(
            '@SsoBackend/token/edit.html.twig',
            [
                'entity' => $entity,
                'edit_form' => $editForm->createView(),
                'delete_form' => $deleteForm->createView(),
            ]
        );
    }

    /**
     * @param Entity\Token $entity
     * @return \Symfony\Component\Form\Form
     */
    private function createEditForm(Entity\Token $entity)
    {
        $form = $this->createForm(new Form\TokenType(), $entity, array(
            'action' => $this->generateUrl('_admin_backend_token_update',
                array('tokenId' => $entity->getServiceToken())),
            'method' => 'PUT',
        ));

        $form->add('submit', 'submit', array('label' => 'Update'));

        return $form;
    }

    /**
     * @param Entity\Token $entity
     * @return \Symfony\Component\Form\Form
     */
    private function createCreateForm(Entity\Token $entity)
    {
        $form = $this->createForm(Form\TokenType::class, $entity, array(
            'action' => $this->generateUrl('_admin_backend_token_create'),
            'method' => 'POST',
        ));

        $form->add('submit', FormType\SubmitType::class, array('label' => 'Create'));
        $form->add('ServiceToken', FormType\TextType::class, array(
            'data' => $this->generateRandomToken()
        ));

        return $form;
    }

    /**
     * @param string $tokenId
     * @return \Symfony\Component\Form\FormInterface
     */
    private function createDeleteForm($tokenId)
    {
        return $this->createFormBuilder()
            ->setAction($this->generateUrl('_admin_backend_token_delete', array('tokenId' => $tokenId)))
            ->setMethod('DELETE')
            ->add('submit', FormType\SubmitType::class, array('label' => 'Delete'))
            ->getForm();
    }

    /**
     * @Route("/delete/{tokenId}", name="_admin_backend_token_delete")
     * @Template()
     *
     * @param \Symfony\Component\HttpFoundation\Request $request
     * @param string $tokenId
     * @return RedirectResponse
     * @throws NotFoundHttpException
     */
    public function deleteAction(Request $request, $tokenId)
    {
        $form = $this->createDeleteForm($tokenId);
        $form->handleRequest($request);

        $entity = $this->tokenHandler()->getTokenByToken($tokenId);

        if (!$entity) {
            throw $this->createNotFoundException('Unable to find Token entity.');
        }

        $this->tokenHandler()->deleteToken($entity);
        $this->get('session')->getFlashBag()->add('success', 'Token deleted.');

        return $this->redirect($this->generateUrl('_admin_backend_token_list'));
    }

    /**
     * @Route("/show/{tokenId}", name="_admin_backend_token_show")
     *
     * @param string $tokenId
     * @return Response
     * @throws NotFoundHttpException
     */
    public function showAction($tokenId)
    {
        $entity = $this->tokenHandler()->getTokenByToken($tokenId);

        if (!$entity) {
            throw $this->createNotFoundException('Unable to find token entity.');
        }

        $deleteForm = $this->createDeleteForm($tokenId);

        return $this->render(
            '@SsoBackend/token/show.html.twig',
            [
                'entity' => $entity,
                'delete_form' => $deleteForm->createView(),
            ]
        );
    }

    /**
     *
     * @return string 1024 Bit
     */
    protected function generateRandomToken()
    {
        // first 16 chars
        $startRandom = $this->generateRandomString(48);

        // middle 32 chars
        $time = microtime();
        $middleRandom = md5($time);

        //end 16 chars
        $endRandom = $this->generateRandomString(48);

        return $startRandom . $middleRandom . $endRandom;
    }

    /**
     * @param int $length
     * @return string
     */
    protected function generateRandomString($length)
    {
        $characters = '0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz';
        $randomString = '';
        for ($i = 0; $i < $length; $i++) {
            $randomString .= $characters[rand(0, strlen($characters) - 1)];
        }
        return $randomString;
    }

    /**
     * @return \Sso\ApiBundle\Database\ServiceProvider\Token
     */
    private function tokenHandler()
    {
        return $this->databaseManager()->serviceProvider()->token();
    }
}
