<?php

/**
 * Class Delete
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\Webservices\ObjectBundle\Model\Request\Attribute\Delete;

use LifeStyle\Tools\WebserviceBundle\Error\Type\External as ExternalError;
use LifeStyle\Tools\WebserviceBundle\Validator\Xml as XmlValidator;
use LifeStyle\Tools\WebserviceBundle\Exception\InvalidRecordException;
use Sso\ApiBundle\Api\Manager as ApiManager;
use Sso\Webservices\ObjectBundle\Entity\Attribute as EntityAttribute;
use Sso\Webservices\ObjectBundle\Model\Request\Attribute\Delete\Attribute as ModelIn;
use Sso\Webservices\ObjectBundle\Database\Manager as DatabaseManager;

/**
 * Class Delete
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package    Sso\Webservices\ObjectBundle\Model\Request\Attribute\Show
 */
class Delete
{

    /**
     * @var ApiManager
     */
    protected $apiM;

    /**
     * @var ModelIn
     */
    protected $modelIn;

    /**
     * @var EntityAttribute
     */
    protected $attributeModel;

    /**
     * @var DatabaseManager
     */
    protected $objectDbM;

    /**
     *
     * @param ApiManager $apiM
     */
    public function __construct(ApiManager $apiM)
    {
        $this->apiM = $apiM;
        $this->objectDbM = $this->apiM->container->get('object_database_manager');
    }

    /**
     * Returns a xml-string with allowed structure
     *
     * @return string
     */
    protected function _xmlAllowedElements()
    {
        return '<AttributeType required="true">'
             . '<Id required="true"/>'
             . '<DeleteReferences/>'
             . '</AttributeType>';
    }

    /**
     * Init model
     *
     * @param \SimpleXMLElement $simpleXML
     */
    public function init(\SimpleXMLElement $simpleXML)
    {
        $validator = new XmlValidator();
        if (!$validator->validate($simpleXML, $validator->strToXml($this->_xmlAllowedElements()))) {
            $this->apiM->errors()->addErrors($validator->errors()->getErrors());
            throw new InvalidRecordException('Invalid xml elements.');
        }

        $this->modelIn = $this->apiM->serializer()->deserialize($simpleXML->AttributeType->saveXML(), 'Sso\Webservices\ObjectBundle\Model\Request\Attribute\Delete\Attribute', 'xml');
        $this->validate($this->modelIn);

        $this->deleteAttribute();
    }

    private function deleteAttribute()
    {
        $attributeId = $this->modelIn->getId();
        $this->attributeModel = $this->objectDbM->data()->attribute()->find($attributeId);

        if (null === $this->attributeModel) {
            $this->apiM->errors()->addError(new ExternalError('ad001', 'attribute id not found', 'attribute id not found', 'attribute id ' . $attributeId . ' not found'));
            throw new InvalidRecordException('attribute not found');
        }

        if (!$this->attributeModel->getAttributeValues()->isEmpty() && !$this->modelIn->isDeleteReferences()) {
            $this->apiM->errors()->addError(new ExternalError('ad002', 'Can not delete Attribute', 'Attribute has AttributeValues!', 'Please delete all referenced AttributeValues first.'));
            throw new InvalidRecordException('Delete Attribute failed');
        }

        if (!$this->attributeModel->getAttributeValues()->isEmpty() && $this->modelIn->isDeleteReferences()) {
            foreach ($this->attributeModel->getAttributeValues() as $attributeValue) {
                $this->objectDbM->data()->objectAttributeValue()->delete($attributeValue);
            }
        }

        $attributeEntity = clone $this->attributeModel;

        $this->objectDbM->data()->attribute()->delete($this->attributeModel);

        $this->attributeModel = $attributeEntity;
    }

    /**
     * @return EntityAttribute
     */
    public function getAttributeModel()
    {
        return $this->attributeModel;
    }

    /**
     * Validate entity
     *
     * @param object $entity
     * @param array $validationGroups
     * @return bool
     */
    protected function validate($entity, $validationGroups = null)
    {
        $errors = $this->apiM->validator()->validate($entity, $validationGroups);
        foreach ($errors as $error) {
            list($errorCode, $shortMessage, $longMessage) = explode('##', $error->getMessage());
            $this->apiM->errors()->addError(new ExternalError('v001', $errorCode, $shortMessage, $longMessage));
        }
        if (0 < count($errors)) {
            throw new InvalidRecordException('Invalid xml values');
        }
        return true;
    }

}
