<?php

/**
 * Class Show
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\Webservices\ObjectBundle\Model\Request\Attribute\Show;

use LifeStyle\Tools\WebserviceBundle\Error\Type\External as ExternalError;
use LifeStyle\Tools\WebserviceBundle\Validator\Xml as XmlValidator;
use LifeStyle\Tools\WebserviceBundle\Exception\InvalidRecordException;
use Sso\ApiBundle\Api\Manager as ApiManager;
use Sso\Webservices\ObjectBundle\Entity\Attribute as EntityAttribute;
use Sso\Webservices\ObjectBundle\Model\Request\Attribute\Show\Entity\Attribute as ModelIn;
use Sso\Webservices\ObjectBundle\Database\Manager as DatabaseManager;

/**
 * Class Show
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package    Sso\Webservices\ObjectBundle\Model\Request\Attribute\Show
 */
class Show
{

    /**
     * @var ApiManager
     */
    protected $apiM;
    
    /**
     * @var EntityAttribute[]
     */
    protected $attributeModels;

    /**
     * @var ModelIn
     */
    protected $modelIn;

    /**
     * @var DatabaseManager
     */
    protected $objectDbM;

    /**
     * 
     * @param ApiManager $apiM
     */
    public function __construct(ApiManager $apiM)
    {
        $this->apiM = $apiM;
        $this->objectDbM = $this->apiM->container->get('object_database_manager');
    }

    /**
     * Returns a xml-string with allowed structure
     * 
     * @return string
     */
    protected function _xmlAllowedElements()
    {
        return '<AttributeType required="true">'
                . '<Id/>'
                . '<Offset/>'
                . '<Limit/>'
                . '</AttributeType>';
    }

    /**
     * Init model
     * 
     * @param \SimpleXMLElement $simpleXML
     */
    public function init(\SimpleXMLElement $simpleXML)
    {
        $validator = new XmlValidator();
        if (!$validator->validate($simpleXML, $validator->strToXml($this->_xmlAllowedElements()))) {
            $this->apiM->errors()->addErrors($validator->errors()->getErrors());
            throw new InvalidRecordException('Invalid xml elements.');
        }

        $this->modelIn = $this->apiM->serializer()->deserialize($simpleXML->AttributeType->saveXML(), 'Sso\Webservices\ObjectBundle\Model\Request\Attribute\Show\Entity\Attribute', 'xml');
        $this->validate($this->modelIn);

        $this->getAttributeEntities();
    }

    public function getAttributeEntities()
    {
        $id = $this->modelIn->getId();
        $offset = $this->modelIn->getOffset();
        $limit = $this->modelIn->getLimit();

        if ($id) {
            $search = ['id' => $id];
            $this->attributeModels = $this->objectDbM->data()->attribute()->findBy($search);
            return;
        }

        $defaultCriteria = [];
        if ($offset !== null || $limit !== null) {
            $this->attributeModels = $this->objectDbM->data()->attribute()->findBy($defaultCriteria, array(), $limit, $offset);
        } else {
            $this->attributeModels = $this->objectDbM->data()->attribute()->findBy($defaultCriteria);
        }
    }

    /**
     * @return EntityAttribute[]
     */
    public function getAttributeModels()
    {
        return $this->attributeModels;
    }

    /**
     * @return int
     */
    public function getAttributesCount()
    {
        return $this->objectDbM->data()->attribute()->getCount();
    }

    /**
     * @return ModelIn
     */
    public function getRequestModel()
    {
        return $this->modelIn;
    }

    /**
     * Validate entity
     * 
     * @param object $entity
     * @param array $validationGroups
     * @return bool
     */
    protected function validate($entity, $validationGroups = null)
    {
        $errors = $this->apiM->validator()->validate($entity, $validationGroups);
        foreach ($errors as $error) {
            list($errorCode, $shortMessage, $longMessage) = explode('##', $error->getMessage());
            $this->apiM->errors()->addError(new ExternalError('v001', $errorCode, $shortMessage, $longMessage));
        }
        if (0 < count($errors)) {
            throw new InvalidRecordException('Invalid xml values');
        }
        return true;
    }

}
