<?php

/**
 * Class Attribute
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\Webservices\ObjectBundle\Model\Request\Attribute\Update;

use JMS\Serializer\Annotation as Serializer;
use Symfony\Component\Validator\Constraints as Assert;

/**
 * Class Attribute
 *
 * @Serializer\XmlRoot("AttributeType")
 * @Serializer\ExclusionPolicy("all")
 *
 * @package Sso\Webservices\ObjectBundle\Model\Request\Attribute\Show\Entity
 */
class Attribute
{
    /**
     * @Serializer\SerializedName("Id")
     * @Serializer\Expose
     * @Serializer\Type("string")
     *
     * @Assert\NotBlank(message = "attributeUpdate00##Invalid Attribute Id##Attribute Id should not be empty",)
     *
     * @Assert\Length(
     *     min = 36,
     *     max = 36,
     *     minMessage = "attributeUpdate00##Invalid id##Your id must be {{ limit }} characters long",
     *     maxMessage = "attributeUpdate00##Invalid id##Your id must be {{ limit }} characters long",
     *     exactMessage = "attributeUpdate00##Invalid id##Your id must be {{ limit }} characters long"
     * )
     *
     * @var string
     */
    public $id;

    /**
     * @Serializer\Expose
     * @Serializer\Type("string")
     * @Serializer\SerializedName("Name")
     *
     * @Assert\Length(
     *      min = 3,
     *      max = 200,
     *      minMessage = "attributeAdd02##Invalid Attribute Name##Attribute Name must be at least {{ limit }} characters long",
     *      maxMessage = "attributeAdd02##Invalid Attribute Name##Attribute Name cannot be longer than {{ limit }} characters",
     *      exactMessage = "attributeAdd02##Invalid Attribute Name##Attribute Name cannot be longer than {{ limit }} characters"
     * )
     *
     * @var string
     */
    private $name;

    /**
     * @Serializer\SerializedName("Mandatory")
     * @Serializer\Expose
     * @Serializer\Type("string")
     * @Serializer\Accessor(setter="setMandatoryFromString")
     *
     * @var bool
     */
    private $mandatory = false;

    /**
     * @Serializer\SerializedName("PossibleValues")
     * @Serializer\Expose
     * @Serializer\Type("array<string>")
     * @Serializer\XmlList(entry="Value")
     *
     * @var mixed
     */
    private $possibleValues;

    /**
     * @Serializer\SerializedName("DefaultValue")
     * @Serializer\Expose
     * @Serializer\Type("string")
     *
     * @var string
     */
    private $defaultValue;

    /**
     * @Serializer\SerializedName("Translatable")
     * @Serializer\Expose
     * @Serializer\Type("string")
     * @Serializer\Accessor(setter="setTranslatableFromString")
     *
     * @var bool
     */
    private $translatable = false;

    /**
     * @Serializer\SerializedName("GroupId")
     * @Serializer\Expose
     * @Serializer\Type("string")
     *
     * @Assert\Length(
     *     min = 36,
     *     max = 36,
     *     minMessage = "attributeUpdate01##Invalid AttributeGroup Id##Your AttributeGroup Id must be {{ limit }} characters long",
     *     maxMessage = "attributeUpdate01##Invalid AttributeGroup Id##Your AttributeGroup Id must be {{ limit }} characters long",
     *     exactMessage = "attributeUpdate01##Invalid AttributeGroup Id##Your AttributeGroup Id must be {{ limit }} characters long"
     * )
     *
     * @var string
     */
    private $attributeGroupId;

    /**
     * @return string
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * @param string $id
     * @return Attribute
     */
    public function setId($id)
    {
        $this->id = $id;

        return $this;
    }

    /**
     * @return string
     */
    public function getName()
    {
        return $this->name;
    }

    /**
     * @param string $name
     * @return Attribute
     */
    public function setName($name)
    {
        $this->name = $name;

        return $this;
    }

    /**
     * Get mandatory
     *
     * @return bool
     */
    public function getMandatory()
    {
        return $this->mandatory;
    }

    /**
     * Set mandatory from string
     *
     * @param string $mandatory
     * @return Attribute
     */
    public function setMandatoryFromString($mandatory)
    {
        $this->mandatory = '1' === $mandatory;

        return $this;
    }

    /**
     * @return array
     */
    public function getPossibleValues()
    {
        $returnPossibleValues = $this->possibleValues;
        if (!is_array($this->possibleValues)) {
            $returnPossibleValues = explode('/', $this->possibleValues);
        }

        return $returnPossibleValues;
    }

    /**
     * @param array|string $possibleValues
     * @return Attribute
     */
    public function setPossibleValues($possibleValues)
    {
        if (!is_array($possibleValues)) {
            $possibleValues = explode('/', $possibleValues);
        }

        $this->possibleValues = $possibleValues;

        return $this;
    }

    /**
     * @return string
     */
    public function getDefaultValue()
    {
        return $this->defaultValue;
    }

    /**
     * @param string $defaultValue
     * @return Attribute
     */
    public function setDefaultValue($defaultValue)
    {
        $this->defaultValue = $defaultValue;

        return $this;
    }

    /**
     * Get translatable
     *
     * @return bool
     */
    public function getTranslatable()
    {
        return $this->translatable;
    }

    /**
     * Set translatable from string
     *
     * @param string $translatable
     * @return Attribute
     */
    public function setTranslatableFromString($translatable)
    {
        $this->translatable = '1' === $translatable;

        return $this;
    }

    /**
     * @return string
     */
    public function getAttributeGroupId()
    {
        return $this->attributeGroupId;
    }

    /**
     * @param string $attributeGroupId
     * @return Attribute
     */
    public function setAttributeGroupId($attributeGroupId)
    {
        $this->attributeGroupId = $attributeGroupId;

        return $this;
    }
}
