<?php

/**
 * Class Entity
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\Webservices\ObjectBundle\Model\Request\Object\Add;

use JMS\Serializer\Annotation as Serializer;
use Symfony\Component\Validator\Constraints as Assert;
use Sso\Webservices\ObjectBundle\Entity\Object as ObjectEntity;
use Sso\Webservices\ObjectBundle\Entity\Type as TypeEntity;

/**
 * Class Update
 *
 * @Serializer\XmlRoot("ObjectType")
 * @Serializer\ExclusionPolicy("all")
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package    Sso\Webservices\ObjectBundle\Model\Request\Object\Add
 */
class Entity
{

    /**
     * @Serializer\Expose
     * @Serializer\Type("string")
     * @Serializer\SerializedName("ReferenceId")
     *
     * @Assert\NotBlank(message = "objectAdd00##Invalid ReferenceId##ReferenceId should not be empty")
     *
     * @Assert\Length(
     *     min = 1,
     *     max = 255,
     *     minMessage = "objectAdd00##Invalid reference id##Your reference id must be {{ limit }} characters long",
     *     maxMessage = "objectAdd00##Invalid reference id##Your reference id must be {{ limit }} characters long",
     *     exactMessage = "objectAdd00##Invalid reference id##Your reference id must be {{ limit }} characters long"
     * )
     *
     * @var string
     */
    private $referenceId;

    /**
     * Validation
     * 
     * @Assert\NotBlank(message = "objectAdd00##Invalid TypeId##TypeId should not be empty",)
     *
     * @Assert\Length(
     *      min = 36,
     *      max = 36,
     *      minMessage = "objectAdd01##Invalid TypeId##TypeId must be at least {{ limit }} characters long",
     *      maxMessage = "objectAdd01##Invalid TypeId##TypeId cannot be longer than {{ limit }} characters",
     *      exactMessage = "objectAdd01##Invalid TypeId##TypeId should have exactly {{ limit }} characters"
     * )
     * @Serializer\Expose
     * @Serializer\Type("string")
     * @Serializer\SerializedName("TypeId")
     * 
     * @var string
     */
    private $typeId;

    /**
     * Validation
     * 
     * @Assert\Length(
     *      min = 36,
     *      max = 36,
     *      minMessage = "objectAdd01##Invalid ParentGuid##ParentGuid must be at least {{ limit }} characters long",
     *      maxMessage = "objectAdd01##Invalid ParentGuid##ParentGuid cannot be longer than {{ limit }} characters",
     *      exactMessage = "objectAdd01##Invalid ParentGuid##ParentGuid should have exactly {{ limit }} characters"
     * )
     * @Serializer\Expose
     * @Serializer\Type("string")
     * @Serializer\SerializedName("ParentGuid")
     * 
     * @var string
     */
    private $parentGuid;

    /**
     * @Serializer\Expose
     * @Serializer\Type("string")
     * @Serializer\SerializedName("Name")
     *
     * @var string
     */
    private $name;

    /**
     * @Serializer\Expose
     * @Serializer\Type("string")
     * @Serializer\SerializedName("Active")
     * @Serializer\Accessor(setter="setActiveFromString")
     * 
     * @var boolean
     */
    private $active = false;

    /**
     * @Serializer\Exclude
     * 
     * @var ObjectEntity
     * */
    private $parent;

    /**
     * @Serializer\Exclude
     *
     * @var TypeEntity
     */
    private $type;

    /**
     * @return string
     */
    public function getReferenceId()
    {
        return $this->referenceId;
    }

    /**
     * @param string $referenceId
     * @return Entity
     */
    public function setReferenceId($referenceId)
    {
        $this->referenceId = $referenceId;

        return $this;
    }

    /**
     * Get name
     *
     * @return string 
     */
    public function getName()
    {
        return $this->name;
    }

    /**
     * Get lowerName
     *
     * @return string 
     */
    public function getLowerName()
    {
        return null !== $this->name ? strtolower($this->name) : null;
    }

    /**
     * Set active from string
     *
     * @param string $active
     * @return Entity
     */
    public function setActiveFromString($active)
    {
        $this->active = '1' === $active;

        return $this;
    }

    /**
     * Get active
     *
     * @return boolean 
     */
    public function getActive()
    {
        return $this->active;
    }

    /**
     * Set $typeId
     *
     * @param string $typeId
     * @return Entity
     */
    public function setTypeId($typeId)
    {
        $this->typeId = $typeId;

        return $this;
    }

    /**
     * Get $typeId
     *
     * @return integer 
     */
    public function getTypeId()
    {
        return $this->typeId;
    }

    /**
     * Set $parentGuid
     *
     * @param string $parentGuid
     * @return Entity
     */
    public function setParentGuid($parentGuid)
    {
        $this->parentGuid = $parentGuid;

        return $this;
    }

    /**
     * Get $parentGuid
     *
     * @return integer 
     */
    public function getParentGuid()
    {
        return $this->parentGuid;
    }

    /**
     * Set parent
     *
     * @param ObjectEntity $parent
     * @return Entity
     */
    public function setParent(ObjectEntity $parent = null)
    {
        $this->parent = $parent;

        return $this;
    }

    /**
     * Get parent
     *
     * @return ObjectEntity
     */
    public function getParent()
    {
        return $this->parent;
    }

    /**
     * Set type
     *
     * @param TypeEntity $type
     * @return Entity
     */
    public function setType(TypeEntity $type = null)
    {
        $this->type = $type;

        return $this;
    }

    /**
     * Get type
     *
     * @return ObjectEntity
     */
    public function getType()
    {
        return $this->type;
    }

}