<?php

/**
 * Class Show
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2015 Lifestyle Webconsulting GmbH
 * @version    $Id:$
 * @link       http://www.life-style.de
 *
 */

namespace Sso\Webservices\ObjectBundle\Model\Request\Object\TreeView;

use LifeStyle\Tools\WebserviceBundle\Error\Type\External as ExternalError;
use LifeStyle\Tools\WebserviceBundle\Validator\Xml as XmlValidator;
use LifeStyle\Tools\WebserviceBundle\Exception\InvalidRecordException;
use Sso\ApiBundle\Api\Manager as ApiManager;
use Sso\Webservices\ObjectBundle\Entity\Object as EntityObject;
use Sso\Webservices\ObjectBundle\Database\Data\TreeObject\Dto\Object as ObjectDto;

/**
 * type add request model
 */
class Show
{

    /**
     * @var ApiManager
     */
    protected $apiM;

    /**
     * @var Entity
     */
    protected $helperModel;

    /**
     *
     * @var ObjectDto[]
     */
    protected $objectModels;

    /**
     * @var \Sso\Webservices\ObjectBundle\Database\Manager
     */
    protected $objectDbM;

    /**
     *
     * @param ApiManager $apiM
     */
    public function __construct(ApiManager $apiM)
    {
        $this->apiM = $apiM;
        $this->objectDbM = $this->apiM->container->get('object_database_manager');
    }

    /**
     * Returns a xml-string with allowed structure
     *
     * @return string
     */
    protected function _xmlAllowedElements()
    {
        return '<ObjectType required="true">'
        .'<Type/>'
        .'<Guid/>'
        .'<ReferenceId/>'
        .'<Offset/>'
        .'<Limit/>'
        .'<LanguageCode/>'
        .'<FullResponse/>'
        .'</ObjectType>';
    }

    /**
     * Init model
     *
     * @param \SimpleXMLElement $simpleXML
     */
    public function init(\SimpleXMLElement $simpleXML)
    {
        $validator = new XmlValidator();
        if (!$validator->validate($simpleXML, $validator->strToXml($this->_xmlAllowedElements()))) {
            $this->apiM->errors()->addErrors($validator->errors()->getErrors());
            throw new InvalidRecordException('Invalid xml elements.');
        }

        // set helper model
        $this->helperModel = $this->apiM->serializer()->deserialize($simpleXML->ObjectType->saveXML(),
            'Sso\Webservices\ObjectBundle\Model\Request\Object\TreeView\Entity', 'xml');
        $this->validateEntity();

        $this->getObjectEntities();
    }


    private function fetchParentTree()
    {
        $search = [];
        if(count($this->objectModels) === 0){
            return;
        }
        $treeIds = explode('.', $this->objectModels[0]->getTreeId());

        $currentTreeId = '';
        $padding = '';
        foreach( $treeIds as $idBit ) {
            $currentTreeId .= $idBit;
            $search['treeId' . $padding] = $currentTreeId;
            $currentTreeId .= '.';
            $padding .= ' ';
        }

        $this->objectModels = $this->objectDbM->data()->treeObject()->findBy($search);
        return true;
    }


    public function getObjectEntities()
    {
        $treeIdGen = $this->apiM->container->get('object_event_treeids_regenerate');
        $treeIdGen->refreshTreeIdNumbers();

        $offset = $this->helperModel->getOffset();
        $limit = $this->helperModel->getLimit();

        // Get objects by given reference id
        if ($this->getObjectEntitiesByReferenceId($this->helperModel->getReferenceId())) {
            $this->fetchParentTree();
            return;
        }

        // Get objects by given guid
        if ($this->getObjectEntitiesByGuId($this->helperModel->getGuid())) {
            $this->fetchParentTree();
            return;
        }

        $defaultCriteria = [];
        if ($offset !== null || $limit !== null) {
            $this->objectModels = $this->objectDbM->data()->treeObject()->findBy($defaultCriteria, array(), $limit,
                $offset);
        } else {
            $this->objectModels = $this->objectDbM->data()->treeObject()->findBy($defaultCriteria);
        }
    }

    /**
     * Get objects by given Guid
     *
     * @param string $guId
     * @return bool|null
     */
    private function getObjectEntitiesByGuId($guId)
    {
        if (!$guId) {
            return null;
        }

        $search = ['guid' => $guId];
        $this->objectModels = $this->objectDbM->data()->treeObject()->findBy($search);
        return true;
    }

    /**
     * Get objects by given reference id
     *
     * @param string $referenceId
     * @return bool|null
     */
    private function getObjectEntitiesByReferenceId($referenceId)
    {
        if (empty($referenceId)) {
            return null;
        }

        $search = ['referenceId' => $referenceId];
        $this->objectModels = $this->objectDbM->data()->treeObject()->findBy($search);
        return true;
    }

    /**
     * @return int
     */
    public function getObjectsCount()
    {
        return $this->objectDbM->data()->treeObject()->getCount();
    }

    /**
     *
     * @return EntityObject[]
     */
    public function getObjectModels()
    {
        return $this->objectModels;
    }

    /**
     *
     * @return Entity
     */
    public function getHelperModel()
    {
        return $this->helperModel;
    }

    /**
     * Validate entity
     *
     * @param array $validationGroups
     * @return bool
     */
    protected function validateEntity($validationGroups = null)
    {
        $errors = $this->apiM->validator()->validate($this->helperModel, $validationGroups);
        foreach ($errors as $error) {
            list($errorCode, $shortMessage, $longMessage) = explode('##', $error->getMessage());
            $this->apiM->errors()->addError(new ExternalError('v001', $errorCode, $shortMessage, $longMessage));
        }
        if (0 < count($errors)) {
            throw new InvalidRecordException('Invalid xml values');
        }
        return true;
    }
}
