<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2018 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\RestBundle\Controller\IdpWs;

use FOS\RestBundle\View\View;
use Nelmio\ApiDocBundle\Annotation\ApiDoc;
use FOS\RestBundle\Controller\Annotations\NamePrefix;
use FOS\RestBundle\Controller\FOSRestController;
use Sso\RestBundle\Worker\Worker as Worker;
use JMS\DiExtraBundle\Annotation as DI;
use FOS\RestBundle\Controller\Annotations\Post;
use FOS\RestBundle\Controller\Annotations\Get;
use Symfony\Component\HttpFoundation\Request;
use LifeStyle\Tools\CachingBundle\Annotations\LsCache;

/**
 * Class ServiceUserController
 *
 * @NamePrefix("api_rest_")
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package    Sso\RestBundle\Controller
 */
class ServiceUserController extends FOSRestController
{
    /**
     * @DI\Inject("rest_worker_manager")
     * @var Worker
     */
    protected $woM;

    /**
     *
     * *This service is used to get the sso useridentifier, it simulates a saml login and
     * returns a useridentifier, which can used to trigger further requests.
     * Header authentication is not required for this service!*
     *
     *
     * ####Requests
     *
     * Request example JSON
     *
     *      /api/idp/userindentifier/get.json
     *
     * Request example XML
     *
     *      /api/idp/userindentifier/get.xml
     *
     * ####Response success
     *
     * Example (header)
     *
     *      200 OK
     *
     * Example JSON (body)
     *
     *      {
     *       "code": 200,
     *       "status": "OK",
     *       "scriptTimeSec": 2.8424,
     *       "trackId": "59af716d20",
     *       "userIdentifier": "b58ea0fd424a7faf0e...",
     *       "sessionLifeTime": "2015-07-29T16:11:39+00:00"
     *      )
     *
     * Example XML (body)
     *
     *      <?xml version="1.0" encoding="UTF-8"?>
     *      <result>
     *          <code>200</code>
     *          <status><![CDATA[OK]]></status>
     *          <scriptTimeSec>3.2065</scriptTimeSec>
     *          <trackId><![CDATA[3732ff09ef]]></trackId>
     *          <userIdentifier><![CDATA[b58ea8fd424a7faf0e...]]></userIdentifier>
     *          <sessionLifeTime><![CDATA[2015-07-29T16:13:55+00:00]]></sessionLifeTime>
     *      </result>
     *
     * ####Response error
     *
     * Example (header)
     *
     *      400 Bad Request
     *
     * Example JSON (body)
     *
     *      {
     *        "status": 400,
     *        "message": "Bad Request",
     *        "trackid": "3c2f39bb43",
     *        "count": 1,
     *        "errors": [
     *          {
     *            "code": 400,
     *            "message": "ErrorCode: ui001 ErrorRef: ui011 ShortMessage: Invalid credentials",
     *            "type": "external",
     *            "exception": "Wrong username or password",
     *            "debug": "not set"
     *          }
     *        ]
     *      }
     *
     *
     * Example XML (body)
     *
     *      <?xml version="1.0" encoding="UTF-8"?>
     *       <response>
     *         <status>400</status>
     *         <message><![CDATA[Bad Request]]></message>
     *         <trackid><![CDATA[bc63563270]]></trackid>
     *         <count>1</count>
     *         <errors>
     *           <error>
     *             <code>400</code>
     *             <message><![CDATA[ErrorCode: ui001 ErrorRef: ui011 ShortMessage: Invalid credentials]]></message>
     *             <type><![CDATA[external]]></type>
     *             <exception><![CDATA[Wrong username or password]]></exception>
     *             <debug><![CDATA[not set]]></debug>
     *           </error>
     *         </errors>
     *       </response>
     *
     * @ApiDoc(
     *  resource="/api/idp/useridentifier",
     *  description="get sso useridentifier",
     *  statusCodes={
     *         200="OK",
     *         400="Bad Request",
     *         401="Unauthorized",
     *         500="Internal Server Error"
     *  },
     *  views = { "idp" },
     *  input="Sso\RestBundle\ModelRest\Request\IdpWs\ServiceUser\Get\UserIdentifier",
     *  output={
     *          "class" = "Sso\RestBundle\ModelRest\Response\IdpWs\ServiceUser\Get\UserIdentifier"
     *      },
     *  tags={
     *      "stable" = "#000"
     *      }
     * )
     * @Post("/idp/userindentifier/get", name="_service", options={ "method_prefix" = true })
     * @LsCache(cacheTTL=60, keyHeader=true)
     * @param Request $request
     * @return View
     */
    public function postUserIdentifierAction(Request $request)
    {
        return $this->woM->idpWs()->serviceUser()->getUserIdentifier()->init($request);
    }

    /**
     *
     * *This service is used to validate a sso useridentifier.
     * Header authentication is not required for this service!*
     *
     *
     * ####Requests
     *
     * Request example JSON
     *
     *      /api/service/validate/user/identifier.json?username=YOURNAME&useridentifier=b58ea8fd424a7faf0e...
     *
     * Request example XML
     *
     *      /api/service/validate/user/identifier.xml?username=YOURNAME&useridentifier=b58ea8fd424a7faf0e...
     *
     * ####Response success
     *
     * Example (header)
     *
     *      200 OK
     *
     * Example JSON (body)
     *
     *      {
     *       "status": 200,
     *        "message": "OK",
     *        "trackId": "6c8ebd522c",
     *        "userIdentifier": "b57ea0fd424a7faf0e...",
     *        "sessionLifeTime": "2015-07-29T16:37:42+00:00"
     *      }
     *
     * Example XML (body)
     *
     *      <?xml version="1.0" encoding="UTF-8"?>
     *       <result>
     *         <status>200</status>
     *         <message><![CDATA[OK]]></message>
     *         <trackId><![CDATA[0c8e9c2d96]]></trackId>
     *         <userIdentifier><![CDATA[b58ea0fd424a7faf0e...]]></userIdentifier>
     *         <sessionLifeTime><![CDATA[2015-07-29T16:35:47+00:00]]></sessionLifeTime>
     *       </result>
     *
     * ####Response error
     *
     * Example (header)
     *
     *      400 Bad Request
     *
     * Example JSON (body)
     *
     *     {
     *          "status": 400,
     *          "message": "Bad Request",
     *          "trackid": "1128a98abb",
     *          "count": 1,
     *          "errors": [
     *            {
     *              "code": 400,
     *              "message": "ErrorCode: ui002 ErrorRef: ui021 ShortMessage: Invalid credentials",
     *              "type": "external",
     *              "exception": "Wrong username or user-identifier",
     *              "debug": "not set"
     *            }
     *          ]
     *        }
     *
     *
     * Example XML (body)
     *
     *      <?xml version="1.0" encoding="UTF-8"?>
     *         <response>
     *           <status>400</status>
     *           <message><![CDATA[Bad Request]]></message>
     *           <trackid><![CDATA[0218eb7452]]></trackid>
     *           <count>1</count>
     *           <errors>
     *             <error>
     *               <code>400</code>
     *               <message><![CDATA[ErrorCode: ui002 ErrorRef: ui021 ShortMessage: Invalid credentials]]></message>
     *               <type><![CDATA[external]]></type>
     *               <exception><![CDATA[Wrong username or user-identifier]]></exception>
     *               <debug><![CDATA[not set]]></debug>
     *             </error>
     *           </errors>
     *         </response>
     *
     * @ApiDoc(
     *  resource="/api/idp/useridentifier",
     *  description="validate sso useridentifier",
     *  statusCodes={
     *         200="OK",
     *         400="Bad Request",
     *         401="Unauthorized",
     *         500="Internal Server Error"
     *  },
     *  views = { "idp" },
     *  input ="Sso\RestBundle\ModelRest\Request\IdpWs\ServiceUser\Validate\UserIdentifier",
     *  output={
     *          "class" = "Sso\RestBundle\ModelRest\Response\IdpWs\ServiceUser\Validate\UserIdentifier"
     *      },
     *  tags={
     *      "stable" = "#000"
     *  }
     * )
     *
     * @Get("idp/userindentifier/validate", name="_service", options={ "method_prefix" = true })
     * @param Request $request
     *
     * @return View
     */
    public function getValidateUserIdentifierAction(Request $request)
    {
        return $this->woM->idpWs()->serviceUser()->getValidateUserIdentifier()->init($request);
    }
}
