<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2018 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\RestBundle\Controller\ObjectWs\Version1;

use FOS\RestBundle\View\View;
use JMS\SecurityExtraBundle\Annotation\Secure;
use Nelmio\ApiDocBundle\Annotation\ApiDoc;
use FOS\RestBundle\Controller\Annotations\NamePrefix;
use Symfony\Component\HttpFoundation\Request;
use FOS\RestBundle\Controller\FOSRestController;
use Sso\RestBundle\Worker\Worker as Worker;
use JMS\DiExtraBundle\Annotation as DI;
use FOS\RestBundle\Controller\Annotations\Get;
use FOS\RestBundle\Controller\Annotations\Delete;
use FOS\RestBundle\Controller\Annotations\Put;
use LifeStyle\Tools\CachingBundle\Annotations\LsCache;

/**
 * Class AttributeController
 *
 * @NamePrefix("api_rest_")
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package    Sso\RestBundle\Controller
 */
class AttributeController extends FOSRestController
{
    /**
     * @DI\Inject("rest_worker_manager")
     * @var Worker
     */
    protected $woM;

    /**
     *
     * This service is used to get all or a single attribute by its id.
     * Returns a list of attributes.
     *
     * #Request (header)
     *
     *      Key                 | Value               | Description
     *      ------------------- | ------------------- | --------------------------------------
     *      username            |  (string)           | your username
     *      useridentifier      |  (string)           | your useridentifier for given username
     *      servicetoken        |  (string)           | servicetoken from ws
     *
     * ##Request
     *
     * **Request example JSON**
     *
     *          /api/v1/attribute.json?id=YOURID&offset=YOUROFFSET&limit=YOURLIMIT
     *
     * **Request example XML**
     *
     *          /api/v1/attribute.xml?id=YOURID&offset=YOUROFFSET&limit=YOURLIMIT
     *
     * ##Response success
     *
     * **Example (header)**
     *
     *          200 OK
     *
     * **Example JSON (body)**
     *
     *       {
     *           "status": "Success",
     *           "code": 200,
     *           "ScriptTimeSec": 2.763,
     *           "trackid": "06e89b75b7",
     *           "date": "2017-05-22T07:41:56+02:00",
     *           "count": "1",
     *           "attributes": [
     *               {
     *                   "id": "2065c110-fe3b-4f63-b2b3-c1a74bfca896",
     *                   "name": "test",
     *                   "mandatory": 1,
     *                   "possibleValues": [
     *                       "yes",
     *                       "no"
     *                   ],
     *                   "defaultValue": "yes",
     *                   "translatable": 1
     *               }
     *           ]
     *       }
     *
     * **Example XML (body)**
     *
     *       <?xml version="1.0" encoding="UTF-8"?>
     *       <SsoResponse>
     *           <status><![CDATA[Success]]></status>
     *           <code>200</code>
     *           <ScriptTimeSec>4.5276</ScriptTimeSec>
     *           <trackid><![CDATA[5e6a5600db]]></trackid>
     *           <date><![CDATA[2017-05-22T07:44:48+02:00]]></date>
     *           <count><![CDATA[1]]></count>
     *           <attributes>
     *               <id><![CDATA[2065c110-fe3b-4f63-b2b3-c1a74bfca896]]></id>
     *               <name><![CDATA[test]]></name>
     *               <mandatory>1</mandatory>
     *               <possibleValues>
     *                   <value><![CDATA[yes]]></value>
     *                   <value><![CDATA[no]]></value>
     *               </possibleValues>
     *               <defaultValue><![CDATA[yes]]></defaultValue>
     *               <translatable>1</translatable>
     *           </attributes>
     *       </SsoResponse>
     *
     * ##Response error
     *
     * **Example (header)**
     *
     *          400 Bad Request
     *
     * **Example JSON (body)**
     *
     *       {
     *           "code": 400,
     *           "trackid": "ad86e1935c",
     *           "count": "1",
     *           "errors": [
     *               {
     *                   "code": "400",
     *                   "message": "ErrorCode: No attributes found. ErrorRef: as001 ShortMessage: No attributes found.",
     *                   "exception": "external"
     *               }
     *           ]
     *       }
     *
     * **Example XML (body)**
     *
     *       <SsoResponse>
     *           <code>400</code>
     *           <trackid><![CDATA[b59f6a84ae]]></trackid>
     *           <count><![CDATA[1]]></count>
     *           <errors>
     *               <error>
     *                   <code><![CDATA[400]]></code>
     *                   <message><![CDATA[ErrorCode: No attributes found. ErrorRef: as001 ShortMessage: No attributes found.]]></message>
     *                   <exception><![CDATA[external]]></exception>
     *               </error>
     *           </errors>
     *       </SsoResponse>
     *
     * ##Response error status 401, 416 or 500
     *
     * **Example JSON (body)**
     *
     *       {
     *       "code": 416,
     *       "status": "Not Found",
     *       "trackId": "2d008cc4c5",
     *       "scriptTimeSec": 1.521,
     *       "count": 1,
     *       "errors": [
     *          {
     *          "code": 416,
     *          "status": "Not Found",
     *          "message": "no valid json string",
     *          "type": "external",
     *          "exception": "InvalidRequest",
     *          "debug": "not set"
     *          }
     *         ]
     *       }
     *
     * **Example XML (body)**
     *
     *      <?xml version="1.0" encoding="UTF-8"?>
     *      <response>
     *          <code>416</code>
     *          <status><![CDATA[Not Found]]></status>
     *          <trackId><![CDATA[16fe523df4]]></trackId>
     *          <scriptTimeSec>2.4459</scriptTimeSec>
     *          <count>1</count>
     *          <errors>
     *              <error>
     *                <code>416</code>
     *                <status><![CDATA[Not Found]]></status>
     *                <message><![CDATA[expected '>']]></message>
     *                <type><![CDATA[external]]></type>
     *                <exception><![CDATA[InvalidRequest]]></exception>
     *                <debug><![CDATA[not set]]></debug>
     *              </error>
     *          </errors>
     *      </response>
     *
     * @ApiDoc(
     *  resource="/api/{version}/sso-ws-object/attribute",
     *  description="secured: Returns a list of attributes from sso-object-ws",
     *  statusCodes={
     *         200="Returned when successful",
     *         204="Returned when no content was send",
     *         400="Returned when an error occurred",
     *         401="Unauthorized - Wrong credentials",
     *         403="Returned when the user is not authorized to do this request",
     *         416="Returned when the request is malformed",
     *         500="Returned on internal server errors"
     *  },
     *  headers={
     *      {
     *        "name"="username",
     *        "description"="sso username",
     *        "required"=true
     *       },
     *      {
     *        "name"="useridentifier",
     *        "description"="sso useridentifier",
     *        "required"=true
     *       },
     *      {
     *        "name"="servicetoken",
     *        "description"="servicetoken",
     *        "required"=true
     *       }
     *  },
     *  tags={
     *      "stable" = "#000"
     *  },
     *  output="Sso\RestBundle\ModelRest\Response\ObjectWs\Version1\Attribute\Show\Response",
     *  input="Sso\RestBundle\ModelRest\Request\ObjectWs\Version1\Attribute\Show\Request",
     *  views = { "sso-ws-object" },
     * )
     *
     * @Get("/v1/sso-ws-object/attribute/show", name="_sso-ws-object_v1", options={ "method_prefix" = true })
     *
     * @Secure(roles="ROLE_API")
     * @LsCache(cacheTTL=300, keyHeader=true)
     * @param Request $request
     * @return View
     */
    public function getAttributeAction(Request $request)
    {
        return $this->woM->objectWs()->version1()->attribute()->show()->init($request);
    }

    /**
     *
     * *This service deletes single or multiple attributes from the object-webservice by its ids.
     * The service expects a JSON or XML in body, and a header authentication.*
     *
     * #Request (header)
     *
     *      Key                 | Value               | Description
     *      ------------------- | ------------------- | --------------------------------------
     *      username            |  (string)           | your username
     *      useridentifier      |  (string)           | your useridentifier for given username
     *      servicetoken        |  (string)           | servicetoken from ws
     *
     * #Request (body)
     *
     *
     * **Request Example JSON**
     *
     *       {
     *         "attribute": [
     *           {
     *             "id": "CB7D874E-A114-BF02-81D4-EC7935C748CA"
     *           },
     *           {
     *             "id": "229B72CA-1927-68B2-5AEB-E87C74AF717E",
     *             "deleteReferences": true
     *           }
     *         ]
     *       }
     *
     * **Request Example XML**
     *
     *
     *       <?xml version="1.0"?>
     *         <attribute>
     *           <item>
     *             <id>CB7D874E-A114-BF02-81D4-EC7935C748CA</id>
     *           </item>
     *           <item>
     *             <id>1229B72CA-1927-68B2-5AEB-E87C74AF717E</id>
     *             <deleteReferences>false</deleteReferences>
     *           </item>
     *         </attribute>
     *
     *
     * #Response success
     *
     * **Example (header)**
     *
     *     200 OK
     *
     *
     * **Example JSON (body)**
     *
     *      {
     *        "attribute": [
     *          {
     *            "status": "200",
     *            "trackId": "b45d14ed3a",
     *            "id": "CB7D874E-A114-BF02-81D4-EC7935C748CA"
     *          },
     *          {
     *            "status": "200",
     *            "trackId": "6c4c00c486",
     *            "id": "B72CA-1927-68B2-5AEB-E87C74AF717E"
     *          }
     *        ]
     *      }
     *
     * **Example XML (body)**
     *
     *      <?xml version="1.0" encoding="UTF-8"?>
     *        <attribute>
     *          <attribute>
     *            <item>
     *              <status><![CDATA[200]]></status>
     *              <trackId><![CDATA[845908f592]]></trackId>
     *              <id><![CDATA[CB7D874E-A114-BF02-81D4-EC7935C748CA]]></id>
     *            </item>
     *            <item>
     *              <status><![CDATA[200]]></status>
     *              <trackId><![CDATA[e829e38460]]></trackId>
     *              <id><![CDATA[B72CA-1927-68B2-5AEB-E87C74AF717E]]></id>
     *            </item>
     *          </attribute>
     *        </attribute>
     *
     *
     * #Response error
     *
     * **Example JSON (body)**
     *
     *      {
     *        "attribute": [
     *          {
     *            "status": "200",
     *            "trackId": "75eb4274bb",
     *            "id": "CB7D874E-A114-BF02-81D4-EC7935C748CA"
     *          },
     *          {
     *            "status": "400",
     *            "id": "B72CA-1927-68B2-5AEB-E87C74AF717E",
     *            "error": {
     *              "status": 400,
     *              "message": "Bad Request",
     *              "trackid": "24e506b04c",
     *              "count": 1,
     *              "errors": [
     *                {
     *                  "code": 400,
     *                  "message": "ErrorCode: Can not delete Attribute ErrorRef: ad002 ShortMessage: Attribute has AttributeValues!",
     *                  "exception": "external"
     *                }
     *              ]
     *            }
     *          },
     *          {
     *            "status": "400",
     *            "id": "UNKNOWNN-1927-68B2-5AEB-E87C74AF717E",
     *            "error": {
     *              "status": 400,
     *              "message": "Bad Request",
     *              "trackid": "8ff721c4fa",
     *              "count": 1,
     *              "errors": [
     *                {
     *                  "code": 400,
     *                  "message": "ErrorCode: attribute id not found ErrorRef: ad001 ShortMessage: attribute id not found",
     *                  "exception": "external"
     *                }
     *              ]
     *            }
     *          }
     *        ]
     *      }
     *
     * **Example XML (body)**
     *
     *      <?xml version="1.0" encoding="UTF-8"?>
     *        <attribute>
     *          <attribute>
     *            <item>
     *              <status><![CDATA[200]]></status>
     *              <trackId><![CDATA[b2b0bdbf3a]]></trackId>
     *              <id><![CDATA[CB7D874E-A114-BF02-81D4-EC7935C748CA]]></id>
     *            </item>
     *            <item>
     *              <status><![CDATA[400]]></status>
     *              <id><![CDATA[B72CA-1927-68B2-5AEB-E87C74AF717E]]></id>
     *              <error>
     *                <status>400</status>
     *                <message><![CDATA[Bad Request]]></message>
     *                <trackid><![CDATA[d685982b1a]]></trackid>
     *                <count>1</count>
     *                <errors>
     *                  <error>
     *                    <code>400</code>
     *                    <message><![CDATA[ErrorCode: Can not delete Attribute ErrorRef: ad002 ShortMessage: Attribute has AttributeValues!]]></message>
     *                    <exception><![CDATA[external]]></exception>
     *                  </error>
     *                </errors>
     *              </error>
     *            </item>
     *            <item>
     *              <status><![CDATA[400]]></status>
     *              <id><![CDATA[UNKNOWNN-1927-68B2-5AEB-E87C74AF717E]]></id>
     *              <error>
     *                <status>400</status>
     *                <message><![CDATA[Bad Request]]></message>
     *                <trackid><![CDATA[ba3ab814b7]]></trackid>
     *                <count>1</count>
     *                <errors>
     *                  <error>
     *                    <code>400</code>
     *                    <message><![CDATA[ErrorCode: attribute id not found ErrorRef: ad001 ShortMessage: attribute id not found]]></message>
     *                    <exception><![CDATA[external]]></exception>
     *                  </error>
     *                </errors>
     *              </error>
     *            </item>
     *          </attribute>
     *        </attribute>
     *
     *
     * @ApiDoc(
     *  resource="/api/{version}/sso-ws-object/attribute",
     *  description="secured: Deletes attributes from sso-object-ws",
     *  statusCodes={
     *         200="Returned when successful",
     *         204="Returned when no content was send",
     *         400="Returned when an error occured",
     *         401="Unauthorized - Wrong credentials",
     *         403="Returned when the user is not authorized to do this request",
     *         500="Returned on internal server errors"
     *  },
     *  headers={
     *      {"name"="username", "description"="sso username"},
     *      {"name"="useridentifier", "description"="sso useridentifier"},
     *      {"name"="servicetoken", "description"="servicetoken"}
     *  },
     *  views = { "sso-ws-object" },
     *  tags={
     *      "stable" = "#000"
     *  }
     * )
     *
     * @Delete("/v1/sso-ws-object/attribute/delete", name="_sso-ws-object_v1", options={ "method_prefix" = true })
     * @Secure(roles="ROLE_API")
     *
     * @param Request $request
     * @return View
     */
    public function deleteAttributeAction(Request $request)
    {
        return $this->woM->objectWs()->version1()->attribute()->delete()->init($request);
    }

    /**
     *
     * *This service adds multiple attributes to the sso-object-ws.
     * The service expects a JSON in body, and a header authentication.*
     *
     * #Request (header)
     *
     *      Key                 | Value               | Description
     *      ------------------- | ------------------- | --------------------------------------
     *      username            |  (string)           | your username
     *      useridentifier      |  (string)           | your useridentifier for given username
     *      servicetoken        |  (string)           | servicetoken from ws
     *
     * #Request (body)
     *
     *
     * **Request Example JSON**
     *
     *       {
     *           "attributes": [
     *               {
     *                   "name": "Test",
     *                   "mandatory": 0,
     *                   "possibleValues": [
     *                       "some option",
     *                       "some other option"
     *                   ],
     *                   "defaultValue": "some option",
     *                   "translatable": 1,
     *                   "groupId": "0f38e869-6fdd-47d2-818a-bad3637b62b3"
     *               }
     *           ]
     *       }
     *
     * **Request Example XML**
     *
     *       <attributeAdd>
     *           <attributes>
     *               <attribute>
     *                   <name>Test</name>
     *                   <mandatory>0</mandatory>
     *                   <possibleValues>
     *                       <value>some option</value>
     *                       <value>some other option</value>
     *                   </possibleValues>
     *                   <defaultValue>some option</defaultValue>
     *                   <translatable>1</translatable>
     *                   <groupId>0f38e869-6fdd-47d2-818a-bad3637b62b3</groupId>
     *               </attribute>
     *           </attributes>
     *       </attributeAdd>
     *
     * #Response success
     *
     * **Example (header)**
     *
     *     200 OK
     *
     * **Example JSON (body)**
     *
     *       {
     *           "code": 200,
     *           "status": "OK",
     *           "trackId": "4c2e480793",
     *           "date": "2017-05-22T09:15:45+02:00",
     *           "scriptTimeSec": 2.3529,
     *           "attributes": [
     *               {
     *                   "code": 200,
     *                   "status": "OK",
     *                   "id": "cf40a8ad-bcc7-4f0a-a375-b2346173eb71",
     *                   "name": "Test",
     *                   "mandatory": 0,
     *                   "possibleValues": [
     *                       "some option",
     *                       "some other option"
     *                   ],
     *                   "defaultValue": "some option",
     *                   "translatable": 1,
     *                   "groupId": "0f38e869-6fdd-47d2-818a-bad3637b62b3"
     *               }
     *           ]
     *       }
     *
     * **Example XML (body)**
     *
     *       <?xml version="1.0" encoding="UTF-8"?>
     *       <attributeAdd>
     *           <code>200</code>
     *           <status><![CDATA[OK]]></status>
     *           <trackId><![CDATA[b7aad51a61]]></trackId>
     *           <date><![CDATA[2017-05-22T09:24:30+02:00]]></date>
     *           <scriptTimeSec>2.1379</scriptTimeSec>
     *           <attributes>
     *               <attribute>
     *                   <code>200</code>
     *                   <status><![CDATA[OK]]></status>
     *                   <id><![CDATA[28cba804-b06b-4ac1-9dd3-30ad117ecc57]]></id>
     *                   <name><![CDATA[Test]]></name>
     *                   <mandatory>0</mandatory>
     *                   <possibleValues>
     *                       <value><![CDATA[some option]]></value>
     *                       <value><![CDATA[some other option]]></value>
     *                   </possibleValues>
     *                   <defaultValue><![CDATA[some option]]></defaultValue>
     *                   <translatable>1</translatable>
     *                   <groupId><![CDATA[0f38e869-6fdd-47d2-818a-bad3637b62b3]]></groupId>
     *               </attribute>
     *           </attributes>
     *       </attributeAdd>
     *
     * #Response error
     *
     * **Example JSON (body)**
     *
     *       {
     *           "code": 400,
     *           "status": "Bad Request",
     *           "trackId": "7e9594faf8",
     *           "date": "2017-05-22T09:26:12+02:00",
     *           "scriptTimeSec": 3.4352,
     *           "attributes": [
     *               {
     *                   "code": 400,
     *                   "status": "Bad Request",
     *                   "errors": [
     *                       {
     *                           "code": 400,
     *                           "message": "ErrorCode: attributeGroup id not found ErrorRef: v001 ShortMessage: attributeGroup id not found",
     *                           "exception": "external"
     *                       }
     *                   ]
     *               }
     *           ]
     *       }
     *
     * **Example XML (body)**
     *
     *       <?xml version="1.0" encoding="UTF-8"?>
     *       <attributeAdd>
     *           <code>400</code>
     *           <status><![CDATA[Bad Request]]></status>
     *           <trackId><![CDATA[1bb8a175b6]]></trackId>
     *           <date><![CDATA[2017-05-22T09:20:09+02:00]]></date>
     *           <scriptTimeSec>7.4622</scriptTimeSec>
     *           <attributes>
     *               <attribute>
     *                   <code>400</code>
     *                   <status><![CDATA[Bad Request]]></status>
     *                   <errors>
     *                       <error>
     *                           <code>400</code>
     *                           <message><![CDATA[ErrorCode: attributeGroup id not found ErrorRef: v001 ShortMessage: attributeGroup id not found]]></message>
     *                           <exception><![CDATA[external]]></exception>
     *                       </error>
     *                   </errors>
     *               </attribute>
     *           </attributes>
     *       </attributeAdd>
     *
     * ##Response error status 401, 416 or 500
     *
     * **Example JSON (body)**
     *
     *       {
     *       "code": 416,
     *       "status": "Not Found",
     *       "trackId": "2d008cc4c5",
     *       "scriptTimeSec": 1.521,
     *       "count": 1,
     *       "errors": [
     *          {
     *          "code": 416,
     *          "status": "Not Found",
     *          "message": "no valid json string",
     *          "type": "external",
     *          "exception": "InvalidRequest",
     *          "debug": "not set"
     *          }
     *         ]
     *       }
     *
     * **Example XML (body)**
     *
     *      <?xml version="1.0" encoding="UTF-8"?>
     *      <response>
     *          <code>416</code>
     *          <status><![CDATA[Not Found]]></status>
     *          <trackId><![CDATA[16fe523df4]]></trackId>
     *          <scriptTimeSec>2.4459</scriptTimeSec>
     *          <count>1</count>
     *          <errors>
     *              <error>
     *                <code>416</code>
     *                <status><![CDATA[Not Found]]></status>
     *                <message><![CDATA[expected '>']]></message>
     *                <type><![CDATA[external]]></type>
     *                <exception><![CDATA[InvalidRequest]]></exception>
     *                <debug><![CDATA[not set]]></debug>
     *              </error>
     *          </errors>
     *      </response>
     *
     * @ApiDoc(
     *  resource="/api/{version}/sso-ws-object/attribute",
     *  description="secured: Adds multiple attributes to sso-object-ws",
     *  statusCodes={
     *         200="Returned when successful",
     *         204="Returned when no content was send",
     *         400="Returned when an error occurred",
     *         401="Unauthorized - Wrong credentials",
     *         403="Returned when the user is not authorized to do this request",
     *         416="Returned when the request is malformed",
     *         500="Returned on internal server errors"
     *  },
     *  headers={
     *      {
     *        "name"="username",
     *        "description"="sso username",
     *        "required"=true
     *       },
     *      {
     *        "name"="useridentifier",
     *        "description"="sso useridentifier",
     *        "required"=true
     *       },
     *      {
     *        "name"="servicetoken",
     *        "description"="servicetoken",
     *        "required"=true
     *       }
     *  },
     *  tags={
     *      "stable" = "#000"
     *  },
     *  output="Sso\RestBundle\ModelRest\Response\ObjectWs\Version1\Attribute\Add\Response",
     *  views = { "sso-ws-object" }
     * )
     *
     * @Put("/v1/sso-ws-object/attribute/add", name="_sso-ws-object_v1", options={ "method_prefix" = true })
     * @Secure(roles="ROLE_API")
     * @param Request $request
     * @return View
     */
    public function putAttributeAction(Request $request)
    {
        return $this->woM->objectWs()->version1()->attribute()->add()->init($request);
    }

    /**
     *
     * *This service updates attributes in the sso-object-ws. The service expects a JSON or XML in body and a header authentication.*
     *
     * #Request (header)
     *
     *      Key                 | Value               | Description
     *      ------------------- | ------------------- | --------------------------------------
     *      username            |  (string)           | your username
     *      useridentifier      |  (string)           | your useridentifier for given username
     *      servicetoken        |  (string)           | servicetoken from ws
     *
     * ##Request
     *
     * **Request Example JSON**
     *
     *       {
     *           "attributes": [
     *               {
     *                   "id": "cf40a8ad-bcc7-4f0a-a375-b2346173eb71",
     *                   "name": "Test",
     *                   "mandatory": 0,
     *                   "possibleValues": [
     *                       "some option",
     *                       "some other option"
     *                   ],
     *                   "defaultValue": "some option",
     *                   "translatable": 1,
     *                   "groupId": "0f38e869-6fdd-47d2-818a-bad3637b62b3"
     *               }
     *           ]
     *       }
     *
     * **Request Example XML**
     *
     *       <?xml version="1.0" encoding="UTF-8"?>
     *       <attributeUpdate>
     *           <attributes>
     *               <attribute>
     *                   <id>28cba804-b06b-4ac1-9dd3-30ad117ecc57</id>
     *                   <name>Test</name>
     *                   <mandatory>0</mandatory>
     *                   <possibleValues>
     *                       <value>some option</value>
     *                       <value>some other option</value>
     *                   </possibleValues>
     *                   <defaultValue>some option</defaultValue>
     *                   <translatable>1</translatable>
     *                   <groupId>0f38e869-6fdd-47d2-818a-bad3637b62b3</groupId>
     *               </attribute>
     *           </attributes>
     *       </attributeUpdate>
     *
     * #Response success
     *
     * **Example JSON (body)**
     *
     *       {
     *           "code": 200,
     *           "status": "OK",
     *           "trackId": "fd73380312",
     *           "date": "2016-12-05T10:17:52+01:00",
     *           "scriptTimeSec": 5.6455,
     *           "attributes": [
     *               {
     *                   "code": 200,
     *                   "status": "OK",
     *                   "id": "28cba804-b06b-4ac1-9dd3-30ad117ecc57",
     *                   "name": "Test",
     *                   "mandatory": 0,
     *                   "possibleValues": [
     *                       "some option",
     *                       "some other option"
     *                   ],
     *                   "defaultValue": "some option",
     *                   "translatable": 1,
     *                   "groupId": "0f38e869-6fdd-47d2-818a-bad3637b62b3"
     *               }
     *           ]
     *       }
     *
     * **Example XML (body)**
     *
     *       <?xml version="1.0" encoding="UTF-8"?>
     *       <attributeUpdate>
     *           <code>200</code>
     *           <status><![CDATA[OK]]></status>
     *           <trackId><![CDATA[e11853991e]]></trackId>
     *           <date><![CDATA[2016-12-05T10:04:39+01:00]]></date>
     *           <scriptTimeSec>3.1904</scriptTimeSec>
     *               <attribute>
     *                   <code>200</code>
     *                   <status><![CDATA[OK]]></status>
     *                   <id><![CDATA[28cba804-b06b-4ac1-9dd3-30ad117ecc57]]></id>
     *                   <name><![CDATA[Test]]></name>
     *                   <mandatory>0</mandatory>
     *                   <possibleValues>
     *                       <value><![CDATA[some option]]></value>
     *                       <value><![CDATA[some other option]]></value>
     *                   </possibleValues>
     *                   <defaultValue><![CDATA[some option]]></defaultValue>
     *                   <translatable>1</translatable>
     *                   <groupId><![CDATA[0f38e869-6fdd-47d2-818a-bad3637b62b3]]></groupId>
     *               </attribute>
     *       </attributeUpdate>
     *
     * #Response error
     *
     * **Example JSON (body)**
     *
     *       {
     *           "code": 400,
     *           "status": "Bad Request",
     *           "trackId": "fd73380312",
     *           "date": "2016-12-05T10:17:52+01:00",
     *           "scriptTimeSec": 5.6455,
     *           "attributes": [
     *               {
     *                   "code": 400,
     *                   "status": "Bad Request",
     *                   "errors": [
     *                       {
     *                           "code": 400,
     *                           "message": "ErrorCode: attributeUpdate00 ErrorRef: v001 ShortMessage: Invalid id",
     *                           "exception": "external"
     *                       }
     *                   ]
     *               }
     *           ]
     *       }
     *
     * **Example XML (body)**
     *
     *       <?xml version="1.0" encoding="UTF-8"?>
     *       <attributeUpdate>
     *           <code>400</code>
     *           <status><![CDATA[Bad Request]]></status>
     *           <trackId><![CDATA[e11853991e]]></trackId>
     *           <date><![CDATA[2016-12-05T10:04:39+01:00]]></date>
     *           <scriptTimeSec>3.1904</scriptTimeSec>
     *           <attributes>
     *               <attribute>
     *                   <code>400</code>
     *                   <status><![CDATA[Bad Request]]></status>
     *                   <errors>
     *                       <error>
     *                           <code>400</code>
     *                           <message><![CDATA[ErrorCode: attributeUpdate00 ErrorRef: v001 ShortMessage: Invalid id]]></message>
     *                           <exception><![CDATA[external]]></exception>
     *                       </error>
     *                   </errors>
     *               </attribute>
     *           </attributes>
     *       </attributeUpdate>
     *
     * ##Response error status 401, 416 or 500
     *
     * **Example JSON (body)**
     *
     *       {
     *       "code": 416,
     *       "status": "Not Found",
     *       "trackId": "2d008cc4c5",
     *       "scriptTimeSec": 1.521,
     *       "count": 1,
     *       "errors": [
     *          {
     *          "code": 416,
     *          "status": "Not Found",
     *          "message": "no valid json string",
     *          "type": "external",
     *          "exception": "InvalidRequest",
     *          "debug": "not set"
     *          }
     *         ]
     *       }
     *
     * **Example XML (body)**
     *
     *      <?xml version="1.0" encoding="UTF-8"?>
     *      <response>
     *          <code>416</code>
     *          <status><![CDATA[Not Found]]></status>
     *          <trackId><![CDATA[16fe523df4]]></trackId>
     *          <scriptTimeSec>2.4459</scriptTimeSec>
     *          <count>1</count>
     *          <errors>
     *              <error>
     *                <code>416</code>
     *                <status><![CDATA[Not Found]]></status>
     *                <message><![CDATA[expected '>']]></message>
     *                <type><![CDATA[external]]></type>
     *                <exception><![CDATA[InvalidRequest]]></exception>
     *                <debug><![CDATA[not set]]></debug>
     *              </error>
     *          </errors>
     *      </response>
     *
     * @ApiDoc(
     *  resource="/api/{version}/sso-ws-object/attribute",
     *  description="secured: Updates multiple attributes in sso-object-ws",
     *  statusCodes={
     *         200="Returned when successful",
     *         204="Returned when no content was send",
     *         400="Returned when an error occurred",
     *         401="Unauthorized - Wrong credentials",
     *         403="Returned when the user is not authorized to do this request",
     *         416="Returned when the request is malformed",
     *         500="Returned on internal server errors"
     *  },
     *  headers={
     *      {
     *        "name"="username",
     *        "description"="sso username",
     *        "required"=true
     *       },
     *      {
     *        "name"="useridentifier",
     *        "description"="sso useridentifier",
     *        "required"=true
     *       },
     *      {
     *        "name"="servicetoken",
     *        "description"="servicetoken",
     *        "required"=true
     *       }
     *  },
     *  tags={
     *      "stable" = "#000"
     *  },
     *  output="Sso\RestBundle\ModelRest\Response\ObjectWs\Version1\Attribute\Update\Response",
     *  views = { "sso-ws-object" }
     * )
     *
     * @Put("/v1/sso-ws-object/attribute/update", name="_sso-ws_v1", options={ "method_prefix" = true })
     * @Secure(roles="ROLE_API")
     * @param Request $request
     * @return View
     */
    public function putAttributeUpdateAction(Request $request)
    {
        return $this->woM->objectWs()->version1()->attribute()->update()->init($request);
    }
}
