<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2018 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\RestBundle\Controller\ObjectWs\Version1;

use FOS\RestBundle\View\View;
use JMS\SecurityExtraBundle\Annotation\Secure;
use Nelmio\ApiDocBundle\Annotation\ApiDoc;
use LifeStyle\Tools\RestAuthBundle\Security\Authentication\Token\UidentifyToken;
use FOS\RestBundle\Controller\Annotations\NamePrefix;
use Symfony\Component\HttpFoundation\Request;
use FOS\RestBundle\Controller\FOSRestController;
use Sso\RestBundle\Worker\Worker as Worker;
use JMS\DiExtraBundle\Annotation as DI;
use FOS\RestBundle\Controller\Annotations\Get;
use FOS\RestBundle\Controller\Annotations\Post;
use FOS\RestBundle\Controller\Annotations\Put;
use LifeStyle\Tools\CachingBundle\Annotations\LsCache;

/**
 * Class ObjectTypeController
 *
 * @NamePrefix("api_rest_")
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package    Sso\RestBundle\Controller
 */
class ObjectTypeController extends FOSRestController
{
    /**
     * @DI\Inject("rest_worker_manager")
     * @var Worker
     */
    protected $woM;

    /**
     * @var UidentifyToken
     */
    protected $userToken;

    /**
     *
     * *This service adds object types to the sso-object-ws.
     * The service expects a JSON in body, and a header authentication.*
     *
     * #Request (header)
     *
     *      Key                 | Value               | Description
     *      ------------------- | ------------------- | --------------------------------------
     *      username            |  (string)           | your username
     *      useridentifier      |  (string)           | your useridentifier for given username
     *      servicetoken        |  (string)           | servicetoken from ws
     *
     * #Request (body)
     *
     *
     * **Request Example JSON**
     *
     *      {
     *           "objectType": [
     *             {
     *               "name": "Location",
     *               "order": "1"
     *             }
     *           ]
     *         }
     * **Request Example XML**
     *
     *
     *       <?xml version="1.0"?>
     *        <objectType>
     *           <item>
     *               <name>Location</name>
     *               <order>1</order>
     *           </item>
     *       </objectType>
     *
     *
     * #Response success
     *
     * **Example (header)**
     *
     *     200 OK
     *
     *
     * **Example JSON (body)**
     *
     *      {
     *         "objectType": [
     *            {
     *             "status": "200",
     *             "name": "Location",
     *             "order": "1",
     *             "created": "015-09-01T15:38:51+0200"
     *             ]
     *           }
     *        ]
     *      }
     *
     * **Example XML (body)**
     *
     *      <?xml version="1.0" encoding="UTF-8"?>
     *       <object>
     *         <object>
     *           <item>
     *               <status><![CDATA[200]]></status>
     *               <name><![CDATA[Location]]></name>
     *               <order><![CDATA[1]]></order>
     *               <created><![CDATA[2015-09-01T15:38:51+0200]]></created>
     *           </item>
     *         </object>
     *       </object>
     *
     *
     *
     * #Response error
     *
     * **Example JSON (body)**
     *
     *      {
     *          "object": [
     *            {
     *              "status": "400",
     *              "name": "Location",
     *              "order": "1",
     *              "created": "",
     *              "error": {
     *                "status": 400,
     *                "message": "Bad Request",
     *                "trackid": "922618f61b",
     *                "count": 1,
     *                "errors": [
     *                  {
     *                    "code": 400,
     *                    "message": "ErrorCode: typeAdd01 ErrorRef: v001 ShortMessage: Invalid Name",
     *                    "exception": "external"
     *                  }
     *                ]
     *              }
     *            }
     *          ]
     *        }
     *
     * **Example XML (body)**
     *
     *      <?xml version="1.0" encoding="UTF-8"?>
     *       <object>
     *         <object>
     *           <item>
     *             <status><![CDATA[400]]></status>
     *             <name><![CDATA[Location]]></name>
     *             <order><![CDATA[1]]></name>
     *             <created><![CDATA[]]></created>
     *             <error>
     *               <status>400</status>
     *               <message><![CDATA[Bad Request]]></message>
     *               <trackid><![CDATA[7b7abf1734]]></trackid>
     *               <count>1</count>
     *               <errors>
     *                 <error>
     *                   <code>400</code>
     *                   <message><![CDATA[ErrorCode: typeAdd01 ErrorRef: v001 ShortMessage: Invalid Name]]></message>
     *                   <exception><![CDATA[external]]></exception>
     *                 </error>
     *               </errors>
     *             </error>
     *           </item>
     *         </object>
     *       </object>
     *
     *
     * @ApiDoc(
     *  resource="/api/{version}/sso-ws-object/objectType",
     *  description="secured: Adds an object type to sso-object-ws",
     *  statusCodes={
     *         200="Returned when successful",
     *         204="Returned when no content was send",
     *         400="Returned when an error occured",
     *         401="Unauthorized - Wrong credentials",
     *         403="Returned when the user is not authorized to do this request",
     *         500="Returned on internal server errors"
     *  },
     *  headers={
     *      {"name"="username", "description"="sso username"},
     *      {"name"="useridentifier", "description"="sso useridentifier"},
     *      {"name"="servicetoken", "description"="servicetoken"}
     *  },
     *  tags={
     *      "stable" = "#000"
     *  },
     *  views = { "sso-ws-object" }
     * )
     *
     * @Put("/v1/sso-ws-object/objectType/add", name="_sso-ws-object_v1", options={ "method_prefix" = true })
     * @Secure(roles="ROLE_API")
     * @param Request $request
     * @return View
     */
    public function putObjectTypeAction(Request $request)
    {
        return $this->woM->objectWs()->version1()->objectType()->add()->init($request);
    }

    /**
     *
     * *This service updates object types to the sso-object-ws.
     * The service expects a JSON in body, and a header authentication.*
     *
     * #Request (header)
     *
     *      Key                 | Value               | Description
     *      ------------------- | ------------------- | --------------------------------------
     *      username            |  (string)           | your username
     *      useridentifier      |  (string)           | your useridentifier for given username
     *      servicetoken        |  (string)           | servicetoken from ws
     *
     * #Request (body)
     *
     *
     * **Request Example JSON**
     *
     *      {
     *        "objectType": [
     *          {
     *            "id": "5F35B266-0B62-2AED-4332-268D7838CFA3",
     *            "name": "My Type Name",
     *            "order": "10"
     *          }
     *        ]
     *      }
     *
     * **Request Example XML**
     *
     *
     *      <?xml version="1.0"?>
     *        <objectType>
     *        <item>
     *          <id>5F35B266-0B62-2AED-4332-268D7838CFA3</id>
     *          <name>My Type Name</name>
     *          <order>10</order>
     *        </item>
     *      </objectType>
     *
     *
     * #Response success
     *
     * **Example (header)**
     *
     *     200 OK
     *
     *
     * **Example JSON (body)**
     *
     *      {
     *        "objectType": [
     *          {
     *            "id": "5F35B266-0B62-2AED-4332-268D7838CFA3",
     *            "status": "200",
     *            "name": "My Type Name",
     *            "order": 10,
     *            "created": "2015-10-14T18:38:12+0200",
     *            "updated": "2015-10-22T14:50:53+0200"
     *          }
     *        ]
     *      }
     *
     * **Example XML (body)**
     *
     *      <?xml version="1.0" encoding="UTF-8"?>
     *      <objectType>
     *        <objectType>
     *          <item>
     *            <id><![CDATA[5F35B266-0B62-2AED-4332-268D7838CFA3]]></id>
     *            <status><![CDATA[200]]></status>
     *            <name><![CDATA[My Type Name]]></name>
     *            <order>10</order>
     *            <created><![CDATA[2015-10-14T18:38:12+0200]]></created>
     *            <updated><![CDATA[2015-10-22T14:49:22+0200]]></updated>
     *          </item>
     *        </objectType>
     *      </objectType>
     *
     *
     *
     * #Response error
     *
     * **Example JSON (body)**
     *
     *      {
     *        "objectType": [
     *          {
     *            "id": "5aaF35B266-0B62-2AED-4332-268D7838CFA3",
     *            "status": "400",
     *            "name": "My Type Name",
     *            "order": 10,
     *            "error": {
     *              "status": 400,
     *              "message": "Bad Request",
     *              "trackid": "b3f76a8378",
     *              "count": 1,
     *              "errors": [
     *                {
     *                  "code": 400,
     *                  "message": "ErrorCode: typeUpdate00 ErrorRef: v001 ShortMessage: Invalid Id",
     *                  "exception": "external"
     *                }
     *              ]
     *            }
     *          }
     *        ]
     *      }
     *
     * **Example XML (body)**
     *
     *      <?xml version="1.0" encoding="UTF-8"?>
     *        <objectType>
     *          <objectType>
     *            <item>
     *              <id><![CDATA[5aaF35B266-0B62-2AED-4332-268D7838CFA3]]></id>
     *              <status><![CDATA[400]]></status>
     *              <name><![CDATA[Cedrik Update Rest Test]]></name>
     *              <order>0</order>
     *              <error>
     *                <status>400</status>
     *                <message><![CDATA[Bad Request]]></message>
     *                <trackid><![CDATA[30a23cf107]]></trackid>
     *                <count>1</count>
     *                <errors>
     *                  <error>
     *                    <code>400</code>
     *                    <message><![CDATA[ErrorCode: typeUpdate00 ErrorRef: v001 ShortMessage: Invalid Id]]></message>
     *                    <exception><![CDATA[external]]></exception>
     *                  </error>
     *                </errors>
     *              </error>
     *            </item>
     *          </objectType>
     *        </objectType>
     *
     * @ApiDoc(
     *  resource="/api/{version}/sso-ws-object/objectType",
     *  description="secured: Updates an object type from sso-object-ws",
     *  statusCodes={
     *         200="Returned when successful",
     *         204="Returned when no content was send",
     *         400="Returned when an error occured",
     *         401="Unauthorized - Wrong credentials",
     *         403="Returned when the user is not authorized to do this request",
     *         500="Returned on internal server errors"
     *  },
     * headers={
     *      {"name"="username", "description"="sso username"},
     *      {"name"="useridentifier", "description"="sso useridentifier"},
     *      {"name"="servicetoken", "description"="servicetoken"}
     *  },
     *  tags={
     *      "stable" = "#000"
     *  },
     *  views = { "sso-ws-object" }
     * )
     * @Post("/v1/sso-ws-object/objectType/update", name="_sso-ws-object_v1", options={ "method_prefix" = true })
     * @Secure(roles="ROLE_API")
     * @param Request $request
     * @return View
     */
    public function postObjectTypeAction(Request $request)
    {
        return $this->woM->objectWs()->version1()->objectType()->update()->init($request);
    }

    /**
     *
     * This service is used to get all or a single object type by its unique id or by its name.
     * Returns a list of object types.
     *
     *#Request (header)
     *
     *      Key                 | Value               | Description
     *      ------------------- | ------------------- | --------------------------------------
     *      username            |  (string)           | your username
     *      useridentifier      |  (string)           | your useridentifier for given username
     *      servicetoken        |  (string)           | servicetoken from ws
     *
     * ##Request
     *
     * **Request example JSON**
     *
     *          /api/v1/object/type.xml?id=YOURID&name=YOURNAME
     *
     * **Request example XML**
     *
     *          /api/v1/object/type.xml?id=YOURID&name=YOURNAME
     *
     * ##Response success
     *
     * **Example (header)**
     *
     *          200 OK
     *
     * **Example JSON (body)**
     *
     *       {
     *           "code": 200,
     *           "status": "OK",
     *           "date": "2017-05-31T14:12:19+02:00",
     *           "scriptTimeSec": 2.9813,
     *           "count": "2",
     *           "types": [
     *               {
     *                   "id": "3dea5ad8-ea1a-455d-9924-292b6ffe1cf2",
     *                   "name": "test",
     *                   "order": "2",
     *                   "attributeGroups": [
     *                       {
     *                           "id": "0f38e869-6fdd-47d2-818a-bad3637b62b3",
     *                           "name": "TestGroup",
     *                           "created": "2017-05-18T12:37:03+02:00",
     *                           "updated": "2017-05-18T12:37:03+02:00"
     *                       }
     *                   ],
     *                   "created": "2016-10-20T12:16:27+02:00",
     *                   "updated": "2017-05-31T14:12:19+02:00"
     *               },
     *               {
     *                   "id": "8a5038ff-5c02-491c-bdbb-82998f4cc381",
     *                   "name": "Location",
     *                   "order": "1",
     *                   "created": "2016-06-06T17:02:53+02:00",
     *                   "updated": "2017-05-31T14:12:19+02:00"
     *               }
     *           ]
     *       }
     *
     * **Example XML (body)**
     *
     *       <?xml version="1.0" encoding="UTF-8"?>
     *       <objectType>
     *           <code>200</code>
     *           <status><![CDATA[OK]]></status>
     *           <date><![CDATA[2017-05-31T14:16:59+02:00]]></date>
     *           <scriptTimeSec>8.1321</scriptTimeSec>
     *           <count><![CDATA[2]]></count>
     *           <type>
     *               <id><![CDATA[3dea5ad8-ea1a-455d-9924-292b6ffe1cf2]]></id>
     *               <name><![CDATA[test]]></name>
     *               <order><![CDATA[2]]></order>
     *               <attributeGroups>
     *                   <attributeGroup>
     *                       <id><![CDATA[0f38e869-6fdd-47d2-818a-bad3637b62b3]]></id>
     *                       <name><![CDATA[TestGroup]]></name>
     *                       <created><![CDATA[2017-05-18T12:37:03+02:00]]></created>
     *                       <updated><![CDATA[2017-05-18T12:37:03+02:00]]></updated>
     *                   </attributeGroup>
     *               </attributeGroups>
     *               <created><![CDATA[2016-10-20T12:16:27+02:00]]></created>
     *               <updated><![CDATA[2017-05-31T14:17:00+02:00]]></updated>
     *           </type>
     *           <type>
     *               <id><![CDATA[8a5038ff-5c02-491c-bdbb-82998f4cc381]]></id>
     *               <name><![CDATA[Location]]></name>
     *               <order><![CDATA[1]]></order>
     *               <created><![CDATA[2016-06-06T17:02:53+02:00]]></created>
     *               <updated><![CDATA[2017-05-31T14:17:00+02:00]]></updated>
     *           </type>
     *       </objectType>
     *
     * ##Response error
     *
     * **Example (header)**
     *
     *          400 Bad Request
     *
     * **Example JSON (body)**
     *
     *          {
     *            "status": "400",
     *            "message": "Bad Request",
     *            "trackid": "2b6f9f9647",
     *            "count": "1",
     *            "errors": [
     *              {
     *                "code": "400",
     *                "message": "ErrorCode: No types found. ErrorRef: ts001 ShortMessage: No types found.",
     *                "exception": "external"
     *              }
     *            ]
     *          }
     *
     *
     * **Example XML (body)**
     *
     *          <?xml version="1.0" encoding="UTF-8"?>
     *            <SsoResponse>
     *              <status><![CDATA[400]]></status>
     *              <message><![CDATA[Bad Request]]></message>
     *              <trackid><![CDATA[6f3838ac82]]></trackid>
     *              <count><![CDATA[1]]></count>
     *              <errors>
     *                <error>
     *                  <code><![CDATA[400]]></code>
     *                  <message><![CDATA[ErrorCode: No types found. ErrorRef: ts001 ShortMessage: No types found.]]></message>
     *                  <exception><![CDATA[external]]></exception>
     *                </error>
     *              </errors>
     *            </SsoResponse>
     *
     * ##Response error status 401, 416 or 500
     *
     * **Example JSON (body)**
     *
     *       {
     *       "code": 416,
     *       "status": "Not Found",
     *       "trackId": "2d008cc4c5",
     *       "scriptTimeSec": 1.521,
     *       "count": 1,
     *       "errors": [
     *          {
     *          "code": 416,
     *          "status": "Not Found",
     *          "message": "no valid json string",
     *          "type": "external",
     *          "exception": "InvalidRequest",
     *          "debug": "not set"
     *          }
     *         ]
     *       }
     *
     * **Example XML (body)**
     *
     *      <?xml version="1.0" encoding="UTF-8"?>
     *      <response>
     *          <code>416</code>
     *          <status><![CDATA[Not Found]]></status>
     *          <trackId><![CDATA[16fe523df4]]></trackId>
     *          <scriptTimeSec>2.4459</scriptTimeSec>
     *          <count>1</count>
     *          <errors>
     *              <error>
     *                <code>416</code>
     *                <status><![CDATA[Not Found]]></status>
     *                <message><![CDATA[expected '>']]></message>
     *                <type><![CDATA[external]]></type>
     *                <exception><![CDATA[InvalidRequest]]></exception>
     *                <debug><![CDATA[not set]]></debug>
     *              </error>
     *          </errors>
     *      </response>
     *
     * @ApiDoc(
     *  resource="/api/{version}/sso-ws-object/objectType",
     *  description="secured: Returns a list of object types from sso-object-ws",
     *  statusCodes={
     *         200="Returned when successful",
     *         204="Returned when no content was send",
     *         400="Returned when an error occurred",
     *         401="Unauthorized - Wrong credentials",
     *         403="Returned when the user is not authorized to do this request",
     *         416="Returned when the request is malformed",
     *         500="Returned on internal server errors"
     *  },
     *  input="Sso\RestBundle\ModelRest\Request\ObjectWs\Version1\ObjectType\Show\Request",
     *  headers={
     *      {
     *        "name"="username",
     *        "description"="sso username",
     *        "required"=true
     *       },
     *      {
     *        "name"="useridentifier",
     *        "description"="sso useridentifier",
     *        "required"=true
     *       },
     *      {
     *        "name"="servicetoken",
     *        "description"="servicetoken",
     *        "required"=true
     *       }
     *  },
     *  tags={
     *      "stable" = "#000"
     *  },
     *  views = { "sso-ws-object" }
     * )
     *
     * @Secure(roles="ROLE_API")
     * @LsCache(cacheTTL=300, keyHeader=true)
     * @Get("/v1/sso-ws-object/objectType/show", name="_sso-ws-object_v1", options={ "method_prefix" = true })
     * @param Request $request
     * @return View
     */
    public function getObjectTypeAction(Request $request)
    {
        return $this->woM->objectWs()->version1()->objectType()->show()->init($request);
    }
}
