<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2018 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\RestBundle\Controller\ObjectWs\Version1;

use FOS\RestBundle\View\View;
use GuzzleHttp\Client;
use JMS\SecurityExtraBundle\Annotation\Secure;
use Nelmio\ApiDocBundle\Annotation\ApiDoc;
use FOS\RestBundle\Controller\Annotations\NamePrefix;
use Symfony\Component\HttpFoundation\Request;
use FOS\RestBundle\Controller\FOSRestController;
use Sso\RestBundle\Worker\Worker as Worker;
use JMS\DiExtraBundle\Annotation as DI;
use FOS\RestBundle\Controller\Annotations\Delete;
use FOS\RestBundle\Controller\Annotations\Get;
use FOS\RestBundle\Controller\Annotations\Put;
use LifeStyle\Tools\CachingBundle\Annotations\LsCache;

/**
 * Class ObjectValueController
 *
 * @NamePrefix("api_rest_")
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package    Sso\RestBundle\Controller
 */
class ObjectValueController extends FOSRestController
{
    /**
     * @DI\Inject("rest_worker_manager")
     * @var Worker
     */
    protected $woM;

    /**
     *
     * This service is used to get all or a single object value by its id.
     * Returns a list of object values.
     *
     * #Request (header)
     *
     *      Key                 | Value               | Description
     *      ------------------- | ------------------- | --------------------------------------
     *      username            |  (string)           | your username
     *      useridentifier      |  (string)           | your useridentifier for given username
     *      servicetoken        |  (string)           | servicetoken from ws
     *
     * ##Request
     *
     * **Request example JSON**
     *
     *          /api/v1/object/value.json?id=YOURID
     *
     * **Request example XML**
     *
     *          /api/v1/object/value.json?id=YOURID
     *
     * ##Response success
     *
     * **Example (header)**
     *
     *          200 OK
     *
     * **Example JSON (body)**
     *
     *          {
     *            "status": "Success",
     *            "trackid": "f4a64b9316",
     *            "date": "2016-04-26T15:55:32+0200",
     *            "count": "4",
     *            "objectValues": [
     *              {
     *                "id": "6BDB537C-883C-9B4D-2599-1DB001D2AB8F",
     *                "value": "TestValueEnglish",
     *                "languageCode": "en",
     *                "objectGuid": "E138F8B4-BCEE-52AC-180E-068D0F4CCC5A",
     *                "objectName": "Object 3.1",
     *                "attributeId": "0C7D874E-A114-BF02-81D4-EC7935C748CA",
     *                "attributeName": "TestAttribute1",
     *                "created": "2016-03-18T10:02:08+0100",
     *                "updated": "2016-03-18T10:02:08+0100"
     *              },
     *              {
     *                "id": "EA5C96E6-9522-F128-1F86-F5A4DB3C0A6C",
     *                "value": "TestValueGerman",
     *                "languageCode": "de",
     *                "objectGuid": "E138F8B4-BCEE-52AC-180E-068D0F4CCC5A",
     *                "objectName": "Object 3.1",
     *                "attributeId": "229B72CA-1927-68B2-5AEB-E87C74AF717E",
     *                "attributeName": "TestAttribute2",
     *                "created": "2016-03-18T10:02:08+0100",
     *                "updated": "2016-03-18T10:02:08+0100"
     *              }
     *            ]
     *          }
     *
     * **Example XML (body)**
     *
     *          <?xml version="1.0" encoding="UTF-8"?>
     *            <SsoResponse>
     *              <status><![CDATA[Success]]></status>
     *              <trackid><![CDATA[5e5a970858]]></trackid>
     *              <date><![CDATA[2016-04-26T16:01:01+0200]]></date>
     *              <count><![CDATA[4]]></count>
     *              <objectValues>
     *                <id><![CDATA[66BDB537C-883C-9B4D-2599-1DB001D2AB8F]]></id>
     *                <value><![CDATA[TestValueEnglish]]></value>
     *                <languageCode><![CDATA[en]]></languageCode>
     *                <objectGuid><![CDATA[E138F8B4-BCEE-52AC-180E-068D0F4CCC5A]]></objectGuid>
     *                <objectName><![CDATA[Object 3.1]]></objectName>
     *                <attributeId><![CDATA[0C7D874E-A114-BF02-81D4-EC7935C748CA]]></attributeId>
     *                <attributeName><![CDATA[TestAttribute1]]></attributeName>
     *                <created><![CDATA[2016-03-18T10:02:08+0100]]></created>
     *                <updated><![CDATA[2016-03-18T10:02:08+0100]]></updated>
     *              </objectValues>
     *              <objectValues>
     *                <id><![CDATA[EA5C96E6-9522-F128-1F86-F5A4DB3C0A6C]]></id>
     *                <value><![CDATA[TestValueGerman]]></value>
     *                <languageCode><![CDATA[de]]></languageCode>
     *                <objectGuid><![CDATA[E138F8B4-BCEE-52AC-180E-068D0F4CCC5A]]></objectGuid>
     *                <objectName><![CDATA[Object 3.1]]></objectName>
     *                <attributeId><![CDATA[0229B72CA-1927-68B2-5AEB-E87C74AF717E]]></attributeId>
     *                <attributeName><![CDATA[TestAttribute2]]></attributeName>
     *                <created><![CDATA[2016-03-18T10:01:43+0100]]></created>
     *                <updated><![CDATA[2016-03-18T10:01:43+0100]]></updated>
     *              </objectValues>
     *            </SsoResponse>
     *
     * ##Response error
     *
     * **Example (header)**
     *
     *          400 Bad Request
     *
     * **Example JSON (body)**
     *
     *          {
     *            "status": "400",
     *            "message": "Bad Request",
     *            "trackid": "b491280401",
     *            "count": "1",
     *            "errors": [
     *              {
     *                "code": "400",
     *                "message": "ErrorCode: No object values found. ErrorRef: os001 ShortMessage: No object values found.",
     *                "exception": "external"
     *              }
     *            ]
     *          }
     *
     *
     * **Example XML (body)**
     *
     *          <?xml version="1.0" encoding="UTF-8"?>
     *            <SsoResponse>
     *              <status><![CDATA[400]]></status>
     *              <message><![CDATA[Bad Request]]></message>
     *              <trackid><![CDATA[115ca5c2c1]]></trackid>
     *              <count><![CDATA[1]]></count>
     *              <errors>
     *                <error>
     *                  <code><![CDATA[400]]></code>
     *                  <message><![CDATA[ErrorCode: No object values found. ErrorRef: os001 ShortMessage: No object values found.]]></message>
     *                  <exception><![CDATA[external]]></exception>
     *                </error>
     *              </errors>
     *            </SsoResponse>
     *
     *
     *
     * @ApiDoc(
     *  resource="/api/{version}/sso-ws-object/objectValue",
     *  description="secured: Returns a list of object attribute values from sso-object-ws",
     *  statusCodes={
     *         200="Returned when successful",
     *         400="Returned when an error occured",
     *         401="Unauthorized - Wrong credentials",
     *         403="Returned when the user is not authorized to do this request",
     *         500="Returned on internal server errors"
     *  },
     *  input="Sso\RestBundle\ModelRest\Request\ObjectWs\Version1\ObjectValue\Show\Request",
     *  tags={
     *      "untested" = "#aaa"
     *  },
     *  headers={
     *      {"name"="username", "description"="sso username", "required"=true},
     *      {"name"="useridentifier", "description"="sso useridentifier", "required"=true},
     *      {"name"="servicetoken", "description"="servicetoken", "required"=true}
     *  },
     *  views = { "sso-ws-object" }
     * )
     *
     * @Secure(roles="ROLE_API")
     * @LsCache(cacheTTL=300, keyHeader=true)
     * @Get("/v1/sso-ws-object/objectValue/show", name="_sso-ws-object_v1", options={ "method_prefix" = true })
     * @param Request $request
     * @return View
     */
    public function getObjectValueAction(Request $request)
    {
        return $this->woM->objectWs()->version1()->objectValue()->show()->init($request);
    }

    /**
     *
     * *This service deletes single or multiple object values from the object-webservice by its ids.
     * The service expects a JSON or XML in body, and a header authentication.*
     *
     * #Request (header)
     *
     *      Key                 | Value               | Description
     *      ------------------- | ------------------- | --------------------------------------
     *      username            |  (string)           | your username
     *      useridentifier      |  (string)           | your useridentifier for given username
     *      servicetoken        |  (string)           | servicetoken from ws
     *
     * #Request (body)
     *
     *
     * **Request Example JSON**
     *
     *       {
     *         "objectValue": [
     *           {
     *             "id": "6BDB537C-883C-9B4D-2599-1DB001D2AB8F"
     *           },
     *           {
     *             "id": "CB430866-CDE7-30F3-A9DE-1F74727124EC"
     *           }
     *         ]
     *       }
     *
     * **Request Example XML**
     *
     *
     *       <?xml version="1.0"?>
     *         <objectValue>
     *           <item>
     *             <id>6BDB537C-883C-9B4D-2599-1DB001D2AB8F</id>
     *           </item>
     *           <item>
     *             <idCB430866-CDE7-30F3-A9DE-1F74727124EC</id>
     *           </item>
     *         </objectValue>
     *
     *
     * #Response success
     *
     * **Example (header)**
     *
     *     200 OK
     *
     *
     * **Example JSON (body)**
     *
     *      {
     *        "objectValue": [
     *          {
     *            "status": "200",
     *            "trackId": "b45d14ed3a",
     *            "id": "6BDB537C-883C-9B4D-2599-1DB001D2AB8F"
     *          },
     *          {
     *            "status": "200",
     *            "trackId": "6c4c00c486",
     *            "id": "CB430866-CDE7-30F3-A9DE-1F74727124EC"
     *          }
     *        ]
     *      }
     *
     * **Example XML (body)**
     *
     *      <?xml version="1.0" encoding="UTF-8"?>
     *        <objectValue>
     *          <objectValue>
     *            <item>
     *              <status><![CDATA[200]]></status>
     *              <trackId><![CDATA[845908f592]]></trackId>
     *              <id><![CDATA[6BDB537C-883C-9B4D-2599-1DB001D2AB8F]]></id>
     *            </item>
     *            <item>
     *              <status><![CDATA[200]]></status>
     *              <trackId><![CDATA[e829e38460]]></trackId>
     *              <id><![CDATA[CB430866-CDE7-30F3-A9DE-1F74727124EC]]></id>
     *            </item>
     *          </objectValue>
     *        </objectValue>
     *
     *
     * #Response error
     *
     * **Example JSON (body)**
     *
     *      {
     *        "objectValue": [
     *          {
     *            "status": "200",
     *            "trackId": "75eb4274bb",
     *            "id": "6BDB537C-883C-9B4D-2599-1DB001D2AB8F"
     *          },
     *          {
     *            "status": "400",
     *            "id": "UNKNOWNN-CDE7-30F3-A9DE-1F74727124EC",
     *            "error": {
     *              "status": 400,
     *              "message": "Bad Request",
     *              "trackid": "24e506b04c",
     *              "count": 1,
     *              "errors": [
     *                {
     *                  "code": 400,
     *                  "message": "ErrorCode: ObjectValue not found ErrorRef: ovd001 ShortMessage: ObjectValue not found",
     *                  "exception": "external"
     *                }
     *              ]
     *            }
     *          }
     *        ]
     *      }
     *
     * **Example XML (body)**
     *
     *      <?xml version="1.0" encoding="UTF-8"?>
     *        <objectValue>
     *          <objectValue>
     *            <item>
     *              <status><![CDATA[200]]></status>
     *              <trackId><![CDATA[b2b0bdbf3a]]></trackId>
     *              <id><![CDATA[6BDB537C-883C-9B4D-2599-1DB001D2AB8F]]></id>
     *            </item>
     *            <item>
     *              <status><![CDATA[400]]></status>
     *              <id><![CDATA[UNKNOWNN-CDE7-30F3-A9DE-1F74727124EC]]></id>
     *              <error>
     *                <status>400</status>
     *                <message><![CDATA[Bad Request]]></message>
     *                <trackid><![CDATA[d685982b1a]]></trackid>
     *                <count>1</count>
     *                <errors>
     *                  <error>
     *                    <code>400</code>
     *                    <message><![CDATA[ErrorCode: ErrorCode: ObjectValue not found ErrorRef: ovd001 ShortMessage: ObjectValue not found]]></message>
     *                    <exception><![CDATA[external]]></exception>
     *                  </error>
     *                </errors>
     *              </error>
     *            </item>
     *          </objectValue>
     *        </objectValue>
     *
     *
     * @ApiDoc(
     *  resource="/api/{version}/sso-ws-object/objectValue",
     *  description="secured: Deletes object values from sso-object-ws",
     *  statusCodes={
     *         200="Returned when successful",
     *         204="Returned when no content was send",
     *         400="Returned when an error occured",
     *         401="Unauthorized - Wrong credentials",
     *         403="Returned when the user is not authorized to do this request",
     *         500="Returned on internal server errors"
     *  },
     *  headers={
     *      {"name"="username", "description"="sso username"},
     *      {"name"="useridentifier", "description"="sso useridentifier"},
     *      {"name"="servicetoken", "description"="servicetoken"}
     *  },
     *  tags={
     *      "stable" = "#000"
     *  },
     *  views = { "sso-ws-object" }
     *
     * )
     *
     *
     * @Secure(roles="ROLE_API")
     * @Delete("/v1/sso-ws-object/objectValue/delete", name="_sso-ws-object_v1", options={ "method_prefix" = true })
     * @param Request $request
     * @return View
     */
    public function deleteObjectValueAction(Request $request)
    {
        $client = new Client();
        return $this->woM->objectWs()->version1()->objectValue()->delete($client)->init($request);
    }

    /**
     *
     *
     * *This service adds multiple object values to the sso-object-ws.
     * The service expects a JSON in body, and a header authentication.*
     *
     * #Request (header)
     *
     *      Key                 | Value               | Description
     *      ------------------- | ------------------- | --------------------------------------
     *      username            |  (string)           | your username
     *      useridentifier      |  (string)           | your useridentifier for given username
     *      servicetoken        |  (string)           | servicetoken from ws
     *
     * #Request (body)
     *
     *
     * **Request Example JSON**
     *
     *       {
     *         "objectValue": [
     *           {
     *             "objectGuid": "1ABE53B9-1926-14E0-B77D-1F8FE316263E",
     *             "objectValue": "myTestValueEN",
     *             "languageCode": "en",
     *             "attributeId": "0C7D874E-A114-BF02-81D4-EC7935C748CA"
     *           },
     *           {
     *             "objectGuid": "1ABE53B9-1926-14E0-B77D-1F8FE316263E",
     *             "objectValue": "myTestValueDE",
     *             "languageCode": "de",
     *             "attributeId": "0C7D874E-A114-BF02-81D4-EC7935C748CA"
     *           }
     *         ]
     *       }
     *
     * **Request Example XML**
     *
     *
     *      <?xml version="1.0"?>
     *        <objectValue>
     *          <item>
     *            <objectGuid>1ABE53B9-1926-14E0-B77D-1F8FE316263E</objectGuid>
     *            <objectValue>myTestValueEN</objectValue>
     *            <languageCode>en</languageCode>
     *            <attributeId>0C7D874E-A114-BF02-81D4-EC7935C748CA</attributeId>
     *          </item>
     *          <item>
     *            <objectGuid>1ABE53B9-1926-14E0-B77D-1F8FE316263E</objectGuid>
     *            <objectValue>myTestValueDE</objectValue>
     *            <languageCode>de</languageCode>
     *            <attributeId>0C7D874E-A114-BF02-81D4-EC7935C748CA</attributeId>
     *          </item>
     *        </objectValue>
     *
     *
     *
     * #Response success
     *
     * **Example (header)**
     *
     *     200 OK
     *
     *
     * **Example JSON (body)**
     *
     *      {
     *        "objectValue": [
     *          {
     *            "status": "200",
     *            "objectGuid": "1ABE53B9-1926-14E0-B77D-1F8FE316263E",
     *            "objectName": "TestObject",
     *            "attributeId": "0C7D874E-A114-BF02-81D4-EC7935C748CA",
     *            "attributeName": "TestAttribute",
     *            "id": "96431665-38B7-3995-F16C-F0FDEB21A9C6",
     *            "value": "myTestValueEN",
     *            "languageCode": "en",
     *            "created": "2016-05-20T08:15:25+0200",
     *            "updated": "2016-05-20T08:15:25+0200"
     *          },
     *          {
     *            "status": "200",
     *            "objectGuid": "1ABE53B9-1926-14E0-B77D-1F8FE316263E",
     *            "objectName": "TestObject",
     *            "attributeId": "0C7D874E-A114-BF02-81D4-EC7935C748CA",
     *            "attributeName": "TestAttribute",
     *            "id": "2E473B54-EEC3-CE53-6FFD-2A468B2A1301",
     *            "value": "myTestValueDE",
     *            "languageCode": "de",
     *            "created": "2016-05-20T08:15:27+0200",
     *            "updated": "2016-05-20T08:15:27+0200"
     *          }
     *        ]
     *      }
     *
     * **Example XML (body)**
     *
     *      <?xml version="1.0" encoding="UTF-8"?>
     *        <objectValue>
     *          <objectValue>
     *            <item>
     *            <status><![CDATA[200]]></status>
     *            <objectGuid><![CDATA[1ABE53B9-1926-14E0-B77D-1F8FE316263E]]></objectGuid>
     *            <objectName><![CDATA[Object3]]></objectName>
     *            <attributeId><![CDATA[0C7D874E-A114-BF02-81D4-EC7935C748CA]]></attributeId>
     *            <attributeName><![CDATA[TestAttribute5]]></attributeName>
     *            <id><![CDATA[C09BCE10-9476-D831-B4A5-687503047372]]></id>
     *            <value><![CDATA[myTestValueEN]]></value>
     *            <languageCode><![CDATA[en]]></languageCode>
     *            <created><![CDATA[2016-05-20T08:19:15+0200]]></created>
     *            <updated><![CDATA[2016-05-20T08:19:15+0200]]></updated>
     *          </item>
     *          <item>
     *            <status><![CDATA[200]]></status>
     *            <objectGuid><![CDATA[1ABE53B9-1926-14E0-B77D-1F8FE316263E]]></objectGuid>
     *            <objectName><![CDATA[Object3]]></objectName>
     *            <attributeId><![CDATA[0C7D874E-A114-BF02-81D4-EC7935C748CA]]></attributeId>
     *            <attributeName><![CDATA[TestAttribute5]]></attributeName>
     *            <id><![CDATA[061D5D4E-6160-5C46-C485-0BA836197139]]></id>
     *            <value><![CDATA[myTestValueDE]]></value>
     *            <languageCode><![CDATA[de]]></languageCode>
     *            <created><![CDATA[2016-05-20T08:19:17+0200]]></created>
     *            <updated><![CDATA[2016-05-20T08:19:17+0200]]></updated>
     *          </item>
     *        </objectValue>
     *      </objectValue>
     *
     *
     *
     *
     * #Response error
     *
     * **Example JSON (body)**
     *
     *      {
     *        "objectValue": [
     *          {
     *            "status": "400",
     *            "error": {
     *              "status": 400,
     *              "message": "Bad Request",
     *              "trackid": "cdc10e6755",
     *              "count": 1,
     *              "errors": [
     *                {
     *                  "code": 400,
     *                  "message": "ErrorCode: object not found ErrorRef: v001 ShortMessage: object not found",
     *                  "exception": "external"
     *                }
     *              ]
     *            }
     *          },
     *          {
     *            "status": "200",
     *            "objectGuid": "1ABE53B9-1926-14E0-B77D-1F8FE316263E",
     *            "objectName": "Object3",
     *            "attributeId": "0C7D874E-A114-BF02-81D4-EC7935C748CA",
     *            "attributeName": "TestAttribute5",
     *            "id": "92B12FF4-0A53-A2CF-07E8-E756AF0C1397",
     *            "value": "myTestValueDE",
     *            "languageCode": "de",
     *            "created": "2016-05-20T08:25:39+0200",
     *            "updated": "2016-05-20T08:25:39+0200"
     *          }
     *        ]
     *      }
     *
     * **Example XML (body)**
     *
     *      <?xml version="1.0" encoding="UTF-8"?>
     *        <response>
     *          <status>400</status>
     *          <message><![CDATA[Bad Request]]></message>
     *          <count>1</count>
     *          <errors>
     *            <error>
     *              <code>400</code>
     *              <message><![CDATA[item[1].attributeId: attributeId should have exactly 36 characters]]></message>
     *              <type><![CDATA[external]]></type>
     *              <exception><![CDATA[not set]]></exception>
     *              <debug><![CDATA[not set]]></debug>
     *            </error>
     *          </errors>
     *        </response>
     *
     *
     * @ApiDoc(
     *  resource="/api/{version}/sso-ws-object/objectValue",
     *  description="secured: Adds multiple object values to sso-object-ws",
     *  statusCodes={
     *         200="Returned when successful",
     *         204="Returned when no content was send",
     *         400="Returned when an error occured",
     *         401="Unauthorized - Wrong credentials",
     *         403="Returned when the user is not authorized to do this request",
     *         500="Returned on internal server errors"
     *  },
     *  headers={
     *      {"name"="username", "description"="sso username"},
     *      {"name"="useridentifier", "description"="sso useridentifier"},
     *      {"name"="servicetoken", "description"="servicetoken"}
     *  },
     *  tags={
     *      "stable" = "#000"
     *  },
     *  views = { "sso-ws-object" }
     * )
     *
     * @Put("/v1/sso-ws-object/objectValue/add", name="_sso-ws-object_v1", options={ "method_prefix" = true })
     * @Secure(roles="ROLE_API")
     * @param Request $request
     * @return View
     */
    public function putObjectValueAction(Request $request)
    {
        return $this->woM->objectWs()->version1()->objectValue()->add()->init($request);
    }

    /**
     *
     * *This service updates object values in the sso-object-ws. The service expects a JSON or XML in body and a header authentication.*
     *
     * #Request (header)
     *
     *      Key                 | Value               | Description
     *      ------------------- | ------------------- | --------------------------------------
     *      username            |  (string)           | your username
     *      useridentifier      |  (string)           | your useridentifier for given username
     *      servicetoken        |  (string)           | servicetoken from ws
     *
     * ##Request
     *
     * **Request Example JSON**
     *
     *       {
     *           "values":[
     *               {
     *                   "id": "0007ef3b-ec62-4625-86f7-808a7029e830",
     *                   "value":"testValue",
     *                   "languageCode": "en"
     *               }
     *           ]
     *       }
     *
     * **Request Example XML**
     *
     *       <?xml version="1.0" encoding="UTF-8"?>
     *       <objectValueUpdate>
     *           <values>
     *               <objectValueType>
     *                   <id>0007ef3b-ec62-4625-86f7-808a7029e830</id>
     *                   <value>testValue</value>
     *                   <languageCode>en</languageCode>
     *               </objectValueType>
     *           </values>
     *       </objectValueUpdate>
     *
     * #Response success
     *
     * **Example JSON (body)**
     *
     *       {
     *           "code": 200,
     *           "status": "OK",
     *           "trackId": "db805109d0",
     *           "date": "2016-12-07T14:30:38+01:00",
     *           "scriptTimeSec": 2.0452,
     *           "objectValue": [
     *               {
     *                   "code": 200,
     *                   "status": "OK",
     *                   "id": "0007ef3b-ec62-4625-86f7-808a7029e830",
     *                   "value": "sdfsdfsdf",
     *                   "languageCode": "aa",
     *                   "created": "2016-10-20T17:30:47+0200",
     *                   "updated": "2016-12-06T09:51:13+0100"
     *               }
     *           ]
     *       }
     *
     * **Example XML (body)**
     *
     *       <?xml version="1.0" encoding="UTF-8"?>
     *       <objectValueUpdate>
     *           <code>200</code>
     *           <status><![CDATA[OK]]></status>
     *           <trackId><![CDATA[537caea8d7]]></trackId>
     *           <date><![CDATA[2016-12-07T14:16:58+01:00]]></date>
     *           <scriptTimeSec>4.8701</scriptTimeSec>
     *           <objectValue>
     *               <objectValueType>
     *                   <code>200</code>
     *                   <status><![CDATA[OK]]></status>
     *                   <id><![CDATA[0007ef3b-ec62-4625-86f7-808a7029e830]]></id>
     *                   <value><![CDATA[sdfsdfsdf]]></value>
     *                   <languageCode><![CDATA[aa]]></languageCode>
     *                   <created><![CDATA[2016-10-20T17:30:47+0200]]></created>
     *                   <updated><![CDATA[2016-12-06T09:51:13+0100]]></updated>
     *               </objectValueType>
     *           </objectValue>
     *       </objectValueUpdate>
     *
     * #Response error
     *
     * **Example JSON (body)**
     *
     *       {
     *           "code": 400,
     *           "status": "Bad Request",
     *           "trackId": "db805109d0",
     *           "date": "2016-12-07T14:30:38+01:00",
     *           "scriptTimeSec": 2.0452,
     *           "attributeGroup": [
     *               {
     *                   "code": 400,
     *                   "status": "Bad Request",
     *                   "errors": [
     *                       {
     *                           "code": 400,
     *                           "message": "ErrorCode: objectValueUpdate00 ErrorRef: v001 ShortMessage: Invalid id",
     *                           "exception": "external"
     *                       }
     *                   ]
     *               }
     *           ]
     *       }
     *
     * **Example XML (body)**
     *
     *       <?xml version="1.0" encoding="UTF-8"?>
     *       <objectValueUpdate>
     *           <code>400</code>
     *           <status><![CDATA[Bad Request]]></status>
     *           <trackId><![CDATA[537caea8d7]]></trackId>
     *           <date><![CDATA[2016-12-07T14:16:58+01:00]]></date>
     *           <scriptTimeSec>4.8701</scriptTimeSec>
     *           <objectValue>
     *               <objectValueType>
     *                   <code>400</code>
     *                   <status><![CDATA[Bad Request]]></status>
     *                   <errors>
     *                       <error>
     *                           <code>400</code>
     *                           <message><![CDATA[ErrorCode: objectValueUpdate00 ErrorRef: v001 ShortMessage: Invalid id]]></message>
     *                           <exception><![CDATA[external]]></exception>
     *                       </error>
     *                   </errors>
     *               </objectValueType>
     *           </objectValue>
     *       </objectValueUpdate>
     *
     * ##Response error status 401, 416 or 500
     *
     * **Example JSON (body)**
     *
     *       {
     *       "code": 416,
     *       "status": "Not Found",
     *       "trackId": "2d008cc4c5",
     *       "scriptTimeSec": 1.521,
     *       "count": 1,
     *       "errors": [
     *          {
     *          "code": 416,
     *          "status": "Not Found",
     *          "message": "no valid json string",
     *          "type": "external",
     *          "exception": "InvalidRequest",
     *          "debug": "not set"
     *          }
     *         ]
     *       }
     *
     * **Example XML (body)**
     *
     *      <?xml version="1.0" encoding="UTF-8"?>
     *      <response>
     *          <code>416</code>
     *          <status><![CDATA[Not Found]]></status>
     *          <trackId><![CDATA[16fe523df4]]></trackId>
     *          <scriptTimeSec>2.4459</scriptTimeSec>
     *          <count>1</count>
     *          <errors>
     *              <error>
     *                <code>416</code>
     *                <status><![CDATA[Not Found]]></status>
     *                <message><![CDATA[expected '>']]></message>
     *                <type><![CDATA[external]]></type>
     *                <exception><![CDATA[InvalidRequest]]></exception>
     *                <debug><![CDATA[not set]]></debug>
     *              </error>
     *          </errors>
     *      </response>
     *
     * @ApiDoc(
     *  resource="/api/{version}/sso-ws-object/objectValue",
     *  description="secured: Updates multiple object values in sso-object-ws",
     *  statusCodes={
     *         200="Returned when successful",
     *         204="Returned when no content was send",
     *         400="Returned when an error occurred",
     *         401="Unauthorized - Wrong credentials",
     *         403="Returned when the user is not authorized to do this request",
     *         416="Returned when the request is malformed",
     *         500="Returned on internal server errors"
     *  },
     *  headers={
     *      {
     *        "name"="username",
     *        "description"="sso username",
     *        "required"=true
     *       },
     *      {
     *        "name"="useridentifier",
     *        "description"="sso useridentifier",
     *        "required"=true
     *       },
     *      {
     *        "name"="servicetoken",
     *        "description"="servicetoken",
     *        "required"=true
     *       }
     *  },
     *  tags={
     *      "stable" = "#000"
     *  },
     *  output="Sso\RestBundle\ModelRest\Response\ObjectWs\Version1\ObjectValue\Update\Response",
     *  views = { "sso-ws-object" }
     * )
     *
     * @Put("/v1/sso-ws-object/objectValue/update", name="_sso-ws_v1", options={ "method_prefix" = true })
     * @Secure(roles="ROLE_API")
     * @param Request $request
     * @return View
     */
    public function putObjectValueUpdateAction(Request $request)
    {
        return $this->woM->objectWs()->version1()->objectValue()->update()->init($request);
    }
}
