<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2018 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\RestBundle\Controller\UserWs\Version1;

use FOS\RestBundle\View\View;
use GuzzleHttp\Client;
use JMS\SecurityExtraBundle\Annotation\Secure;
use Nelmio\ApiDocBundle\Annotation\ApiDoc;
use FOS\RestBundle\Controller\Annotations\NamePrefix;
use Symfony\Component\HttpFoundation\Request;
use FOS\RestBundle\Controller\FOSRestController;
use Sso\RestBundle\Worker\Worker as Worker;
use JMS\DiExtraBundle\Annotation as DI;
use FOS\RestBundle\Controller\Annotations\Get;
use FOS\RestBundle\Controller\Annotations\Delete;
use FOS\RestBundle\Controller\Annotations\Put;
use LifeStyle\Tools\CachingBundle\Annotations\LsCache;

/**
 * Class AttributeController
 *
 * @NamePrefix("api_rest_")
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package    Sso\RestBundle\Controller
 */
class AttributeController extends FOSRestController
{
    /**
     * @DI\Inject("rest_worker_manager")
     * @var Worker
     */
    protected $woM;

    /**
     *
     * This service is used to get all or a single attribute by its id.
     * Returns a list of attributes.
     *
     * #Request (header)
     *
     *      Key                 | Value               | Description
     *      ------------------- | ------------------- | --------------------------------------
     *      username            |  (string)           | your username
     *      useridentifier      |  (string)           | your useridentifier for given username
     *      servicetoken        |  (string)           | servicetoken from ws
     *
     * ##Request
     *
     * **Request example JSON**
     *
     *          /api/v1/attribute.json?id=YOURID&offset=YOUROFFSET&limit=YOURLIMIT
     *
     * **Request example XML**
     *
     *          /api/v1/attribute.xml?id=YOURID&offset=YOUROFFSET&limit=YOURLIMIT
     *
     * ##Response success
     *
     * **Example (header)**
     *
     *          200 OK
     *
     * **Example JSON (body)**
     *
     *          {
     *            "status": "Success",
     *            "trackid": "ebb1779e7b",
     *            "date": "2016-04-26T13:57:50+0200",
     *            "count": "3",
     *            "offset": "0",
     *            "limit": "3",
     *            "attributes": [
     *              {
     *                "id": "0C7D874E-A114-BF02-81D4-EC7935C748CA",
     *                "name": "FirstName"
     *              },
     *              {
     *                "id": "229B72CA-1927-68B2-5AEB-E87C74AF717E",
     *                "name": "LastName"
     *              },
     *              {
     *                "id": "431B6B3D-A276-D925-25E4-3A5071C2070C",
     *                "name": "Phone"
     *              }
     *            ]
     *          }
     *
     * **Example XML (body)**
     *
     *          <?xml version="1.0" encoding="UTF-8"?>
     *            <SsoResponse>
     *              <status><![CDATA[Success]]></status>
     *              <trackid><![CDATA[392029e282]]></trackid>
     *              <date><![CDATA[2016-04-26T13:59:01+0200]]></date>
     *              <count><![CDATA[3]]></count>
     *              <offset><![CDATA[0]]></offset>
     *              <limit><![CDATA[3]]></limit>
     *              <attributes>
     *                <id><![CDATA[0C7D874E-A114-BF02-81D4-EC7935C748CA]]></id>
     *                <name><![CDATA[FirstName]]></name>
     *              </attributes>
     *              <attributes>
     *                <id><![CDATA[229B72CA-1927-68B2-5AEB-E87C74AF717E]]></id>
     *                <name><![CDATA[LastName]]></name>
     *              </attributes>
     *              <attributes>
     *                <id><![CDATA[431B6B3D-A276-D925-25E4-3A5071C2070C]]></id>
     *                <name><![CDATA[Phone]]></name>
     *              </attributes>
     *            </SsoResponse>
     *
     * ##Response error
     *
     * **Example (header)**
     *
     *          400 Bad Request
     *
     * **Example JSON (body)**
     *
     *          {
     *            "status": "400",
     *            "message": "Bad Request",
     *            "trackid": "b491280401",
     *            "count": "1",
     *            "errors": [
     *              {
     *                "code": "400",
     *                "message": "ErrorCode: No attributes found. ErrorRef: os001 ShortMessage: No attributes found.",
     *                "exception": "external"
     *              }
     *            ]
     *          }
     *
     *
     * **Example XML (body)**
     *
     *          <?xml version="1.0" encoding="UTF-8"?>
     *            <SsoResponse>
     *              <status><![CDATA[400]]></status>
     *              <message><![CDATA[Bad Request]]></message>
     *              <trackid><![CDATA[115ca5c2c1]]></trackid>
     *              <count><![CDATA[1]]></count>
     *              <errors>
     *                <error>
     *                  <code><![CDATA[400]]></code>
     *                  <message><![CDATA[ErrorCode: No attributes found. ErrorRef: os001 ShortMessage: No attributes found.]]></message>
     *                  <exception><![CDATA[external]]></exception>
     *                </error>
     *              </errors>
     *            </SsoResponse>
     *
     *
     *
     * @ApiDoc(
     *  resource="/api/{version}/sso-ws/attribute",
     *  description="secured: Returns a list of attributes from sso-object-ws",
     *  statusCodes={
     *         200="Returned when successful",
     *         400="Returned when an error occured",
     *         401="Unauthorized - Wrong credentials",
     *         403="Returned when the user is not authorized to do this request",
     *         500="Returned on internal server errors"
     *  },
     *  headers={
     *      {"name"="username", "description"="sso username"},
     *      {"name"="useridentifier", "description"="sso useridentifier"},
     *      {"name"="servicetoken", "description"="servicetoken"}
     *  },
     *  input="Sso\RestBundle\ModelRest\Request\UserWs\Version1\Attribute\Show\Request",
     *  views = { "sso-ws" },
     *  tags={
     *      "stable" = "#000"
     *  }
     * )
     *
     * @Get("/v1/sso-ws/attribute/show", name="_sso-ws_v1", options={ "method_prefix" = true })
     * @LsCache(cacheTTL=300, keyHeader=true)
     * @Secure(roles="ROLE_API")
     *
     * @param Request $request
     * @return View
     */
    public function getAttributeAction(Request $request)
    {
        return $this->woM->userWs()->version1()->attribute()->show()->init($request);
    }

    /**
     *
     * *This service deletes single or multiple attributes from the object-webservice by its ids.
     * The service expects a JSON or XML in body, and a header authentication.*
     *
     * #Request (header)
     *
     *      Key                 | Value               | Description
     *      ------------------- | ------------------- | --------------------------------------
     *      username            |  (string)           | your username
     *      useridentifier      |  (string)           | your useridentifier for given username
     *      servicetoken        |  (string)           | servicetoken from ws
     *
     * #Request (body)
     *
     *
     * **Request Example JSON**
     *
     *       {
     *         "attribute": [
     *           {
     *             "id": "CB7D874E-A114-BF02-81D4-EC7935C748CA"
     *           },
     *           {
     *             "id": "229B72CA-1927-68B2-5AEB-E87C74AF717E",
     *             "deleteReferences": true
     *           }
     *         ]
     *       }
     *
     * **Request Example XML**
     *
     *
     *       <?xml version="1.0"?>
     *         <attribute>
     *           <item>
     *             <id>CB7D874E-A114-BF02-81D4-EC7935C748CA</id>
     *           </item>
     *           <item>
     *             <id>1229B72CA-1927-68B2-5AEB-E87C74AF717E</id>
     *             <deleteReferences>false</deleteReferences>
     *           </item>
     *         </attribute>
     *
     *
     * #Response success
     *
     * **Example (header)**
     *
     *     200 OK
     *
     *
     * **Example JSON (body)**
     *
     *      {
     *        "attribute": [
     *          {
     *            "status": "200",
     *            "trackId": "b45d14ed3a",
     *            "id": "CB7D874E-A114-BF02-81D4-EC7935C748CA"
     *          },
     *          {
     *            "status": "200",
     *            "trackId": "6c4c00c486",
     *            "id": "B72CA-1927-68B2-5AEB-E87C74AF717E"
     *          }
     *        ]
     *      }
     *
     * **Example XML (body)**
     *
     *      <?xml version="1.0" encoding="UTF-8"?>
     *        <attribute>
     *          <attribute>
     *            <item>
     *              <status><![CDATA[200]]></status>
     *              <trackId><![CDATA[845908f592]]></trackId>
     *              <id><![CDATA[CB7D874E-A114-BF02-81D4-EC7935C748CA]]></id>
     *            </item>
     *            <item>
     *              <status><![CDATA[200]]></status>
     *              <trackId><![CDATA[e829e38460]]></trackId>
     *              <id><![CDATA[B72CA-1927-68B2-5AEB-E87C74AF717E]]></id>
     *            </item>
     *          </attribute>
     *        </attribute>
     *
     *
     * #Response error
     *
     * **Example JSON (body)**
     *
     *      {
     *        "attribute": [
     *          {
     *            "status": "200",
     *            "trackId": "75eb4274bb",
     *            "id": "CB7D874E-A114-BF02-81D4-EC7935C748CA"
     *          },
     *          {
     *            "status": "400",
     *            "id": "B72CA-1927-68B2-5AEB-E87C74AF717E",
     *            "error": {
     *              "status": 400,
     *              "message": "Bad Request",
     *              "trackid": "24e506b04c",
     *              "count": 1,
     *              "errors": [
     *                {
     *                  "code": 400,
     *                  "message": "ErrorCode: Can not delete Attribute ErrorRef: ad002 ShortMessage: Attribute has AttributeValues!",
     *                  "exception": "external"
     *                }
     *              ]
     *            }
     *          },
     *          {
     *            "status": "400",
     *            "id": "UNKNOWNN-1927-68B2-5AEB-E87C74AF717E",
     *            "error": {
     *              "status": 400,
     *              "message": "Bad Request",
     *              "trackid": "8ff721c4fa",
     *              "count": 1,
     *              "errors": [
     *                {
     *                  "code": 400,
     *                  "message": "ErrorCode: attribute id not found ErrorRef: ad001 ShortMessage: attribute id not found",
     *                  "exception": "external"
     *                }
     *              ]
     *            }
     *          }
     *        ]
     *      }
     *
     * **Example XML (body)**
     *
     *      <?xml version="1.0" encoding="UTF-8"?>
     *        <attribute>
     *          <attribute>
     *            <item>
     *              <status><![CDATA[200]]></status>
     *              <trackId><![CDATA[b2b0bdbf3a]]></trackId>
     *              <id><![CDATA[CB7D874E-A114-BF02-81D4-EC7935C748CA]]></id>
     *            </item>
     *            <item>
     *              <status><![CDATA[400]]></status>
     *              <id><![CDATA[B72CA-1927-68B2-5AEB-E87C74AF717E]]></id>
     *              <error>
     *                <status>400</status>
     *                <message><![CDATA[Bad Request]]></message>
     *                <trackid><![CDATA[d685982b1a]]></trackid>
     *                <count>1</count>
     *                <errors>
     *                  <error>
     *                    <code>400</code>
     *                    <message><![CDATA[ErrorCode: Can not delete Attribute ErrorRef: ad002 ShortMessage: Attribute has AttributeValues!]]></message>
     *                    <exception><![CDATA[external]]></exception>
     *                  </error>
     *                </errors>
     *              </error>
     *            </item>
     *            <item>
     *              <status><![CDATA[400]]></status>
     *              <id><![CDATA[UNKNOWNN-1927-68B2-5AEB-E87C74AF717E]]></id>
     *              <error>
     *                <status>400</status>
     *                <message><![CDATA[Bad Request]]></message>
     *                <trackid><![CDATA[ba3ab814b7]]></trackid>
     *                <count>1</count>
     *                <errors>
     *                  <error>
     *                    <code>400</code>
     *                    <message><![CDATA[ErrorCode: attribute id not found ErrorRef: ad001 ShortMessage: attribute id not found]]></message>
     *                    <exception><![CDATA[external]]></exception>
     *                  </error>
     *                </errors>
     *              </error>
     *            </item>
     *          </attribute>
     *        </attribute>
     *
     *
     * @ApiDoc(
     *  resource="/api/{version}/sso-ws/attribute",
     *  description="secured: Deletes attributes from sso-object-ws",
     *  statusCodes={
     *         200="Returned when successful",
     *         204="Returned when no content was send",
     *         400="Returned when an error occured",
     *         401="Unauthorized - Wrong credentials",
     *         403="Returned when the user is not authorized to do this request",
     *         500="Returned on internal server errors"
     *  },
     *  headers={
     *      {"name"="username", "description"="sso username"},
     *      {"name"="useridentifier", "description"="sso useridentifier"},
     *      {"name"="servicetoken", "description"="servicetoken"}
     *  },
     *  views = { "sso-ws" },
     *  tags={
     *      "stable" = "#000"
     *  }
     * )
     *
     * @Delete("/v1/sso-ws/attribute/delete", name="_sso-ws_v1", options={ "method_prefix" = true })
     * @Secure(roles="ROLE_API")
     *
     * @param Request $request
     * @return View
     */
    public function deleteAttributeAction(Request $request)
    {
        return $this->woM->userWs()->version1()->attribute()->delete()->init($request);
    }

    /**
     *
     * *This service adds multiple attributes to the sso-object-ws.
     * The service expects a JSON in body, and a header authentication.*
     *
     * #Request (header)
     *
     *      Key                 | Value               | Description
     *      ------------------- | ------------------- | --------------------------------------
     *      username            |  (string)           | your username
     *      useridentifier      |  (string)           | your useridentifier for given username
     *      servicetoken        |  (string)           | servicetoken from ws
     *
     * #Request (body)
     *
     *
     * **Request Example JSON**
     *
     *       {
     *         "attribute": [
     *           {
     *             "name": "Weight",
     *             "groupId": "027958B9-1521-6DAD-3AB8-A8F509BF574E"
     *           },
     *           {
     *             "name": "Size",
     *             "groupId": "027958B9-1521-6DAD-3AB8-A8F509BF574E"
     *           }
     *         ]
     *       }
     *
     * **Request Example XML**
     *
     *
     *      <?xml version="1.0"?>
     *        <attribute>
     *          <item>
     *            <name>Weight</name>
     *            <groupId>027958B9-1521-6DAD-3AB8-A8F509BF574E</groupId>
     *          </item>
     *          <item>
     *            <name>Size</name>
     *            <groupId>027958B9-1521-6DAD-3AB8-A8F509BF574E</groupId>
     *          </item>
     *        </attribute>
     *
     *
     *
     *
     * #Response success
     *
     * **Example (header)**
     *
     *     200 OK
     *
     *
     * **Example JSON (body)**
     *
     *      {
     *        "attribute": [
     *          {
     *            "status": "200",
     *            "id": "45591B42-207C-D8EB-1EE4-7C100C8DEC0B",
     *            "name": "Weight",
     *            "groupId": "027958B9-1521-6DAD-3AB8-A8F509BF574E"
     *          },
     *          {
     *            "status": "200",
     *            "id": "46B800D8-E617-DAAE-769A-8531A8D53342",
     *            "name": "Size",
     *            "groupId": "027958B9-1521-6DAD-3AB8-A8F509BF574E"
     *          }
     *        ]
     *      }
     *
     * **Example XML (body)**
     *
     *      <?xml version="1.0" encoding="UTF-8"?>
     *        <attribute>
     *          <attribute>
     *            <item>
     *              <status><![CDATA[200]]></status>
     *              <id><![CDATA[E96D61CD-C54F-DB0B-4F34-03966B8F3E60]]></id>
     *              <name><![CDATA[Weight]]></name>
     *              <groupId><![CDATA[027958B9-1521-6DAD-3AB8-A8F509BF574E]]></groupId>
     *            </item>
     *            <item>
     *              <status><![CDATA[200]]></status>
     *              <id><![CDATA[75DFC9F4-079B-F9E8-6B62-E8ED90B79197]]></id>
     *              <name><![CDATA[Size]]></name>
     *              <groupId><![CDATA[027958B9-1521-6DAD-3AB8-A8F509BF574E]]></groupId>
     *            </item>
     *          </attribute>
     *        </attribute>
     *
     *
     *
     *
     * #Response error
     *
     * **Example JSON (body)**
     *
     *      {
     *        "status": 400,
     *        "message": "Bad Request",
     *        "count": 3,
     *        "errors": [
     *          {
     *            "code": 400,
     *            "message": "item[0].attributeGroupId: Your group id must be 36 characters long",
     *            "type": "external",
     *            "exception": "not set",
     *            "debug": "not set"
     *          },
     *          {
     *            "code": 400,
     *            "message": "item[1].attributeName: Attribute Name should not be empty",
     *            "type": "external",
     *            "exception": "not set",
     *            "debug": "not set"
     *          },
     *          {
     *            "code": 400,
     *            "message": "item[2].attributeGroupId: Your group id should not be empty",
     *            "type": "external",
     *            "exception": "not set",
     *            "debug": "not set"
     *          }
     *        ]
     *      }
     *
     * **Example XML (body)**
     *
     *      <?xml version="1.0" encoding="UTF-8"?>
     *        <attribute>
     *          <attribute>
     *            <item>
     *              <status><![CDATA[400]]></status>
     *              <error>
     *                <status>400</status>
     *                <message><![CDATA[Bad Request]]></message>
     *                <trackid><![CDATA[dede3d3be6]]></trackid>
     *                <count>1</count>
     *                <errors>
     *                  <error>
     *                    <code>400</code>
     *                    <message><![CDATA[ErrorCode: attributeGroup id not found ErrorRef: v001 ShortMessage: attributeGroup id not found]]></message>
     *                    <exception><![CDATA[external]]></exception>
     *                  </error>
     *                </errors>
     *              </error>
     *            </item>
     *            <item>
     *              <status><![CDATA[200]]></status>
     *              <id><![CDATA[92D394B3-B950-892D-9B59-EA47AA2C248A]]></id>
     *              <name><![CDATA[Size]]></name>
     *              <groupId><![CDATA[027958B9-1521-6DAD-3AB8-A8F509BF574E]]></groupId>
     *            </item>
     *          </attribute>
     *        </attribute>
     *
     *
     * @ApiDoc(
     *  resource="/api/{version}/sso-ws/attribute",
     *  description="secured: Adds multiple attributes to sso-object-ws",
     *  statusCodes={
     *         200="Returned when successful",
     *         204="Returned when no content was send",
     *         400="Returned when an error occured",
     *         401="Unauthorized - Wrong credentials",
     *         403="Returned when the user is not authorized to do this request",
     *         500="Returned on internal server errors"
     *  },
     *  headers={
     *      {"name"="username", "description"="sso username"},
     *      {"name"="useridentifier", "description"="sso useridentifier"},
     *      {"name"="servicetoken", "description"="servicetoken"}
     *  },
     *  tags={
     *      "stable" = "#000"
     *  },
     *  views = { "sso-ws" }
     * )
     *
     * @Put("/v1/sso-ws/attribute/add", name="_sso-ws_v1", options={ "method_prefix" = true })
     * @Secure(roles="ROLE_API")
     * @param Request $request
     * @return View
     */
    public function putAttributeAction(Request $request)
    {
        return $this->woM->userWs()->version1()->attribute()->add()->init($request);
    }
}
