<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2018 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\RestBundle\Controller\UserWs\Version1;

use FOS\RestBundle\Controller\Annotations\Delete;
use FOS\RestBundle\Controller\Annotations\Get;
use FOS\RestBundle\Controller\Annotations\Post;
use FOS\RestBundle\View\View;
use JMS\SecurityExtraBundle\Annotation\Secure;
use Nelmio\ApiDocBundle\Annotation\ApiDoc;
use FOS\RestBundle\Controller\Annotations\NamePrefix;
use Symfony\Component\HttpFoundation\Request;
use FOS\RestBundle\Controller\FOSRestController;
use Sso\RestBundle\Worker\Worker as Worker;
use JMS\DiExtraBundle\Annotation as DI;
use LifeStyle\Tools\CachingBundle\Annotations\LsCache;

/**
 * @NamePrefix("api_rest_")
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package    Sso\RestBundle\Controller
 */
class UserApplicationAttributeController extends FOSRestController
{
    /**
     * @DI\Inject("rest_worker_manager")
     * @var Worker
     */
    protected $woM;

    /**
     *
     * *This service adds application attributes to the sso-ws user. The service expects a JSON or XML in body and a header authentication.*
     *
     * #Request (header)
     *
     *      Key                 | Value               | Description
     *      ------------------- | ------------------- | --------------------------------------
     *      username            |  (string)           | your username
     *      useridentifier      |  (string)           | your useridentifier for given username
     *      servicetoken        |  (string)           | servicetoken from ws
     *
     * ##Request
     *
     * **Request Example JSON**
     *
     *       {
     *           "userAttributes": [
     *               {
     *                   "name": "testAttribute",
     *                   "type": "one",
     *                   "value": "testValue",
     *                   "userType": {
     *                       "identifier": "testUser"
     *                   },
     *                   "applicationType": {
     *                       "name": "testApplication"
     *                   }
     *               },
     *               {
     *                   "name": "Fruits",
     *                   "type": "many",
     *                   "value": ["Apple", "Orange"],
     *                   "userType": {
     *                       "identifier": "testUser"
     *                   },
     *                   "applicationType": {
     *                       "name": "testApplication"
     *                   }
     *               }
     *           ]
     *       }
     *
     * **Request Example XML**
     *
     *       <?xml version="1.0" encoding="UTF-8"?>
     *       <userAttributeAdd>
     *           <userAttributes>
     *               <attributeType>
     *                   <name>testAttribute</name>
     *                   <type>one</type>
     *                   <value>testValue</value>
     *                   <userType>
     *                       <identifier>testUser</identifier>
     *                   </userType>
     *                   <applicationType>
     *                       <name>testApplication</name>
     *                   </applicationType>
     *               </attributeType>
     *               <attributeType>
     *                   <name>Fruits</name>
     *                   <type>many</type>
     *                   <value>Apple</value>
     *                   <value>Orange</value>
     *                   <userType>
     *                       <identifier>testUser</identifier>
     *                   </userType>
     *                   <applicationType>
     *                       <name>testApplication</name>
     *                   </applicationType>
     *               </attributeType>
     *           </userAttributes>
     *       </userAttributeAdd>
     *
     * #Response success
     *
     * **Example JSON (body)**
     *
     *       {
     *           "code": 200,
     *           "status": "OK",
     *           "trackId": "1f6424dfb8",
     *           "date": "2016-11-21T08:50:52+01:00",
     *           "scriptTimeSec": 4.3799,
     *           "userAttributes": [
     *               {
     *                   "code": 200,
     *                   "status": "OK",
     *                   "message": "User application added"
     *               }
     *           ]
     *       }
     *
     * **Example XML (body)**
     *
     *       <?xml version="1.0" encoding="UTF-8"?>
     *       <userApplicationAttributeAdd>
     *           <code>200</code>
     *           <status><![CDATA[OK]]></status>
     *           <trackId><![CDATA[54bb1c647e]]></trackId>
     *           <date><![CDATA[2016-11-21T08:47:04+01:00]]></date>
     *           <scriptTimeSec>1.2066</scriptTimeSec>
     *           <userAttributes>
     *               <attributeType>
     *                   <code>200</code>
     *                   <status><![CDATA[OK]]></status>
     *                   <message><![CDATA[User application added]]></message>
     *               </attributeType>
     *           </userAttributes>
     *       </userApplicationAttributeAdd>
     *
     * #Response error
     *
     * **Example JSON (body)**
     *
     *       {
     *           "code": 400,
     *           "status": "Bad Request",
     *           "trackId": "7426346a16",
     *           "date": "2016-11-21T08:55:21+01:00",
     *           "scriptTimeSec": 5.653,
     *           "roles": [
     *               {
     *                   "code": 400,
     *                   "status": "Bad Request",
     *                   "errors": [
     *                       {
     *                           "code": 400,
     *                           "message": "ErrorCode: u002 ErrorRef: uat023 ShortMessage: Invalid user application",
     *                           "exception": "external"
     *                       }
     *                   ]
     *               }
     *           ]
     *       }
     *
     * **Example XML (body)**
     *
     *       <?xml version="1.0" encoding="UTF-8"?>
     *       <userApplicationAttributeAdd>
     *           <code>400</code>
     *           <status><![CDATA[Bad Request]]></status>
     *           <trackId><![CDATA[7e72d2d30a]]></trackId>
     *           <date><![CDATA[22016-11-21T08:55:22+01:00]]></date>
     *           <scriptTimeSec>4.7045</scriptTimeSec>
     *           <userAttributes>
     *               <attributeType>
     *                   <code>400</code>
     *                   <status><![CDATA[Bad Request]]></status>
     *                   <errors>
     *                       <error>
     *                           <code>400</code>
     *                           <message><![CDATA[ErrorCode: u002 ErrorRef: uat023 ShortMessage: Invalid user application]]></message>
     *                           <exception><![CDATA[external]]></exception>
     *                       </error>
     *                   </errors>
     *               </attributeType>
     *           </userAttributes>
     *       </userApplicationAttributeAdd>
     *
     * ##Response error status 401, 416 or 500
     *
     * **Example JSON (body)**
     *
     *       {
     *       "code": 416,
     *       "status": "Not Found",
     *       "trackId": "2d008cc4c5",
     *       "scriptTimeSec": 1.521,
     *       "count": 1,
     *       "errors": [
     *          {
     *          "code": 416,
     *          "status": "Not Found",
     *          "message": "no valid json string",
     *          "type": "external",
     *          "exception": "InvalidRequest",
     *          "debug": "not set"
     *          }
     *         ]
     *       }
     *
     * **Example XML (body)**
     *
     *      <?xml version="1.0" encoding="UTF-8"?>
     *      <response>
     *          <code>416</code>
     *          <status><![CDATA[Not Found]]></status>
     *          <trackId><![CDATA[16fe523df4]]></trackId>
     *          <scriptTimeSec>2.4459</scriptTimeSec>
     *          <count>1</count>
     *          <errors>
     *              <error>
     *                <code>416</code>
     *                <status><![CDATA[Not Found]]></status>
     *                <message><![CDATA[expected '>']]></message>
     *                <type><![CDATA[external]]></type>
     *                <exception><![CDATA[InvalidRequest]]></exception>
     *                <debug><![CDATA[not set]]></debug>
     *              </error>
     *          </errors>
     *      </response>
     *
     * @ApiDoc(
     *  resource="/api/{version}/sso-ws/userApplicationAttribute",
     *  description="secured: Adds multiple user attributes to sso-ws user",
     *  statusCodes={
     *         200="Returned when successful",
     *         204="Returned when no content was send",
     *         400="Returned when an error occurred",
     *         401="Unauthorized - Wrong credentials",
     *         403="Returned when the user is not authorized to do this request",
     *         416="Returned when the request is malformed",
     *         500="Returned on internal server errors"
     *  },
     *  headers={
     *      {
     *        "name"="username",
     *        "description"="sso username",
     *        "required"=true
     *       },
     *      {
     *        "name"="useridentifier",
     *        "description"="sso useridentifier",
     *        "required"=true
     *       },
     *      {
     *        "name"="servicetoken",
     *        "description"="servicetoken",
     *        "required"=true
     *       }
     *  },
     *  tags={
     *      "stable" = "#000"
     *  },
     *  output="Sso\RestBundle\ModelRest\Response\UserWs\Version1\UserApplicationAttribute\Add\Response",
     *  views = { "sso-ws" }
     * )
     *
     * @Post("/v1/sso-ws/userApplicationAttribute/add", name="_sso-ws_v1", options={ "method_prefix" = true })
     * @Secure(roles="ROLE_API")
     * @param Request $request
     * @return View
     */
    public function postUserApplicationAttributeAction(Request $request)
    {
        return $this->woM->userWs()->version1()->userApplicationAttribute()->add()->init($request);
    }

    /**
     *
     * *This service returns user application attributes informations. The service expects a JSON or XML in body and a header authentication.*
     *
     * #Request (header)
     *
     *      Key                 | Value               | Description
     *      ------------------- | ------------------- | --------------------------------------
     *      username            |  (string)           | your username
     *      useridentifier      |  (string)           | your useridentifier for given username
     *      servicetoken        |  (string)           | servicetoken from ws
     *
     * #Response success
     *
     * **Example JSON (body)**
     *
     *       {
     *           "code": 200,
     *           "status": "OK",
     *           "trackId": "3b9e4fb8d5",
     *           "date": "2016-11-16T10:45:40+01:00",
     *           "scriptTimeSec": 1.2284,
     *           "roles": [
     *               {
     *                   "code": 200,
     *                   "status": "OK",
     *                   "name": "testRole",
     *                   "description": "Test description"
     *               }
     *           ]
     *       }
     *
     * **Example XML (body)**
     *
     *       <?xml version="1.0" encoding="UTF-8"?>
     *       <applicationRole>
     *           <code>200</code>
     *           <status><![CDATA[OK]]></status>
     *           <trackId><![CDATA[5b3bcd6d4e]]></trackId>
     *           <date><![CDATA[2016-11-16T10:45:05+01:00]]></date>
     *           <scriptTimeSec>1.8048</scriptTimeSec>
     *           <roles>
     *               <roleType>
     *                   <code>200</code>
     *                   <status><![CDATA[OK]]></status>
     *                   <name><![CDATA[testRole]]></name>
     *                   <description><![CDATA[Test description]]></description>
     *               </roleType>
     *           </roles>
     *       </applicationRole>
     *
     * #Response error
     *
     * **Example JSON (body)**
     *
     *       {
     *           "code": 400,
     *           "status": "Bad Request",
     *           "trackId": "badea2f6fb",
     *           "date": "2016-11-16T10:48:05+01:00",
     *           "scriptTimeSec": 1.3689,
     *           "roles": [
     *               {
     *                   "code": 400,
     *                   "status": "Bad Request",
     *                   "errors": [
     *                       {
     *                           "code": 400,
     *                           "message": "ErrorCode: a001 ErrorRef: r031 ShortMessage: Invalid application",
     *                           "exception": "external"
     *                       }
     *                   ]
     *               }
     *           ]
     *       }
     *
     * **Example XML (body)**
     *
     *       <?xml version="1.0" encoding="UTF-8"?>
     *       <applicationRole>
     *           <code>400</code>
     *           <status><![CDATA[Bad Request]]></status>
     *           <trackId><![CDATA[3c3c12d96c]]></trackId>
     *           <date><![CDATA[2016-11-16T10:48:48+01:00]]></date>
     *           <scriptTimeSec>5.4091</scriptTimeSec>
     *           <roles>
     *               <roleType>
     *                   <code>400</code>
     *                   <status><![CDATA[Bad Request]]></status>
     *                   <errors>
     *                       <error>
     *                           <code>400</code>
     *                           <message><![CDATA[ErrorCode: a001 ErrorRef: r031 ShortMessage: Invalid application]]></message>
     *                           <exception><![CDATA[external]]></exception>
     *                       </error>
     *                   </errors>
     *               </roleType>
     *           </roles>
     *       </applicationRole>
     *
     * ##Response error status 401, 416 or 500
     *
     * **Example JSON (body)**
     *
     *       {
     *       "code": 416,
     *       "status": "Not Found",
     *       "trackId": "2d008cc4c5",
     *       "scriptTimeSec": 1.521,
     *       "count": 1,
     *       "errors": [
     *          {
     *          "code": 416,
     *          "status": "Not Found",
     *          "message": "no valid json string",
     *          "type": "external",
     *          "exception": "InvalidRequest",
     *          "debug": "not set"
     *          }
     *         ]
     *       }
     *
     * **Example XML (body)**
     *
     *      <?xml version="1.0" encoding="UTF-8"?>
     *      <response>
     *          <code>416</code>
     *          <status><![CDATA[Not Found]]></status>
     *          <trackId><![CDATA[16fe523df4]]></trackId>
     *          <scriptTimeSec>2.4459</scriptTimeSec>
     *          <count>1</count>
     *          <errors>
     *              <error>
     *                <code>416</code>
     *                <status><![CDATA[Not Found]]></status>
     *                <message><![CDATA[expected '>']]></message>
     *                <type><![CDATA[external]]></type>
     *                <exception><![CDATA[InvalidRequest]]></exception>
     *                <debug><![CDATA[not set]]></debug>
     *              </error>
     *          </errors>
     *      </response>
     *
     * @ApiDoc(
     *  resource="/api/{version}/sso-ws/userApplicationAttribute",
     *  description="secured: Gets user attributes informations from sso-ws user",
     *  statusCodes={
     *         200="Returned when successful",
     *         204="Returned when no content was send",
     *         400="Returned when an error occurred",
     *         401="Unauthorized - Wrong credentials",
     *         403="Returned when the user is not authorized to do this request",
     *         416="Returned when the request is malformed",
     *         500="Returned on internal server errors"
     *  },
     *  headers={
     *      {
     *        "name"="username",
     *        "description"="sso username",
     *        "required"=true
     *       },
     *      {
     *        "name"="useridentifier",
     *        "description"="sso useridentifier",
     *        "required"=true
     *       },
     *      {
     *        "name"="servicetoken",
     *        "description"="servicetoken",
     *        "required"=true
     *       }
     *  },
     *  parameters={
     *      {"name"="roleName", "dataType"="string", "required"=true, "description"="Role name"},
     *      {"name"="applicationName", "dataType"="string", "required"=true, "description"="Application name"}
     *  },
     *  tags={
     *      "in development" = "#d9534f"
     *  },
     *  output="Sso\RestBundle\ModelRest\Response\UserWs\Version1\UserApplicationAttribute\Show\Response",
     *  views = { "sso-ws" }
     * )
     *
     * @Get("/v1/sso-ws/userApplicationAttribute/show", name="_sso-ws_v1", options={ "method_prefix" = true })
     * @Secure(roles="ROLE_API")
     * @LsCache(cacheTTL=300, keyHeader=true)
     * @param Request $request
     * @return View
     */
    public function getUserApplicationAttributeAction(Request $request)
    {
        return $this->woM->userWs()->version1()->userApplicationAttribute()->show()->init($request);
    }

    /**
     *
     * *This service deletes user application attribute. The service expects a JSON or XML in body and a header authentication.*
     *
     * #Request (header)
     *
     *      Key                 | Value               | Description
     *      ------------------- | ------------------- | --------------------------------------
     *      username            |  (string)           | your username
     *      useridentifier      |  (string)           | your useridentifier for given username
     *      servicetoken        |  (string)           | servicetoken from ws
     *
     * ##Request
     *
     * **Request Example JSON**
     *
     *       {
     *           "userAttributes": [
     *               {
     *                   "name": "testAttribute",
     *                   "applicationType": {
     *                       "name": "testApplication"
     *                   },
     *                   "userType": {
     *                       "identifier": "testUser"
     *                   }
     *               }
     *           ]
     *       }
     *
     * **Request Example XML**
     *
     *       <?xml version="1.0" encoding="UTF-8"?>
     *       <userAttributeDelete>
     *           <userAttributes>
     *               <userAttributeType>
     *                   <name>testAttribute</name>
     *                   <applicationType>
     *                       <name>testApplication</name>
     *                   </applicationType>
     *                   <userType>
     *                       <identifier>testUser</identifier>
     *                   </userType>
     *               </userAttributeType>
     *           </userAttributes>
     *      </userAttributeDelete>
     *
     * #Response success
     *
     * **Example JSON (body)**
     *
     *       {
     *           "code": 200,
     *           "status": "OK",
     *           "trackId": "8b5b8e49ea",
     *           "date": "2016-11-16T09:08:15+01:00",
     *           "scriptTimeSec": 5.7169,
     *           "userAttributes": [
     *               {
     *                   "code": 200,
     *                   "status": "OK"
     *                   "message": "Attribute \"testAttribute\" deleted"
     *               }
     *           ]
     *       }
     *
     * **Example XML (body)**
     *
     *       <?xml version="1.0" encoding="UTF-8"?>
     *       <userAttributeDelete>
     *           <code>200</code>
     *           <status><![CDATA[OK]]></status>
     *           <trackId><![CDATA[b1138e8562]]></trackId>
     *           <date><![CDATA[2016-11-16T09:09:17+01:00]]></date>
     *           <scriptTimeSec>4.6424</scriptTimeSec>
     *           <userAttributes>
     *               <attributeType>
     *                   <code>200</code>
     *                   <status><![CDATA[OK]]></status>
     *                   <message><![CDATA[Attribute "testAttribute" deleted]]></message>
     *               </attributeType>
     *           </userAttributes>
     *       </userAttributeDelete>
     *
     * #Response error
     *
     * **Example JSON (body)**
     *
     *       {
     *           "code": 400,
     *           "status": "Bad Request",
     *           "trackId": "dda28d1f77",
     *           "date": "2016-11-21T12:18:49+01:00",
     *           "scriptTimeSec": 4.6506,
     *           "roles": [
     *               {
     *                   "code": 400,
     *                   "status": "Bad Request",
     *                   "errors": [
     *                       {
     *                           "code": 400,
     *                           "message": "ErrorCode: u002 ErrorRef: uat044 ShortMessage: Invalid user application",
     *                           "exception": "external"
     *                       }
     *                   ]
     *               }
     *           ]
     *       }
     *
     * **Example XML (body)**
     *
     *       <?xml version="1.0" encoding="UTF-8"?>
     *       <userAttributeDelete>
     *           <code>400</code>
     *           <status><![CDATA[Bad Request]]></status>
     *           <trackId><![CDATA[70146819db]]></trackId>
     *           <date><![CDATA[2016-11-21T12:17:43+01:00]]></date>
     *           <scriptTimeSec>1.1263</scriptTimeSec>
     *           <roles>
     *               <roleType>
     *                   <code>400</code>
     *                   <status><![CDATA[Bad Request]]></status>
     *                   <errors>
     *                       <error>
     *                           <code>400</code>
     *                           <message><![CDATA[ErrorCode: u002 ErrorRef: uat044 ShortMessage: Invalid user application]]></message>
     *                           <exception><![CDATA[external]]></exception>
     *                       </error>
     *                   </errors>
     *               </roleType>
     *           </roles>
     *       </userAttributeDelete>
     *
     * ##Response error status 401, 416 or 500
     *
     * **Example JSON (body)**
     *
     *       {
     *       "code": 416,
     *       "status": "Not Found",
     *       "trackId": "2d008cc4c5",
     *       "scriptTimeSec": 1.521,
     *       "count": 1,
     *       "errors": [
     *          {
     *          "code": 416,
     *          "status": "Not Found",
     *          "message": "no valid json string",
     *          "type": "external",
     *          "exception": "InvalidRequest",
     *          "debug": "not set"
     *          }
     *         ]
     *       }
     *
     * **Example XML (body)**
     *
     *      <?xml version="1.0" encoding="UTF-8"?>
     *      <response>
     *          <code>416</code>
     *          <status><![CDATA[Not Found]]></status>
     *          <trackId><![CDATA[16fe523df4]]></trackId>
     *          <scriptTimeSec>2.4459</scriptTimeSec>
     *          <count>1</count>
     *          <errors>
     *              <error>
     *                <code>416</code>
     *                <status><![CDATA[Not Found]]></status>
     *                <message><![CDATA[expected '>']]></message>
     *                <type><![CDATA[external]]></type>
     *                <exception><![CDATA[InvalidRequest]]></exception>
     *                <debug><![CDATA[not set]]></debug>
     *              </error>
     *          </errors>
     *      </response>
     *
     * @ApiDoc(
     *  resource="/api/{version}/sso-ws/userApplicationAttribute",
     *  description="secured: Deletes given user attribute",
     *  statusCodes={
     *         200="Returned when successful",
     *         204="Returned when no content was send",
     *         400="Returned when an error occurred",
     *         401="Unauthorized - Wrong credentials",
     *         403="Returned when the user is not authorized to do this request",
     *         416="Returned when the request is malformed",
     *         500="Returned on internal server errors"
     *  },
     *  headers={
     *      {
     *        "name"="username",
     *        "description"="sso username",
     *        "required"=true
     *       },
     *      {
     *        "name"="useridentifier",
     *        "description"="sso useridentifier",
     *        "required"=true
     *       },
     *      {
     *        "name"="servicetoken",
     *        "description"="servicetoken",
     *        "required"=true
     *       }
     *  },
     *  tags={
     *      "stable" = "#000"
     *  },
     *  output="Sso\RestBundle\ModelRest\Response\UserWs\Version1\UserApplicationAttribute\Delete\Response",
     *  views = { "sso-ws" }
     * )
     *
     * @Delete("/v1/sso-ws/userApplicationAttribute/delete", name="_sso-ws_v1", options={ "method_prefix" = true })
     * @Secure(roles="ROLE_API")
     * @param Request $request
     * @return View
     */
    public function deleteUserApplicationAttributeAction(Request $request)
    {
        return $this->woM->userWs()->version1()->userApplicationAttribute()->delete()->init($request);
    }
}
