<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2018 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\RestBundle\Controller\UserWs\Version1;

use FOS\RestBundle\View\View;
use JMS\SecurityExtraBundle\Annotation\Secure;
use Nelmio\ApiDocBundle\Annotation\ApiDoc;
use LifeStyle\Tools\RestAuthBundle\Security\Authentication\Token\UidentifyToken;
use FOS\RestBundle\Controller\Annotations\NamePrefix;
use FOS\RestBundle\Controller\FOSRestController;
use Sso\RestBundle\Worker\Worker as Worker;
use JMS\DiExtraBundle\Annotation as DI;
use FOS\RestBundle\Controller\Annotations\Post;
use FOS\RestBundle\Controller\Annotations\Delete;
use Symfony\Component\HttpFoundation\Request;
use LifeStyle\Tools\CachingBundle\Annotations\LsCache;

/**
 * @NamePrefix("api_rest_")
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package    Sso\RestBundle\Controller
 */
class UserApplicationController extends FOSRestController
{
    /**
     * @DI\Inject("rest_worker_manager")
     * @var Worker
     */
    protected $woM;

    /**
     * @var UidentifyToken
     */
    protected $userToken;

    /**
     * *This service adds a new applications to the user. The service expects a JSON or XML in body and a header authentication.*
     *
     * #Request (header)
     *
     *      Key                 | Value               | Description
     *      ------------------- | ------------------- | --------------------------------------
     *      username            |  (string)           | your username
     *      useridentifier      |  (string)           | your useridentifier for given username
     *      servicetoken        |  (string)           | servicetoken from ws
     *
     * ##Request
     *
     * **Request example JSON**
     *
     *       {
     *           "users": [
     *               {
     *                   "identifier": "user@domain.tld",
     *                   "applicationType": {
     *                       "name": "testApplication",
     *                       "active": 1
     *                   }
     *               }
     *           ]
     *       }
     *
     * **Request example XML**
     *
     *       <userApplicationAdd>
     *           <users>
     *               <userType>
     *                   <identifier><![CDATA[user@domain.tld]]></identifier>
     *                   <applicationType>
     *                       <name><![CDATA[testApplication]]></name>
     *                       <active>1</active>
     *                   </applicationType>
     *               </userType>
     *               <userType>
     *                   <identifier><![CDATA[user@domain.tld]]></identifier>
     *                   <applicationType>
     *                       <name><![CDATA[AppName221]]></name>
     *                       <active>0</active>
     *                   </applicationType>
     *               </userType>
     *           </users>
     *       </userApplicationAdd>
     *
     * ##Response success
     *
     * **Example JSON (body)**
     *
     *       {
     *           "code": 200,
     *           "status": "OK",
     *           "trackId": "ab9f75d36a",
     *           "date": "2016-11-07T11:23:18+01:00",
     *           "scriptTimeSec": 7.7,
     *           "users": [
     *               {
     *                   "code": 200,
     *                   "status": "OK",
     *                   "message": "User added to application"
     *               }
     *           ]
     *       }
     *
     * **Example XML (body)**
     *
     *       <?xml version="1.0" encoding="UTF-8"?>
     *       <userApplicationAdd>
     *           <code>200</code>
     *           <status><![CDATA[OK]]></status>
     *           <trackId><![CDATA[df1ac675f2]]></trackId>
     *           <date><![CDATA[2016-11-07T11:23:57+01:00]]></date>
     *           <scriptTimeSec>3.1062</scriptTimeSec>
     *           <users>
     *               <userType>
     *                   <code>200</code>
     *                   <status><![CDATA[OK]]></status>
     *                   <message><![CDATA[User added to application]]></message>
     *               </userType>
     *           </users>
     *       </userApplicationAdd>
     *
     * ##Response error
     *
     * **Example JSON (body)**
     *
     *       {
     *           "code": 400,
     *           "status": "Bad Request",
     *           "trackId": "4924c71738",
     *           "date": "2016-11-07T10:34:30+01:00",
     *           "scriptTimeSec": 2.0151,
     *           "users": [
     *               {
     *                   "code": 400,
     *                   "status": "Bad Request",
     *                   "errors": [
     *                       {
     *                           "code": 400,
     *                           "message": "ErrorCode: u003 ErrorRef: ua013 ShortMessage: Invalid user application",
     *                           "exception": "external"
     *                       }
     *                   ]
     *               }
     *           ]
     *       }
     *
     * **Example XML (body)**
     *
     *       <?xml version="1.0" encoding="UTF-8"?>
     *       <userApplicationAdd>
     *           <code>400</code>
     *           <status><![CDATA[Bad Request]]></status>
     *           <trackId><![CDATA[9b7105467d]]></trackId>
     *           <date><![CDATA[2016-11-07T10:07:02+01:00]]></date>
     *           <scriptTimeSec>1.7327</scriptTimeSec>
     *           <users>
     *               <userType>
     *               <code>400</code>
     *               <status><![CDATA[Bad Request]]></status>
     *               <errors>
     *                   <error>
     *                       <code>400</code>
     *                       <message><![CDATA[ErrorCode: u003 ErrorRef: ua013 ShortMessage: Invalid user application]]></message>
     *                       <exception><![CDATA[external]]></exception>
     *                   </error>
     *               </errors>
     *               </userType>
     *           </users>
     *       </userApplicationAdd>
     *
     * ##Response error status 401, 416 or 500
     *
     * **Example JSON (body)**
     *
     *       {
     *       "code": 416,
     *       "status": "Not Found",
     *       "trackId": "2d008cc4c5",
     *       "scriptTimeSec": 1.521,
     *       "count": 1,
     *       "errors": [
     *          {
     *          "code": 416,
     *          "status": "Not Found",
     *          "message": "no valid json string",
     *          "type": "external",
     *          "exception": "InvalidRequest",
     *          "debug": "not set"
     *          }
     *         ]
     *       }
     *
     * **Example XML (body)**
     *
     *      <?xml version="1.0" encoding="UTF-8"?>
     *      <response>
     *          <code>416</code>
     *          <status><![CDATA[Not Found]]></status>
     *          <trackId><![CDATA[16fe523df4]]></trackId>
     *          <scriptTimeSec>2.4459</scriptTimeSec>
     *          <count>1</count>
     *          <errors>
     *              <error>
     *                <code>416</code>
     *                <status><![CDATA[Not Found]]></status>
     *                <message><![CDATA[expected '>']]></message>
     *                <type><![CDATA[external]]></type>
     *                <exception><![CDATA[InvalidRequest]]></exception>
     *                <debug><![CDATA[not set]]></debug>
     *              </error>
     *          </errors>
     *      </response>
     *
     * @ApiDoc(
     *  resource="/api/{version}/sso-ws/userApplication",
     *  description="secured: add userApplication to sso-ws",
     *  headers={
     *      {
     *        "name"="username",
     *        "description"="sso username",
     *        "required" = true
     *       },
     *      {
     *        "name"="useridentifier",
     *        "description"="sso useridentifier",
     *        "required" = true
     *       },
     *      {
     *        "name"="servicetoken",
     *        "description"="servicetoken",
     *        "required" = true
     *       }
     *   },
     *  statusCodes={
     *         200="Returned when successful",
     *         204="Returned when no content was send",
     *         400="Returned when an error occurred",
     *         401="Unauthorized - Wrong credentials",
     *         403="Returned when the user is not authorized to do this request",
     *         416="Returned when the request is malformed",
     *         500="Returned on internal server errors"
     *  },
     *  headers={
     *      {"name"="username", "description"="sso username"},
     *      {"name"="useridentifier", "description"="sso useridentifier"},
     *      {"name"="servicetoken", "description"="servicetoken"}
     *  },
     *  tags={
     *      "stable" = "#000"
     *  },
     *  output="Sso\RestBundle\ModelRest\Response\UserWs\Version1\UserApplication\Add\Response",
     *  views = { "sso-ws" }
     * )
     * @Post("/v1/sso-ws/userApplication/add", name="_sso-ws_v1", options={ "method_prefix" = true })
     * @param Request $request
     * @Secure(roles="ROLE_API")
     * @return View
     */
    public function userApplicationAddAction(Request $request)
    {
        return $this->woM->userWs()->version1()->userApplication()->add()->init($request);
    }

    /**
     * *This service deletes user application from sso-ws.*
     *
     * #Request (header)
     *
     *      Key                 | Value               | Description
     *      ------------------- | ------------------- | --------------------------------------
     *      username            |  (string)           | your username
     *      useridentifier      |  (string)           | your useridentifier for given username
     *      servicetoken        |  (string)           | your servicetoken
     *
     * ##Request
     *
     * **Request example JSON**
     *
     *          /api/v1/user/application/delete.json
     *
     * **Request example JSON (body)**
     *
     *          {
     *              "users": [
     *                  {
     *                      "identifier": "user@domain.tld",
     *                      "applicationType": {
     *                          "name": "testApplication"
     *                      }
     *                  },
     *                  {
     *                      "identifier": "user@domain.tld",
     *                      "applicationType": {
     *                          "name": "AppName221"
     *                      }
     *                  }
     *              ]
     *          }
     *
     * **Request example XML**
     *
     *          /api/v1/user/application/delete.xml
     *
     * **Request example XML (body)**
     *
     *          <userApplicationDelete>
     *              <users>
     *                  <userType>
     *                      <identifier><![CDATA[user@domain.tld]]></identifier>
     *                      <applicationType>
     *                          <name>testApplication</name>
     *                      </applicationType>
     *                  </userType>
     *                  <userType>
     *                      <identifier><![CDATA[testApplication]]></identifier>
     *                      <applicationType>
     *                          <name>AppName221</name>
     *                      </applicationType>
     *                  </userType>
     *              </users>
     *          </userApplicationDelete>
     *
     * ##Response success
     *
     * **Example (header)**
     *
     *          200 OK
     *
     * **Example JSON (body)**
     *
     *          {
     *              "code": 200,
     *              "status": "OK",
     *              "trackId": "9ad7b676af",
     *              "date": "2016-11-09T09:00:21+01:00",
     *              "scriptTimeSec": 2.2295,
     *              "users": [
     *                  {
     *                      "code": 200,
     *                      "status": "OK",
     *                      "message": "User deleted from application"
     *                  },
     *                  {
     *                      "code": 200,
     *                      "status": "OK",
     *                      "message": "User deleted from application"
     *                  }
     *              ]
     *          }
     *
     * **Example XML (body)**
     *
     *          <?xml version="1.0" encoding="UTF-8"?>
     *          <userApplicationDelete>
     *               <code>200</code>
     *               <status>
     *                  <![CDATA[OK]]>
     *               </status>
     *               <trackId>
     *                  <![CDATA[c1cfc80dbd]]>
     *               </trackId>
     *               <date>
     *                  <![CDATA[2016-11-09T08:57:19+01:00]]>
     *               </date>
     *               <scriptTimeSec>1.6522</scriptTimeSec>
     *               <users>
     *                  <userType>
     *                      <code>200</code>
     *                      <status>
     *                          <![CDATA[OK]]>
     *                      </status>
     *                      <message>
     *                          <![CDATA[User deleted from application]]>
     *                      </message>
     *                  </userType>
     *                  <userType>
     *                      <code>200</code>
     *                      <status>
     *                          <![CDATA[OK]]>
     *                      </status>
     *                      <message>
     *                          <![CDATA[User deleted from application]]>
     *                      </message>
     *                  </userType>
     *              </users>
     *          </userApplicationDelete>
     *
     * ##Response error
     *
     * **Example (header)**
     *
     *          400 Bad Request
     *
     * **Example JSON (body)**
     *
     *          {
     *              "code": 400,
     *              "status": "Bad Request",
     *              "trackId": "b4cea79f2a",
     *              "date": "2016-11-09T09:16:04+01:00",
     *              "scriptTimeSec": 2.1917,
     *              "users": [
     *                  {
     *                      "code": 400,
     *                      "status": "Bad Request",
     *                      "errors": [
     *                          {
     *                              "code": 400,
     *                              "message": "ErrorCode: u002 ErrorRef: ua033 ShortMessage: Invalid user application",
     *                              "exception": "external"
     *                          }
     *                      ]
     *                  },
     *                  {
     *                      "code": 400,
     *                      "status": "Bad Request",
     *                      "errors": [
     *                          {
     *                              "code": 400,
     *                              "message": "ErrorCode: u002 ErrorRef: ua033 ShortMessage: Invalid user application",
     *                              "exception": "external"
     *                          }
     *                      ]
     *                  }
     *              ]
     *          }
     *
     * **Example XML (body)**
     *
     *          <?xml version="1.0" encoding="UTF-8"?>
     *          <userApplicationDelete>
     *              <code>400</code>
     *              <status>
     *                  <![CDATA[Bad Request]]>
     *              </status>
     *              <trackId>
     *                  <![CDATA[914bbca641]]>
     *              </trackId>
     *              <date>
     *                  <![CDATA[2016-11-09T09:19:32+01:00]]>
     *              </date>
     *              <scriptTimeSec>2.1022</scriptTimeSec>
     *              <users>
     *                  <userType>
     *                      <code>400</code>
     *                      <status>
     *                          <![CDATA[Bad Request]]>
     *                      </status>
     *                      <errors>
     *                          <error>
     *                              <code>400</code>
     *                              <message>
     *                                  <![CDATA[ErrorCode: a001 ErrorRef: ua032 ShortMessage: Invalid application]]>
     *                              </message>
     *                              <exception>
     *                                  <![CDATA[external]]>
     *                              </exception>
     *                          </error>
     *                      </errors>
     *                  </userType>
     *                  <userType>
     *                      <code>400</code>
     *                      <status>
     *                          <![CDATA[Bad Request]]>
     *                      </status>
     *                      <errors>
     *                          <error>
     *                              <code>400</code>
     *                              <message>
     *                                  <![CDATA[ErrorCode: u002 ErrorRef: ua033 ShortMessage: Invalid user application]]>
     *                              </message>
     *                              <exception>
     *                                  <![CDATA[external]]>
     *                              </exception>
     *                          </error>
     *                      </errors>
     *                  </userType>
     *              </users>
     *          </userApplicationDelete>
     *
     * @ApiDoc(
     *  resource="/api/{version}/sso-ws/userApplication",
     *  description="secured: delete user application from sso-ws",
     *  statusCodes={
     *         200="Returned when successful",
     *         400="Returned when an error occured",
     *         500="Returned on internal server errors"
     *  },
     *  headers={
     *      {"name"="username", "description"="sso username", "required"="true"},
     *      {"name"="useridentifier", "description"="sso useridentifier", "required"="true"},
     *      {"name"="servicetoken", "description"="servicetoken", "required"="true"}
     *  },
     *  tags={
     *      "untested"
     *  },
     *  views={ "sso-ws" }
     * )
     *
     * @Delete("/v1/sso-ws/userApplication/delete", name="_sso-ws_v1", options={ "method_prefix" = true })
     * @Secure(roles="ROLE_API")
     * @param Request $request
     * @return View
     */
    public function deleteUserApplicationAction(Request $request)
    {
        return $this->woM->userWs()->version1()->userApplication()->delete()->init($request);
    }
}
