<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2018 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\RestBundle\Controller\UserWs\Version1;

use Sso\RestBundle\Worker\Worker as Worker;
use FOS\RestBundle\Controller\Annotations\Delete;
use FOS\RestBundle\Controller\Annotations\NamePrefix;
use FOS\RestBundle\Controller\Annotations\Post;
use FOS\RestBundle\View\View;
use JMS\DiExtraBundle\Annotation as DI;
use JMS\SecurityExtraBundle\Annotation\Secure;
use Nelmio\ApiDocBundle\Annotation\ApiDoc;
use Symfony\Component\HttpFoundation\Request;
use LifeStyle\Tools\CachingBundle\Annotations\LsCache;

/**
 * @NamePrefix("api_rest_")
 *
 * Class UserApplicationRoleController
 * @package Sso\RestBundle\Controller\UserWs\Version1
 */
class UserApplicationRoleController
{
    /**
     * @DI\Inject("rest_worker_manager")
     * @var Worker
     */
    protected $woM;

    /**
     * *This service deletes user application from sso-ws.*
     *
     * #Request (header)
     *
     *      Key                 | Value               | Description
     *      ------------------- | ------------------- | --------------------------------------
     *      username            |  (string)           | your username
     *      useridentifier      |  (string)           | your useridentifier for given username
     *      servicetoken        |  (string)           | your servicetoken
     *
     * ##Request
     *
     * **Request example JSON**
     *
     *          /api/v1/user/application/role/delete.json
     *
     * **Request example JSON (body)**
     *
     *          {
     *              "userApplicationRoles": [
     *                  {
     *                      "name": "testRole",
     *                      "userType": {
     *                          "identifier": "user@domain.tld"
     *                      },
     *                      "applicationType": {
     *                          "name": "testApplication"
     *                      }
     *                  }
     *              ]
     *          }
     *
     * **Request example XML**
     *
     *          /api/v1/user/application/role/delete.xml
     *
     * **Request example XML (body)**
     *
     *          <userApplicationRoleDelete>
     *              <userApplicationRoles>
     *                  <userApplicationRoleType>
     *                      <name><![CDATA[testRole]]></name>
     *                      <userType>
     *                          <identifier><![CDATA[user@domain.tld]]></identifier>
     *                      </userType>
     *                      <applicationType>
     *                          <name><![CDATA[testApplication]]></name>
     *                      </applicationType>
     *                  </userApplicationRoleType>
     *              </userApplicationRoles>
     *          </userApplicationRoleDelete>
     *
     * ##Response success
     *
     * **Example (header)**
     *
     *          200 OK
     *
     * **Example JSON (body)**
     *
     *          {
     *              "code": 200,
     *              "status": "OK",
     *              "trackId": "9ad7b676af",
     *              "date": "2016-11-09T09:00:21+01:00",
     *              "scriptTimeSec": 2.2295,
     *              "UserApplicationRoles": [
     *                  {
     *                      "code": 200,
     *                      "status": "OK",
     *                      "message": "Deleted user application role"
     *                  }
     *              ]
     *          }
     *
     * **Example XML (body)**
     *
     *          <?xml version="1.0" encoding="UTF-8"?>
     *          <userApplicationRoleDelete>
     *              <code>200</code>
     *              <status>
     *                  <![CDATA[OK]]>
     *              </status>
     *              <trackId>
     *                  <![CDATA[d4447cf8c4]]>
     *              </trackId>
     *              <date>
     *                  <![CDATA[2016-11-14T11:30:47+01:00]]>
     *              </date>
     *              <scriptTimeSec>0.9748</scriptTimeSec>
     *              <UserApplicationRoles>
     *                  <UserApplicationRoleType>
     *                      <code>200</code>
     *                      <status>
     *                          <![CDATA[OK]]>
     *                      </status>
     *                      <message>
     *                          <![CDATA[Deleted user application role]]>
     *                      </message>
     *                  </UserApplicationRoleType>
     *              </UserApplicationRoles>
     *          </userApplicationRoleDelete>>
     *
     * ##Response error
     *
     * **Example (header)**
     *
     *          400 Bad Request
     *
     * **Example JSON (body)**
     *
     *          {
     *              "code": 400,
     *              "status": "Bad Request",
     *              "trackId": "6b603ac6c8",
     *              "date": "2016-11-14T14:33:12+01:00",
     *              "scriptTimeSec": 1.2468,
     *              "UserApplicationRoles": [
     *                  {
     *                      "code": 400,
     *                      "status": "Bad Request",
     *                      "errors": [
     *                          {
     *                              "code": 400,
     *                              "message": "ErrorCode: u008 ErrorRef: uar044 ShortMessage: Invalid role",
     *                              "exception": "external"
     *                          }
     *                      ]
     *                  }
     *              ]
     *          }
     *
     * **Example XML (body)**
     *
     *          <?xml version="1.0" encoding="UTF-8"?>
     *          <userApplicationRoleDelete>
     *              <code>400</code>
     *              <status>
     *                  <![CDATA[Bad Request]]>
     *              </status>
     *              <trackId>
     *                  <![CDATA[db2136d911]]>
     *              </trackId>
     *              <date>
     *                  <![CDATA[2016-11-14T14:30:48+01:00]]>
     *              </date>
     *              <scriptTimeSec>1.3361</scriptTimeSec>
     *              <UserApplicationRoles>
     *                  <UserApplicationRoleType>
     *                      <code>400</code>
     *                      <status>
     *                          <![CDATA[Bad Request]]>
     *                      </status>
     *                      <errors>
     *                          <error>
     *                              <code>400</code>
     *                              <message>
     *                                  <![CDATA[ErrorCode: u008 ErrorRef: uar044 ShortMessage: Invalid role]]>
     *                              </message>
     *                              <exception>
     *                                  <![CDATA[external]]>
     *                              </exception>
     *                          </error>
     *                      </errors>
     *                  </UserApplicationRoleType>
     *              </UserApplicationRoles>
     *          </userApplicationRoleDelete>
     *
     * @ApiDoc(
     *  resource="/api/{version}/sso-ws/userApplicationRole",
     *  description="secured: delete user application from sso-ws",
     *  statusCodes={
     *         200="Returned when successful",
     *         400="Returned when an error occured",
     *         500="Returned on internal server errors"
     *  },
     *  headers={
     *      {"name"="username", "description"="sso username", "required"="true"},
     *      {"name"="useridentifier", "description"="sso useridentifier", "required"="true"},
     *      {"name"="servicetoken", "description"="servicetoken", "required"="true"}
     *  },
     *  tags={
     *      "untested"
     *  },
     *  views = { "sso-ws" }
     * )
     *
     * @Delete("/v1/sso-ws/userApplicationRole/delete", name="_sso-ws_v1", options={ "method_prefix" = true })
     *
     * @Secure(roles="ROLE_API")
     *
     * @param Request $request
     * @return View
     */
    public function deleteUserApplicationRoleAction(Request $request)
    {
        return $this->woM->userWs()->version1()->userApplicationRole()->delete()->init($request);
    }

    /**
     * *This service adds role to user application*
     *
     * #Request (header)
     *
     *      Key                 | Value               | Description
     *      ------------------- | ------------------- | --------------------------------------
     *      username            |  (string)           | your username
     *      useridentifier      |  (string)           | your useridentifier for given username
     *      servicetoken        |  (string)           | your servicetoken
     *
     * ##Request
     *
     * **Request example JSON**
     *
     *          /api/v1/user/application/role/add.json
     *
     * **Request example JSON (body)**
     *
     *          {
     *              "userRoles": [
     *                  {
     *                      "name": "testRole",
     *                      "active": "1",
     *                      "userType": {
     *                          "identifier": "test.identifier"
     *                      },
     *                      "applicationType": {
     *                          "name": "testApplication"
     *                      }
     *                  }
     *              ]
     *          }
     *
     * **Request example XML**
     *
     *          /api/v1/user/application/role/add.xml
     *
     * **Request example XML (body)**
     *
     *          <userRoleAdd>
     *              <userRoles>
     *                  <roleType>
     *                      <name>testRole</name>
     *                      <active>1</active>
     *                      <userType>
     *                          <identifier>testIdentifier</identifier>
     *                      </userType>
     *                      <applicationType>
     *                          <name>testApplication</name>
     *                      </applicationType>
     *                  </roleType>
     *              </userRoles>
     *          </userRoleAdd>
     *
     * ##Response success
     *
     * **Example (header)**
     *
     *          200 OK
     *
     * **Example JSON (body)**
     *
     *          {
     *              "code": 200,
     *              "status": "OK",
     *              "trackId": "c47d48dcf0",
     *              "date": "2016-11-24T12:09:31+01:00",
     *              "scriptTimeSec": 0.8316,
     *              "userRoles": [
     *                  {
     *                      "code": 200,
     *                      "status": "OK",
     *                      "message": "Roles added to users application"
     *                  }
     *              ]
     *          }
     *
     * **Example XML (body)**
     *
     *          <? xml version = "1.0" encoding = "UTF-8"?>
     *          <userApplicationRoleAdd>
     *              <code>200</code>
     *              <status>
     *                  <![CDATA[OK]]>
     *              </status>
     *              <trackId>
     *                  <![CDATA[fdaa096739]]>
     *              </trackId>
     *              <date>
     *                  <![CDATA[2016-11-24T12:04:05+01:00]]>
     *              </date>
     *              <scriptTimeSec>0.9641</scriptTimeSec>
     *              <userRoles>
     *                  <roleType>
     *                      <code>200</code>
     *                      <status>
     *                          <![CDATA[OK]]>
     *                      </status>
     *                      <message>
     *                          <![CDATA[Roles added to users application]]>
     *                      </message>
     *                  </roleType>
     *              </userRoles>
     *          </userApplicationRoleAdd>
     *
     * ##Response error
     *
     * **Example (header)**
     *
     *          400 Bad Request
     *
     * **Example JSON (body)**
     *
     *          {
     *              "code": 400,
     *              "status": "Bad Request",
     *              "trackId": "8e0340afbc",
     *              "date": "2016-11-24T12:18:33+01:00",
     *              "scriptTimeSec": 1.8654,
     *              "userRoles": [
     *                  {
     *                      "code": 400,
     *                      "status": "Bad Request",
     *                      "errors": [
     *                          {
     *                              "code": 400,
     *                              "message": "ErrorCode: a001 ErrorRef: uar022 ShortMessage: Invalid application",
     *                              "exception": "external"
     *                          }
     *                      ]
     *                  }
     *              ]
     *          }
     *
     * **Example XML (body)**
     *
     *          <?xml version="1.0" encoding="UTF-8"?>
     *          <userApplicationRoleAdd>
     *              <code>400</code>
     *              <status>
     *                  <![CDATA[Bad Request]]>
     *              </status>
     *              <trackId>
     *                  <![CDATA[552031071f]]>
     *              </trackId>
     *              <date>
     *                  <![CDATA[2016-11-24T12:07:12+01:00]]>
     *              </date>
     *              <scriptTimeSec>1.7136</scriptTimeSec>
     *              <userRoles>
     *                  <roleType>
     *                      <code>400</code>
     *                      <status>
     *                          <![CDATA[Bad Request]]>
     *                      </status>
     *                      <errors>
     *                          <error>
     *                              <code>400</code>
     *                              <message>
     *                                  <![CDATA[ErrorCode: u001 ErrorRef: uar021 ShortMessage: Invalid user]]>
     *                              </message>
     *                              <exception>
     *                                  <![CDATA[external]]>
     *                              </exception>
     *                          </error>
     *                      </errors>
     *                  </roleType>
     *              </userRoles>
     *          </userApplicationRoleAdd>
     *
     * @ApiDoc(
     *  resource="/api/{version}/sso-ws/userApplicationRole",
     *  description="secured: delete user application from sso-ws",
     *  statusCodes={
     *         200="Returned when successful",
     *         400="Returned when an error occured",
     *         500="Returned on internal server errors"
     *  },
     *  headers={
     *      {"name"="username", "description"="sso username", "required"="true"},
     *      {"name"="useridentifier", "description"="sso useridentifier", "required"="true"},
     *      {"name"="servicetoken", "description"="servicetoken", "required"="true"}
     *  },
     *  tags={
     *      "untested"
     *  },
     *  views = { "sso-ws" }
     * )
     *
     * @Post("/v1/sso-ws/userApplicationRole/add", name="_sso-ws_v1", options={ "method_prefix" = true })
     *
     * @Secure(roles="ROLE_API")
     *
     * @param Request $request
     * @return View
     */
    public function postUserApplicationRoleAction(Request $request)
    {
        return $this->woM->userWs()->version1()->userApplicationRole()->add()->init($request);
    }
}
