<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2018 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\RestBundle\Controller\UserWs\Version1;

use Sso\RestBundle\Worker\Worker as Worker;
use FOS\RestBundle\Controller\Annotations\NamePrefix;
use FOS\RestBundle\Controller\Annotations\Post;
use FOS\RestBundle\Controller\FOSRestController;
use FOS\RestBundle\View\View;
use JMS\DiExtraBundle\Annotation as DI;
use JMS\SecurityExtraBundle\Annotation\Secure;
use LifeStyle\Tools\RestAuthBundle\Security\Authentication\Token\UidentifyToken;
use Nelmio\ApiDocBundle\Annotation\ApiDoc;
use Symfony\Component\HttpFoundation\Request;
use LifeStyle\Tools\CachingBundle\Annotations\LsCache;

/**
 * @NamePrefix("api_rest_")
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package    Sso\RestBundle\Controller
 */
class UserServiceController extends FOSRestController
{
    /**
     * @DI\Inject("rest_worker_manager")
     * @var Worker
     */
    protected $woM;

    /**
     * @var UidentifyToken
     */
    protected $userToken;

    /**
     * *This service sends user activation request to sso-ws.*
     *
     * #Request (header)
     *
     *      Key                 | Value               | Description
     *      ------------------- | ------------------- | --------------------------------------
     *      username            |  (string)           | your username
     *      useridentifier      |  (string)           | your useridentifier for given username
     *      servicetoken        |  (string)           | your servicetoken
     *
     * ##Request
     *
     * **Request example JSON**
     *
     *          /api/v1/user/userService/activateRequest.json
     *
     * **Request example JSON (body)**
     *
     *          {
     *              "userActivateRequest": [
     *                  {
     *                      "identifier": "user@domain.tld"
     *                  }
     *              ]
     *          }
     *
     * **Request example XML**
     *
     *          /api/v1/user/userService/activateRequest.xml
     *
     * **Request example XML (body)**
     *
     *          <userServiceActivateRequest>
     *              <userActivateRequest>
     *                  <userType>
     *                      <identifier>user@domain.tld</identifier>
     *                  </userType>
     *              </userActivateRequest>
     *          </userServiceActivateRequest>
     *
     * ##Response success
     *
     * **Example (header)**
     *
     *          200 OK
     *
     * **Example JSON (body)**
     *
     *          {
     *              "code": 200,
     *              "status": "OK",
     *              "trackId": "2185802eee",
     *              "date": "2016-11-28T17:05:34+01:00",
     *              "scriptTimeSec": 5.4331,
     *              "userActivateRequest": [
     *                  {
     *                      "code": 200,
     *                      "status": "OK"
     *                  }
     *              ]
     *          }
     *
     * **Example XML (body)**
     *
     *          <?xml version="1.0" encoding="UTF-8"?>
     *          <userServiceActivateRequest>
     *              <code>200</code>
     *              <status>
     *                  <![CDATA[OK]]>
     *              </status>
     *              <trackId>
     *                  <![CDATA[23fe20de85]]>
     *              </trackId>
     *              <date>
     *                  <![CDATA[2016-11-28T17:43:16+01:00]]>
     *              </date>
     *              <scriptTimeSec>1.7979</scriptTimeSec>
     *              <userActivateRequest>
     *                  <userType>
     *                      <code>200</code>
     *                      <status>
     *                          <![CDATA[OK]]>
     *                      </status>
     *                  </userType>
     *              </userActivateRequest>
     *          </userServiceActivateRequest>
     *
     * ##Response error
     *
     * **Example (header)**
     *
     *          400 Bad Request
     *
     * **Example JSON (body)**
     *
     *          {
     *              "code": 400,
     *              "status": "Bad Request",
     *              "trackId": "643a092449",
     *              "date": "2016-11-28T17:40:53+01:00",
     *              "scriptTimeSec": 2.0954,
     *              "userActivateRequest": [
     *                  {
     *                      "code": 400,
     *                      "status": "Bad Request",
     *                      "errors": [
     *                          {
     *                              "code": 400,
     *                              "message": "ErrorCode: u001 ErrorRef: us031 ShortMessage: Invalid user",
     *                              "exception": "external"
     *                          }
     *                      ]
     *                  }
     *              ]
     *          }
     *
     * **Example XML (body)**
     *
     *          <?xml version="1.0" encoding="UTF-8"?>
     *          <userServiceActivateRequest>
     *              <code>400</code>
     *              <status>
     *                  <![CDATA[Bad Request]]>
     *              </status>
     *              <trackId>
     *                  <![CDATA[88973fcd0a]]>
     *              </trackId>
     *              <date>
     *                  <![CDATA[2016-11-28T17:42:06+01:00]]>
     *              </date>
     *              <scriptTimeSec>5.6352</scriptTimeSec>
     *              <userActivateRequest>
     *                  <userType>
     *                      <code>400</code>
     *                      <status>
     *                          <![CDATA[Bad Request]]>
     *                      </status>
     *                      <errors>
     *                          <error>
     *                              <code>400</code>
     *                              <message>
     *                                  <![CDATA[ErrorCode: u001 ErrorRef: us031 ShortMessage: Invalid user]]>
     *                              </message>
     *                              <exception>
     *                                  <![CDATA[external]]>
     *                              </exception>
     *                          </error>
     *                      </errors>
     *                  </userType>
     *              </userActivateRequest>
     *          </userServiceActivateRequest>
     *
     * @ApiDoc(
     *  resource="/api/{version}/sso-ws/userService",
     *  description="secured: send user activation request to sso-ws",
     *  statusCodes={
     *         200="Returned when successful",
     *         400="Returned when an error occured",
     *         500="Returned on internal server errors"
     *  },
     *  headers={
     *      {"name"="username", "description"="sso username", "required"="true"},
     *      {"name"="useridentifier", "description"="sso useridentifier", "required"="true"},
     *      {"name"="servicetoken", "description"="servicetoken", "required"="true"}
     *  },
     *  tags={
     *      "untested"
     *  },
     *  output="Sso\RestBundle\ModelRest\Request\UserWs\Version1\UserService\ActivateRequest\Request",
     *  views={ "sso-ws" }
     * )
     *
     * @Post("/v1/sso-ws/userService/activateRequest", name="_sso-ws_v1", options={ "method_prefix" = true })
     * @Secure(roles="ROLE_API")
     * @param Request $request
     * @return View
     */
    public function postUserServiceActivateRequestAction(Request $request)
    {
        return $this->woM->userWs()->version1()->userService()->activateRequest()->init($request);
    }

    /**
     * *This service confirms user activation in sso-ws.*
     *
     * #Request (header)
     *
     *      Key                 | Value               | Description
     *      ------------------- | ------------------- | --------------------------------------
     *      username            |  (string)           | your username
     *      useridentifier      |  (string)           | your useridentifier for given username
     *      servicetoken        |  (string)           | your servicetoken
     *
     * ##Request
     *
     * **Request example JSON**
     *
     *          /api/v1/user/userService/activateConfirm.json
     *
     * **Request example JSON (body)**
     *
     *          {
     *              "userActivateConfirm": [
     *                  {
     *                      "email": "user@domain.tld",
     *                      "token": "c8e252d472554db7f77cdf591c80f79cc9783bbcf6ae78c56e682c4965d258e5"
     *                  },
     *                  {
     *                      "email": "user2@domain.tld",
     *                      "token": "dae322d472554db7f12cdfa61c7cf79c5a683345f61778ccde682c4af5d258c7"
     *                  }
     *              ]
     *          }
     *
     * **Request example XML**
     *
     *          /api/v1/user/userService/activateConfirm.xml
     *
     * **Request example XML (body)**
     *
     *          <userServiceActivateConfirm>
     *              <userActivateConfirm>
     *                  <userType>
     *                      <email>user@domain.tld</identifier>
     *                      <token>c8e252d472554db7f77cdf591c80f79cc9783bbcf6ae78c56e682c4965d258e5</token>
     *                  </userType>
     *                  <userType>
     *                      <email>user2@domain.tld</identifier>
     *                      <token>dae322d472554db7f12cdfa61c7cf79c5a683345f61778ccde682c4af5d258c7</token>
     *                  </userType>
     *              </userActivateConfirm>
     *          </userServiceActivateConfirm>
     *
     * ##Response success
     *
     * **Example (header)**
     *
     *          200 OK
     *
     * **Example JSON (body)**
     *
     *          {
     *              "code": 200,
     *              "status": "OK",
     *              "trackId": "2185802eee",
     *              "date": "2016-11-28T17:05:34+01:00",
     *              "scriptTimeSec": 5.4331,
     *              "userActivateConfirm": [
     *                  {
     *                      "code": 200,
     *                      "status": "OK",
     *                      "message": "User activated"
     *                  }
     *              ]
     *          }
     *
     * **Example XML (body)**
     *
     *          <?xml version="1.0" encoding="UTF-8"?>
     *          <userServiceActivateConfirm>
     *              <code>200</code>
     *              <status>
     *                  <![CDATA[OK]]>
     *              </status>
     *              <trackId>
     *                  <![CDATA[7a5cdf5026]]>
     *              </trackId>
     *              <date>
     *                  <![CDATA[2016-11-29T14:17:52+01:00]]>
     *              </date>
     *              <scriptTimeSec>27.3867</scriptTimeSec>
     *              <userActivateConfirm>
     *                  <userType>
     *                      <code>200</code>
     *                      <status>
     *                          <![CDATA[OK]]>
     *                      </status>
     *                      <message>
     *                          <![CDATA[User activated]]>
     *                      </message>
     *                  </userType>
     *              </userActivateConfirm>
     *          </userServiceActivateConfirm>
     *
     * ##Response error
     *
     * **Example (header)**
     *
     *          400 Bad Request
     *
     * **Example JSON (body)**
     *
     *          {
     *              "code": 400,
     *              "status": "Bad Request",
     *              "trackId": "643a092449",
     *              "date": "2016-11-28T17:40:53+01:00",
     *              "scriptTimeSec": 2.0954,
     *              "userActivateConfirm": [
     *                  {
     *                      "code": 400,
     *                      "status": "Bad Request",
     *                      "errors": [
     *                          {
     *                              "code": 400,
     *                              "message": "ErrorCode: u010 ErrorRef: us041 ShortMessage: Invalid token",
     *                              "exception": "external"
     *                          }
     *                      ]
     *                  }
     *              ]
     *          }
     *
     * **Example XML (body)**
     *
     *          <?xml version="1.0" encoding="UTF-8"?>
     *          <userServiceActivateConfirm>
     *              <code>400</code>
     *              <status>
     *                  <![CDATA[Bad Request]]>
     *              </status>
     *              <trackId>
     *                  <![CDATA[7a5cdf5026]]>
     *              </trackId>
     *              <date>
     *                  <![CDATA[2016-11-29T14:17:52+01:00]]>
     *              </date>
     *              <scriptTimeSec>27.3867</scriptTimeSec>
     *              <userActivateConfirm>
     *                  <userType>
     *                      <code>400</code>
     *                      <status>
     *                          <![CDATA[Bad Request]]>
     *                      </status>
     *                      <errors>
     *                          <error>
     *                              <code>400</code>
     *                              <message>
     *                                  <![CDATA[ErrorCode: u010 ErrorRef: us041 ShortMessage: Invalid token]]>
     *                              </message>
     *                              <exception>
     *                                  <![CDATA[external]]>
     *                              </exception>
     *                          </error>
     *                      </errors>
     *                  </userType>
     *              </userActivateConfirm>
     *          </userServiceActivateConfirm>
     *
     * @ApiDoc(
     *  resource="/api/{version}/sso-ws/userService",
     *  description="secured: confirms user activation in sso-ws",
     *  statusCodes={
     *         200="Returned when successful",
     *         400="Returned when an error occured",
     *         500="Returned on internal server errors"
     *  },
     *  headers={
     *      {"name"="username", "description"="sso username", "required"="true"},
     *      {"name"="useridentifier", "description"="sso useridentifier", "required"="true"},
     *      {"name"="servicetoken", "description"="servicetoken", "required"="true"}
     *  },
     *  tags={
     *      "untested"
     *  },
     *  output="Sso\RestBundle\ModelRest\Request\UserWs\Version1\UserService\ActivateConfirm\Request",
     *  views={ "sso-ws" }
     * )
     *
     * @Post("/v1/sso-ws/userService/activateConfirm", name="_sso-ws_v1", options={ "method_prefix" = true })
     * @Secure(roles="ROLE_API")
     * @param Request $request
     * @return View
     */
    public function postUserServiceActivateConfirmAction(Request $request)
    {
        return $this->woM->userWs()->version1()->userService()->activateConfirm()->init($request);
    }

    /**
     * *This service sends a new password request. The service expects a JSON or XML in body and a header authentication.*
     *
     * #Request (header)
     *
     *      Key                 | Value               | Description
     *      ------------------- | ------------------- | --------------------------------------
     *      username            |  (string)           | your username
     *      useridentifier      |  (string)           | your useridentifier for given username
     *      servicetoken        |  (string)           | servicetoken from ws
     *
     * ##Request
     *
     * **Request example JSON**
     *
     *       {
     *           "userPasswordRequest":
     *              {
     *                  "userType": [
     *                      {
     *                          "identifier": "testuser"
     *                      }
     *                  ]
     *              }
     *       }
     *
     * **Request example XML**
     *
     *       <userServicePasswordRequest>
     *           <userPasswordRequest>
     *               <userType>
     *                   <identifier>testuser</identifier>
     *               </userType>
     *           </userPasswordRequest>
     *       </userServicePasswordRequest>
     *
     * ##Response success
     *
     * **Example JSON (body)**
     *
     *       {
     *           "code": 200,
     *           "status": "OK",
     *           "trackId": "ff18227f8d",
     *           "date": "2016-11-29T09:44:36+01:00",
     *           "scriptTimeSec": 2.2618,
     *           "userPasswordRequest": {
     *               "userType": [
     *                   {
     *                       "code": 200,
     *                       "status": "OK",
     *                       "username": "testuser",
     *                       "email": "user@domain.tld",
     *                       "passwordToken": "f60dbd33610371235ac07ab89a6b4075b19cc98095516784d8d315cc65bd505c",
     *                       "passwordTokenExpire": "2016-11-29T11:58:21+01:00"
     *                   }
     *               ]
     *           }
     *       }
     *
     * **Example XML (body)**
     *
     *       <?xml version="1.0" encoding="UTF-8"?>
     *       <userServicePasswordRequest>
     *           <code>200</code>
     *           <status><![CDATA[OK]]></status>
     *           <trackId><![CDATA[1a78648cf8]]></trackId>
     *           <date><![CDATA[2016-11-29T09:48:23+01:00]]></date>
     *           <scriptTimeSec>5.7664</scriptTimeSec>
     *           <userPasswordRequest>
     *               <userType>
     *                   <code>200</code>
     *                   <status><![CDATA[OK]]></status>
     *                   <username><![CDATA[testuser]]></username>
     *                   <email><![CDATA[user@domain.tld]]></email>
     *                   <passwordToken><![CDATA[236ca6f33d49306b190fc7bcd192937a02fd67532374365ead135e5d4efc7515]]></passwordToken>
     *                   <passwordTokenExpire><![CDATA[2016-11-29T11:58:21+01:00]]></passwordTokenExpire>
     *               </userType>
     *           </userPasswordRequest>
     *       </userServicePasswordRequest>
     *
     * ##Response error
     *
     * **Example JSON (body)**
     *
     *       {
     *           "code": 400,
     *           "status": "Bad Request",
     *           "trackId": "ff18227f8d",
     *           "date": "2016-11-29T09:44:36+01:00",
     *           "scriptTimeSec": 2.2618,
     *           "userPasswordRequest": {
     *               "userType": [
     *                   {
     *                       "code": 400,
     *                       "status": "Bad Request",
     *                       "errors": [
     *                           {
     *                               "code": 400,
     *                               "message": "ErrorCode: u001 ErrorRef: us011 ShortMessage: Invalid user",
     *                               "exception": "external"
     *                           }
     *                       ]
     *                   }
     *               ]
     *           }
     *       }
     *
     * **Example XML (body)**
     *
     *       <?xml version="1.0" encoding="UTF-8"?>
     *       <userServicePasswordRequest>
     *           <code>400</code>
     *           <status><![CDATA[Bad Request]]></status>
     *           <trackId><![CDATA[1a78648cf8]]></trackId>
     *           <date><![CDATA[2016-11-29T09:48:23+01:00]]></date>
     *           <scriptTimeSec>5.7664</scriptTimeSec>
     *           <userPasswordRequest>
     *               <userType>
     *                   <code>400</code>
     *                   <status><![CDATA[Bad Request]]></status>
     *                   <errors>
     *                       <error>
     *                           <code>400</code>
     *                           <message><![CDATA[ErrorCode: u001 ErrorRef: us011 ShortMessage: Invalid user]]></message>
     *                           <exception><![CDATA[external]]></exception>
     *                       </error>
     *                   </errors>
     *               </userType>
     *           </userPasswordRequest>
     *       </userServicePasswordRequest>
     *
     * ##Response error status 401, 416 or 500
     *
     * **Example JSON (body)**
     *
     *       {
     *       "code": 416,
     *       "status": "Not Found",
     *       "trackId": "2d008cc4c5",
     *       "scriptTimeSec": 1.521,
     *       "count": 1,
     *       "errors": [
     *          {
     *          "code": 416,
     *          "status": "Not Found",
     *          "message": "no valid json string",
     *          "type": "external",
     *          "exception": "InvalidRequest",
     *          "debug": "not set"
     *          }
     *         ]
     *       }
     *
     * **Example XML (body)**
     *
     *      <?xml version="1.0" encoding="UTF-8"?>
     *      <response>
     *          <code>416</code>
     *          <status><![CDATA[Not Found]]></status>
     *          <trackId><![CDATA[16fe523df4]]></trackId>
     *          <scriptTimeSec>2.4459</scriptTimeSec>
     *          <count>1</count>
     *          <errors>
     *              <error>
     *                <code>416</code>
     *                <status><![CDATA[Not Found]]></status>
     *                <message><![CDATA[expected '>']]></message>
     *                <type><![CDATA[external]]></type>
     *                <exception><![CDATA[InvalidRequest]]></exception>
     *                <debug><![CDATA[not set]]></debug>
     *              </error>
     *          </errors>
     *      </response>
     *
     * @ApiDoc(
     *  resource="/api/{version}/sso-ws/userService",
     *  description="secured: send password request",
     *  headers={
     *      {
     *        "name"="username",
     *        "description"="sso username",
     *        "required" = true
     *       },
     *      {
     *        "name"="useridentifier",
     *        "description"="sso useridentifier",
     *        "required" = true
     *       },
     *      {
     *        "name"="servicetoken",
     *        "description"="servicetoken",
     *        "required" = true
     *       }
     *   },
     *  statusCodes={
     *         200="Returned when successful",
     *         204="Returned when no content was send",
     *         400="Returned when an error occurred",
     *         401="Unauthorized - Wrong credentials",
     *         403="Returned when the user is not authorized to do this request",
     *         416="Returned when the request is malformed",
     *         500="Returned on internal server errors"
     *  },
     *  headers={
     *      {"name"="username", "description"="sso username"},
     *      {"name"="useridentifier", "description"="sso useridentifier"},
     *      {"name"="servicetoken", "description"="servicetoken"}
     *  },
     *  tags={
     *      "stable" = "#000"
     *  },
     *  output="Sso\RestBundle\ModelRest\Response\UserWs\Version1\UserService\PasswordRequest\Response",
     *  views = { "sso-ws" }
     * )
     * @Post("/v1/sso-ws/userService/passwordRequest", name="_sso-ws_v1", options={ "method_prefix" = true })
     * @param Request $request
     * @Secure(roles="ROLE_API")
     * @return View
     */
    public function postPasswordRequestAction(Request $request)
    {
        return $this->woM->userWs()->version1()->userService()->passwordRequest()->init($request);
    }

    /**
     * *This service confirms a new password. The service expects a JSON or XML in body and a header authentication.*
     *
     * #Request (header)
     *
     *      Key                 | Value               | Description
     *      ------------------- | ------------------- | --------------------------------------
     *      username            |  (string)           | your username
     *      useridentifier      |  (string)           | your useridentifier for given username
     *      servicetoken        |  (string)           | servicetoken from ws
     *
     * ##Request
     *
     * **Request example JSON**
     *
     *       {
     *           "userPasswordConfirm":
     *              {
     *                  "userType": [
     *                      {
     *                           "email": "user@domain.tld",
     *                           "token": "1e3a3daeb6497c528a62b31174e1731333e64563e60b67141e077ec7359045f0",
     *                           "password": "testPassword"
     *                      }
     *                  ]
     *              }
     *       }
     *
     * **Request example XML**
     *
     *       <userServicePasswordConfirm>
     *           <userPasswordConfirm>
     *               <userType>
     *                   <email>user@domain.tld</email>
     *                   <token>1e3a3daeb6497c528a62b31174e1731333e64563e60b67141e077ec7359045f0</token>
     *                   <password>testPassword</password>
     *               </userType>
     *           </userPasswordConfirm>
     *       </userServicePasswordConfirm>
     *
     * ##Response success
     *
     * **Example JSON (body)**
     *
     *       {
     *           "code": 200,
     *           "status": "OK",
     *           "trackId": "0a5f7dfdc8",
     *           "date": "2016-11-30T08:08:09+01:00",
     *           "scriptTimeSec": 1.4369,
     *           "userPasswordConfirm": {
     *               "userType": [
     *                   {
     *                       "code": 200,
     *                       "status": "OK"
     *                   }
     *               ]
     *           }
     *       }
     *
     * **Example XML (body)**
     *
     *       <?xml version="1.0" encoding="UTF-8"?>
     *       <userServicePasswordConfirm>
     *           <code>200</code>
     *           <status><![CDATA[OK]]></status>
     *           <trackId><![CDATA[0a5f7dfdc8]]></trackId>
     *           <date><![CDATA[2016-11-30T08:08:09+01:00]]></date>
     *           <scriptTimeSec>1.4369</scriptTimeSec>
     *           <userPasswordConfirm>
     *               <userType>
     *                   <code>200</code>
     *                   <status><![CDATA[OK]]></status>
     *               </userType>
     *           </userPasswordConfirm>
     *       </userServicePasswordConfirm>
     *
     * ##Response error
     *
     * **Example JSON (body)**
     *
     *       {
     *           "code": 400,
     *           "status": "Bad Request",
     *           "trackId": "0f0bca1c6c",
     *           "date": "2016-11-30T08:01:29+01:00",
     *           "scriptTimeSec": 1.5724,
     *           "userPasswordConfirm": {
     *               "userType": [
     *                   {
     *                       "code": 400,
     *                       "status": "Bad Request",
     *                       "errors": [
     *                           {
     *                               "code": 400,
     *                               "message": "ErrorCode: u31 ErrorRef: v001 ShortMessage: Invalid password",
     *                               "exception": "external"
     *                           }
     *                       ]
     *                   }
     *               ]
     *           }
     *       }
     *
     * **Example XML (body)**
     *
     *       <?xml version="1.0" encoding="UTF-8"?>
     *       <userServicePasswordRequest>
     *           <code>400</code>
     *           <status><![CDATA[Bad Request]]></status>
     *           <trackId><![CDATA[40de8e0a57]]></trackId>
     *           <date><![CDATA[2016-11-30T07:59:30+01:00]]></date>
     *           <scriptTimeSec>5.8291</scriptTimeSec>
     *           <userPasswordConfirm>
     *               <userType>
     *                   <code>400</code>
     *                   <status><![CDATA[Bad Request]]></status>
     *                   <errors>
     *                       <error>
     *                           <code>400</code>
     *                           <message><![CDATA[ErrorCode: u31 ErrorRef: v001 ShortMessage: Invalid password]]></message>
     *                           <exception><![CDATA[external]]></exception>
     *                       </error>
     *                   </errors>
     *               </userType>
     *           </userPasswordRequest>
     *       </userServicePasswordRequest>
     *
     * ##Response error status 401, 416 or 500
     *
     * **Example JSON (body)**
     *
     *       {
     *       "code": 416,
     *       "status": "Not Found",
     *       "trackId": "2d008cc4c5",
     *       "scriptTimeSec": 1.521,
     *       "count": 1,
     *       "errors": [
     *          {
     *          "code": 416,
     *          "status": "Not Found",
     *          "message": "no valid json string",
     *          "type": "external",
     *          "exception": "InvalidRequest",
     *          "debug": "not set"
     *          }
     *         ]
     *       }
     *
     * **Example XML (body)**
     *
     *      <?xml version="1.0" encoding="UTF-8"?>
     *      <response>
     *          <code>416</code>
     *          <status><![CDATA[Not Found]]></status>
     *          <trackId><![CDATA[16fe523df4]]></trackId>
     *          <scriptTimeSec>2.4459</scriptTimeSec>
     *          <count>1</count>
     *          <errors>
     *              <error>
     *                <code>416</code>
     *                <status><![CDATA[Not Found]]></status>
     *                <message><![CDATA[expected '>']]></message>
     *                <type><![CDATA[external]]></type>
     *                <exception><![CDATA[InvalidRequest]]></exception>
     *                <debug><![CDATA[not set]]></debug>
     *              </error>
     *          </errors>
     *      </response>
     *
     * @ApiDoc(
     *  resource="/api/{version}/sso-ws/userService",
     *  description="secured: send password confirmation",
     *  headers={
     *      {
     *        "name"="username",
     *        "description"="sso username",
     *        "required" = true
     *       },
     *      {
     *        "name"="useridentifier",
     *        "description"="sso useridentifier",
     *        "required" = true
     *       },
     *      {
     *        "name"="servicetoken",
     *        "description"="servicetoken",
     *        "required" = true
     *       }
     *   },
     *  statusCodes={
     *         200="Returned when successful",
     *         204="Returned when no content was send",
     *         400="Returned when an error occurred",
     *         401="Unauthorized - Wrong credentials",
     *         403="Returned when the user is not authorized to do this request",
     *         416="Returned when the request is malformed",
     *         500="Returned on internal server errors"
     *  },
     *  headers={
     *      {"name"="username", "description"="sso username"},
     *      {"name"="useridentifier", "description"="sso useridentifier"},
     *      {"name"="servicetoken", "description"="servicetoken"}
     *  },
     *  tags={
     *      "stable" = "#000"
     *  },
     *  output="Sso\RestBundle\ModelRest\Response\UserWs\Version1\UserService\PasswordConfirm\Response",
     *  views = { "sso-ws" }
     * )
     * @Post("/v1/sso-ws/userService/passwordConfirm", name="_sso-ws_v1", options={ "method_prefix" = true })
     * @param Request $request
     * @Secure(roles="ROLE_API")
     * @return View
     */
    public function postPasswordConfirmAction(Request $request)
    {
        return $this->woM->userWs()->version1()->userService()->passwordConfirm()->init($request);
    }
}
