<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2018 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\RestBundle\Controller\UserWs\Version2;

use FOS\RestBundle\Controller\Annotations\Get;
use FOS\RestBundle\View\View;
use JMS\SecurityExtraBundle\Annotation\Secure;
use Nelmio\ApiDocBundle\Annotation\ApiDoc;
use FOS\RestBundle\Controller\Annotations\NamePrefix;
use Symfony\Component\HttpFoundation\Request;
use FOS\RestBundle\Controller\FOSRestController;
use Sso\RestBundle\Worker\Worker as Worker;
use JMS\DiExtraBundle\Annotation as DI;
use LifeStyle\Tools\CachingBundle\Annotations\LsCache;

/**
 * @NamePrefix("api_rest_")
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package    Sso\RestBundle\Controller
 */
class UserApplicationAttributeController extends FOSRestController
{
    /**
     * @DI\Inject("rest_worker_manager")
     * @var Worker
     */
    protected $woM;

    /**
     *
     * *This service returns user application attributes informations. The service expects a JSON or XML in body and a header authentication.*
     *
     * #Request (header)
     *
     *      Key                 | Value               | Description
     *      ------------------- | ------------------- | --------------------------------------
     *      username            |  (string)           | your username
     *      useridentifier      |  (string)           | your useridentifier for given username
     *      servicetoken        |  (string)           | servicetoken from ws
     *
     * #Response success
     *
     * **Example JSON (body)**
     *
     *       {
     *           "code": 200,
     *           "status": "OK",
     *           "trackId": "6e6794d02d",
     *           "date": "2016-11-22T11:08:38+01:00",
     *           "scriptTimeSec": 1.203,
     *           "userApplicationAttributes": [
     *               {
     *                   "code": 200,
     *                   "status": "OK",
     *                   "guid": "c1f7e15f81cd4647f02ec0508781b32",
     *                   "username": "testUser",
     *                   "email": "",
     *                   "firstname": "test",
     *                   "lastname": "test",
     *                   "active": 0,
     *                   "deleted": 0,
     *                   "createdAt": "2016-09-06T14:06:45+02:00",
     *                   "updatedAt": "2016-09-06T14:06:45+02:00",
     *                   "authId": "",
     *                   "mfaEnabled": 0,
     *                   "applicationType": [
     *                       {
     *                           "name": "testApplication",
     *                           "active": 1,
     *                           "applicationAttribute": [
     *                               {
     *                                   "name": "SUSPENDED",
     *                                   "type": "one",
     *                                   "value": "0"
     *                               },
     *                               {
     *                                   "name": "FANTASTIC_FRUITS",
     *                                   "type": "many",
     *                                   "value": "Apple"
     *                               },
     *                               {
     *                                   "name": "FANTASTIC_FRUITS",
     *                                   "type": "many",
     *                                   "value": "Orange"
     *                               },
     *                               {
     *                                   "name": "MIDDLENAME",
     *                                   "type": "one",
     *                                   "value": "E."
     *                               }
     *                           ]
     *                       }
     *                   ]
     *               }
     *           ]
     *       }
     *
     * **Example XML (body)**
     *
     *       <?xml version="1.0" encoding="UTF-8"?>
     *       <userApplicationAttribute>
     *           <code>200</code>
     *           <status><![CDATA[OK]]></status>
     *           <trackId><![CDATA[b4197d09e1]]></trackId>
     *           <date><![CDATA[2016-11-22T11:09:03+01:00]]></date>
     *           <scriptTimeSec>1.0841</scriptTimeSec>
     *           <userApplicationAttributes>
     *               <userType>
     *                   <code>200</code>
     *                   <status><![CDATA[OK]]></status>
     *                   <guid><![CDATA[c1f7e15f81cd4647f02ec0508781b32]]></guid>
     *                   <username><![CDATA[testUser]]></username>
     *                   <email><![CDATA[]]></email>
     *                   <firstname><![CDATA[test]]></firstname>
     *                   <lastname><![CDATA[test]]></lastname>
     *                   <active>0</active>
     *                   <deleted>0</deleted>
     *                   <createdAt><![CDATA[2016-09-06T14:06:45+02:00]]></createdAt>
     *                   <updatedAt><![CDATA[2016-09-06T14:06:45+02:00]]></updatedAt>
     *                   <authId><![CDATA[]]></authId>
     *                   <mfaEnabled>0</mfaEnabled>
     *                   <applicationType>
     *                       <name><![CDATA[testApplication]]></name>
     *                       <active>1</active>
     *                       <applicationAttribute>
     *                           <name><![CDATA[SUSPENDED]]></name>
     *                           <type><![CDATA[one]]></type>
     *                           <value> <![CDATA[0]]></value>
     *                       </applicationAttribute>
     *                       <applicationAttribute>
     *                           <name><![CDATA[FANTASTIC_FRUITS]]></name>
     *                           <type><![CDATA[many]]></type>
     *                           <value> <![CDATA[Apple]]></value>
     *                       </applicationAttribute>
     *                       <applicationAttribute>
     *                           <name><![CDATA[FANTASTIC_FRUITS]]></name>
     *                           <type><![CDATA[many]]></type>
     *                           <value> <![CDATA[Orange]]></value>
     *                       </applicationAttribute>
     *                       <applicationAttribute>
     *                           <name><![CDATA[MIDDLENAME]]></name>
     *                           <type><![CDATA[one]]></type>
     *                           <value><![CDATA[E.]]></value>
     *                       </applicationAttribute>
     *                   </applicationType>
     *               </userType>
     *           </userApplicationAttributes>
     *       </userApplicationAttribute>
     *
     * #Response error
     *
     * **Example JSON (body)**
     *
     *       {
     *           "code": 400,
     *           "status": "Bad Request",
     *           "trackId": "2f0728187e",
     *           "date": "2016-11-22T11:22:34+01:00",
     *           "scriptTimeSec": 4.7498,
     *           "userApplicationAttributes": [
     *               {
     *                   "code": 400,
     *                   "status": "Bad Request",
     *                   "errors": [
     *                       {
     *                           "code": 400,
     *                           "message": "ErrorCode: u001 ErrorRef: u2015 ShortMessage: Invalid user",
     *                           "exception": "external"
     *                       }
     *                   ]
     *               }
     *           ]
     *       }
     *
     * **Example XML (body)**
     *
     *       <?xml version="1.0" encoding="UTF-8"?>
     *       <userApplicationAttribute>
     *           <code>400</code>
     *           <status><![CDATA[Bad Request]]></status>
     *           <trackId><![CDATA[1cf336ffcd]]></trackId>
     *           <date><![CDATA[2016-11-22T11:21:07+01:00]]></date>
     *           <scriptTimeSec>5.8278</scriptTimeSec>
     *           <userApplicationAttributes>
     *               <userType>
     *                   <code>404</code>
     *                   <status><![CDATA[Not Found]]></status>
     *                   <errors>
     *                       <error>
     *                           <code>400</code>
     *                           <message><![CDATA[ErrorCode: u001 ErrorRef: u2015 ShortMessage: Invalid user]]></message>
     *                           <exception><![CDATA[external]]></exception>
     *                       </error>
     *                   </errors>
     *               </userType>
     *           </userApplicationAttributes>
     *       </userApplicationAttribute>
     *
     * ##Response error status 401, 416 or 500
     *
     * **Example JSON (body)**
     *
     *       {
     *       "code": 416,
     *       "status": "Not Found",
     *       "trackId": "2d008cc4c5",
     *       "scriptTimeSec": 1.521,
     *       "count": 1,
     *       "errors": [
     *          {
     *          "code": 416,
     *          "status": "Not Found",
     *          "message": "no valid json string",
     *          "type": "external",
     *          "exception": "InvalidRequest",
     *          "debug": "not set"
     *          }
     *         ]
     *       }
     *
     * **Example XML (body)**
     *
     *      <?xml version="1.0" encoding="UTF-8"?>
     *      <response>
     *          <code>416</code>
     *          <status><![CDATA[Not Found]]></status>
     *          <trackId><![CDATA[16fe523df4]]></trackId>
     *          <scriptTimeSec>2.4459</scriptTimeSec>
     *          <count>1</count>
     *          <errors>
     *              <error>
     *                <code>416</code>
     *                <status><![CDATA[Not Found]]></status>
     *                <message><![CDATA[expected '>']]></message>
     *                <type><![CDATA[external]]></type>
     *                <exception><![CDATA[InvalidRequest]]></exception>
     *                <debug><![CDATA[not set]]></debug>
     *              </error>
     *          </errors>
     *      </response>
     *
     * @ApiDoc(
     *  resource="/api/{version}/sso-ws/userApplicationAttribute",
     *  description="secured: Gets user attributes informations from sso-ws user",
     *  statusCodes={
     *         200="Returned when successful",
     *         204="Returned when no content was send",
     *         400="Returned when an error occurred",
     *         401="Unauthorized - Wrong credentials",
     *         403="Returned when the user is not authorized to do this request",
     *         416="Returned when the request is malformed",
     *         500="Returned on internal server errors"
     *  },
     *  headers={
     *      {
     *        "name"="username",
     *        "description"="sso username",
     *        "required"=true
     *       },
     *      {
     *        "name"="useridentifier",
     *        "description"="sso useridentifier",
     *        "required"=true
     *       },
     *      {
     *        "name"="servicetoken",
     *        "description"="servicetoken",
     *        "required"=true
     *       }
     *  },
     *  parameters={
     *      {"name"="identifier", "dataType"="string", "required"=true, "description"="User identifier"},
     *      {"name"="applicationName", "dataType"="string", "required"=true, "description"="Application name"}
     *  },
     *  tags={
     *      "stable" = "#000"
     *  },
     *  output="Sso\RestBundle\ModelRest\Response\UserWs\Version2\UserApplicationAttribute\Show\Response",
     *  views = { "sso-ws" }
     * )
     *
     * @Get("/v2/sso-ws/userApplicationAttribute/show", name="_sso-ws_v2", options={ "method_prefix" = true })
     * @Secure(roles="ROLE_API")
     * @LsCache(cacheTTL=300, keyHeader=true)
     * @param Request $request
     * @return View
     */
    public function getUserApplicationAttributeAction(Request $request)
    {
        return $this->woM->userWs()->version2()->userApplicationAttribute()->show()->init($request);
    }
}
