<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2018 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\RestBundle\Controller\UserWs\Version2;

use FOS\RestBundle\View\View;
use JMS\SecurityExtraBundle\Annotation\Secure;
use Nelmio\ApiDocBundle\Annotation\ApiDoc;
use LifeStyle\Tools\RestAuthBundle\Security\Authentication\Token\UidentifyToken;
use FOS\RestBundle\Controller\Annotations\NamePrefix;
use FOS\RestBundle\Controller\FOSRestController;
use Sso\RestBundle\Worker\Worker as Worker;
use JMS\DiExtraBundle\Annotation as DI;
use FOS\RestBundle\Controller\Annotations\Post;
use FOS\RestBundle\Controller\Annotations\Get;
use FOS\RestBundle\Controller\Annotations\Put;
use Symfony\Component\HttpFoundation\Request;
use LifeStyle\Tools\CachingBundle\Annotations\LsCache;

/**
 * @NamePrefix("api_rest_")
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package    Sso\RestBundle\Controller
 */
class UserController extends FOSRestController
{
    /**
     * @DI\Inject("rest_worker_manager")
     * @var Worker
     */
    protected $woM;

    /**
     * *This service adds a new users to the sso-ws. The service expects a JSON or XML in body and a header authentication.*
     *
     * #Request (header)
     *
     *      Key                 | Value               | Description
     *      ------------------- | ------------------- | --------------------------------------
     *      username            |  (string)           | your username
     *      useridentifier      |  (string)           | your useridentifier for given username
     *      servicetoken        |  (string)           | servicetoken from ws
     *
     * ##Request
     *
     * **Request example JSON**
     *
     *       {
     *           "users" : [
     *               {
     *                   "email" : "user@domain.tld",
     *                   "username" : "testuser",
     *                   "firstname" : "Tester",
     *                   "lastname" : "Testuser",
     *                   "password" : "xxxxxx",
     *                   "active" : 1,
     *                   "authId" : "24446666688888889",
     *                   "ldapSearchAttribute" : "",
     *                   "ldapSearchValue" : "",
     *                   "mfaEnabled" : "1"
     *               }
     *           ]
     *       }
     *
     * **Request example XML**
     *
     *       <userAdd>
     *         <users>
     *           <userType>
     *             <email><![CDATA[user@domain.tld]]></email>
     *             <username><![CDATA[testuser]]></username>
     *             <firstname><![CDATA[Tester]]></firstname>
     *             <lastname><![CDATA[Testuser]]></lastname>
     *             <password><![CDATA[xxxxxx]]></password>
     *             <active>1</active>
     *             <authId><![CDATA[24446666688888889]]></authId>
     *             <ldapSearchAttribute><![CDATA[]]></ldapSearchAttribute>
     *             <ldapSearchValue><![CDATA[]]></ldapSearchValue>
     *             <mfaEnabled><![CDATA[1]]></mfaEnabled>
     *           </userType>
     *         </users>
     *       </userAdd>
     *
     * ##Response success
     *
     * **Example JSON (body)**
     *
     *       {
     *           "code": 200,
     *           "status": "OK",
     *           "trackId": "a2b2634b2a",
     *           "date": "2016-10-31T14:50:15+01:00",
     *           "scriptTimeSec": 0.5391,
     *           "users": [
     *               {
     *                   "code": 200,
     *                   "status": "OK",
     *                   "guid": "d9005e4517a3c36e19d905b273ae2aea",
     *                   "username": "testuser",
     *                   "email": "user@domain.tld",
     *                   "firstname": "Tester",
     *                   "lastname": "Testuser",
     *                   "active": 1,
     *                   "deleted": 0,
     *                   "createdAt": "2016-10-31T14:54:18+01:00",
     *                   "updatedAt": "2016-10-31T14:54:18+01:00",
     *                   "authId": "24446666688888889",
     *                   "mfaEnabled": 1
     *               }
     *           ]
     *       }
     *
     * **Example XML (body)**
     *
     *       <?xml version="1.0" encoding="UTF-8"?>
     *       <response>
     *           <code>200</code>
     *           <status><![CDATA[OK]]></status>
     *           <trackId><![CDATA[e6e5c380a4]]></trackId>
     *           <date><![CDATA[2016-10-31T14:53:10+01:00]]></date>
     *           <scriptTimeSec>0.5568</scriptTimeSec>
     *           <users>
     *               <userType>
     *                   <code>200</code>
     *                   <status><![CDATA[OK]]></status>
     *                   <guid><![CDATA[f8f3996f8d2482c7f08da88c7f5b27eb]]></guid>
     *                   <username><![CDATA[testuser]]></username>
     *                   <email><![CDATA[user@domain.tld]]></email>
     *                   <firstname><![CDATA[Tester]]></firstname>
     *                   <lastname><![CDATA[Testuser]]></lastname>
     *                   <active>1</active>
     *                   <deleted>0</deleted>
     *                   <createdAt><![CDATA[2016-10-31T14:53:10+01:00]]></createdAt>
     *                   <updatedAt><![CDATA[2016-10-31T14:53:10+01:00]]></updatedAt>
     *                   <authId><![CDATA[24446666688888889]]></authId>
     *                   <mfaEnabled>1</mfaEnabled>
     *               </userType>
     *           </users>
     *       </response>
     *
     * ##Response error
     *
     * **Example JSON (body)**
     *
     *       {
     *           "code": 400,
     *           "status": "Bad Request",
     *           "trackId": "82d2846f7f",
     *           "date": "2016-10-31T14:54:18+01:00",
     *           "scriptTimeSec": 2.4736,
     *           "users": [
     *               {
     *                   "code": 400,
     *                   "status": "Bad Request",
     *                   "errors": [
     *                       {
     *                       "code": 400,
     *                       "message": "ErrorCode: UserAlreadyExists ErrorRef: u011 ShortMessage: Usernmae",
     *                       "exception": "external"
     *                       }
     *                   ]
     *               }
     *           ]
     *       }
     *
     * **Example XML (body)**
     *
     *       <?xml version="1.0" encoding="UTF-8"?>
     *       <response>
     *           <code>400</code>
     *           <status><![CDATA[Bad Request]]></status>
     *           <trackId><![CDATA[cf2fc5cf8d]]></trackId>
     *           <date><![CDATA[2016-10-31T14:55:28+01:00]]></date>
     *           <scriptTimeSec>2.255</scriptTimeSec>
     *           <users>
     *               <userType>
     *                   <code>400</code>
     *                   <status><![CDATA[Bad Request]]></status>
     *                   <errors>
     *                       <error>
     *                           <code>400</code>
     *                           <message><![CDATA[ErrorCode: UserAlreadyExists ErrorRef: u011 ShortMessage: Usernmae]]></message>
     *                           <exception><![CDATA[external]]></exception>
     *                       </error>
     *                   </errors>
     *               </userType>
     *           </users>
     *       </response>
     *
     * ##Response error status 401, 416 or 500
     *
     * **Example JSON (body)**
     *
     *       {
     *       "code": 416,
     *       "status": "Not Found",
     *       "trackId": "2d008cc4c5",
     *       "scriptTimeSec": 1.521,
     *       "count": 1,
     *       "errors": [
     *          {
     *          "code": 416,
     *          "status": "Not Found",
     *          "message": "no valid json string",
     *          "type": "external",
     *          "exception": "InvalidRequest",
     *          "debug": "not set"
     *          }
     *         ]
     *       }
     *
     * **Example XML (body)**
     *
     *      <?xml version="1.0" encoding="UTF-8"?>
     *      <response>
     *          <code>416</code>
     *          <status><![CDATA[Not Found]]></status>
     *          <trackId><![CDATA[16fe523df4]]></trackId>
     *          <scriptTimeSec>2.4459</scriptTimeSec>
     *          <count>1</count>
     *          <errors>
     *              <error>
     *                <code>416</code>
     *                <status><![CDATA[Not Found]]></status>
     *                <message><![CDATA[expected '>']]></message>
     *                <type><![CDATA[external]]></type>
     *                <exception><![CDATA[InvalidRequest]]></exception>
     *                <debug><![CDATA[not set]]></debug>
     *              </error>
     *          </errors>
     *      </response>
     *
     * @ApiDoc(
     *  resource="/api/{version}/sso-ws/user",
     *  description="secured: Adds an user to a sso-ws",
     *  headers={
     *      {
     *        "name"="username",
     *        "description"="sso username",
     *        "required"=true
     *       },
     *      {
     *        "name"="useridentifier",
     *        "description"="sso useridentifier",
     *        "required"=true
     *       },
     *      {
     *        "name"="servicetoken",
     *        "description"="servicetoken",
     *        "required"=true
     *       }
     *   },
     *  statusCodes={
     *         200="Returned when successful",
     *         204="Returned when no content was send",
     *         400="Returned when an error occurred",
     *         401="Unauthorized - Wrong credentials",
     *         403="Returned when the user is not authorized to do this request",
     *         416="Returned when the request is malformed",
     *         500="Returned on internal server errors"
     *  },
     *  tags={
     *      "stable" = "#000"
     *  },
     *  views = { "sso-ws" }
     * )
     * @Post("/v2/sso-ws/user/add", name="_sso-ws_v2", options={ "method_prefix" = true })
     * @param Request $request
     * @Secure(roles="ROLE_API")
     * @return View
     */
    public function userAddAction(Request $request)
    {
        return $this->woM->userWs()->version2()->user()->add()->init($request);
    }

    /**
     * *This service gets a user from the user ws. The services requires a header authentication!*
     *
     * #Request (header)
     *
     *      Key                 | Value               | Description
     *      ------------------- | ------------------- | --------------------------------------
     *      username            |  (string)           | your username
     *      useridentifier      |  (string)           | your useridentifier for given username
     *      servicetoken        |  (string)           | servicetoken from ws
     *
     * ##Request
     *
     * **Request example JSON**
     *
     *      GET /api/v2/sso-ws/user/show.json?identifier[0]=UserToCheck1&identifier[1]=UserToCheck2
     *
     * **Request example XML**
     *
     *      GET /api/v2/sso-ws/user/show.xml?identifier[0]=UserToCheck1&identifier[1]=UserToCheck2
     *
     * ##Response success
     *
     * **Example JSON (body)**
     *
     *      {
     *          "code" : 200,
     *          "status" : "OK",
     *          "trackId" : "b7e6d36424",
     *          "date" : "2016-12-12T12:17:10+01:00",
     *          "scriptTimeSec" : 4.1672,
     *          "users" : [
     *              {
     *               "code": 200,
     *               "status": "OK",
     *               "guid": "sdfsdafsdfsdfsdfsdfsdfsdfdsfsd",
     *               "username": "sb@life-style.de",
     *               "email": "sb@life-style.de",
     *               "firstname": "John",
     *               "lastname": "Doe",
     *               "active": "1",
     *               "deleted": "0",
     *               "createdAt": "2016-09-06T14:06:45+02:00",
     *               "lastLoginAt": "2016-12-12T09:38:01+01:00",
     *               "updatedAt": "2016-12-02T12:16:17+01:00",
     *               "authId": "",
     *               "ldapSearchAttributes": "",
     *               "ldapSearchValue": "",
     *               "mfaEnabled": "0"
     *              }
     *          ]
     *      }
     *
     * **Example XML (body)**
     *
     *      <?xml version="1.0" encoding="UTF-8"?>
     *      <userShow>
     *          <code>200</code>
     *          <status><![CDATA[OK]]></status>
     *          <trackId><![CDATA[bc4a4b60e1]]></trackId>
     *          <date><![CDATA[2016-12-12T12:18:27+01:00]]></date>
     *          <scriptTimeSec>3.5844</scriptTimeSec>
     *          <users>
     *              <userType>
     *                  <code>200</code>
     *                  <status><![CDATA[OK]]></status>
     *                  <guid><![CDATA[7sdfsdfsdfsdfsdfsdfsdfsdfsdf]]></guid>
     *                  <username><![CDATA[sb@life-style.de]]></username>
     *                  <email><![CDATA[sb@life-style.de]]></email>
     *                  <firstname><![CDATA[John]]></firstname>
     *                  <lastname><![CDATA[Doe]]></lastname>
     *                  <active><![CDATA[1]]></active>
     *                  <deleted><![CDATA[0]]></deleted>
     *                  <createdAt><![CDATA[2016-09-06T14:06:45+02:00]]></createdAt>
     *                  <lastLoginAt><![CDATA[2016-12-12T09:38:01+01:00]]></lastLoginAt>
     *                  <updatedAt><![CDATA[2016-12-02T12:16:17+01:00]]></updatedAt>
     *                  <authId><![CDATA[]]></authId>
     *                  <ldapSearchAttributes><![CDATA[]]></ldapSearchAttributes>
     *                  <ldapSearchValue><![CDATA[]]></ldapSearchValue>
     *                  <mfaEnabled><![CDATA[0]]></mfaEnabled>
     *              </userType>
     *          </users>
     *      </userShow>
     *
     * ##Response error status 400
     *
     * **Example JSON (body)**
     *
     *       {
     *           "code": 400,
     *           "status": "Bad Request",
     *           "trackId": "3113f4788d",
     *           "date": "2016-10-31T14:38:47+01:00",
     *           "scriptTimeSec": 1.6056,
     *           "users": [
     *               {
     *                   "code": 404,
     *                   "status": "Bad Request",
     *                   "errors": [
     *                       {
     *                       "code": 404,
     *                       "message": "ErrorCode: u001 ErrorRef: u011 ShortMessage: Invalid user",
     *                       "exception": "external"
     *                       }
     *                   ]
     *               }
     *           ]
     *       }
     *
     * **Example XML (body)**
     *
     *       <?xml version="1.0" encoding="UTF-8"?>
     *       <userShow>
     *           <code>400</code>
     *           <status><![CDATA[Bad Request]]></status>
     *           <trackId><![CDATA[dd454a1d10]]></trackId>
     *           <date><![CDATA[2016-10-31T14:40:30+01:00]]></date>
     *           <scriptTimeSec>1.9725</scriptTimeSec>
     *           <users>
     *               <userType>
     *                   <code>404</code>
     *                   <status><![CDATA[Bad Request]]></status>
     *                   <errors>
     *                       <error>
     *                           <code>404</code>
     *                           <message><![CDATA[ErrorCode: u001 ErrorRef: u011 ShortMessage: Invalid user]]></message>
     *                           <exception><![CDATA[external]]></exception>
     *                       </error>
     *                   </errors>
     *               </userType>
     *           </users>
     *       </userShow>
     *
     * @ApiDoc(
     *  resource="/api/{version}/sso-ws/user",
     *  description="secured: get user scope from sso-ws",
     *  statusCodes={
     *         200="Returned when successful",
     *         400="Returned when an error occured",
     *         500="Returned on internal server errors"
     *  },
     *  headers={
     *      {"name"="username", "description"="sso username"},
     *      {"name"="useridentifier", "description"="sso useridentifier"},
     *      {"name"="servicetoken", "description"="servicetoken"}
     *  },
     *  tags={
     *      "stable" = "#000"
     *  },
     *  input="Sso\RestBundle\ModelRest\Request\UserWs\Version2\User\Show\Request",
     *  views = { "sso-ws" }
     * )
     *
     * @Get("/v2/sso-ws/user/show", name="_sso-ws_v2", options={ "method_prefix" = true })
     * @Secure(roles="ROLE_API")
     * @LsCache(cacheTTL=300, keyHeader=true)
     * @param Request $request
     * @return View
     */
    public function showUserAction(Request $request)
    {
        return $this->woM->userWs()->version2()->user()->show()->init($request);
    }

    /**
     * *This service updates an users in the sso-ws. The service expects a JSON or XML in body and a header authentication.*
     *
     * #Request (header)
     *
     *      Key                 | Value               | Description
     *      ------------------- | ------------------- | --------------------------------------
     *      username            |  (string)           | your username
     *      useridentifier      |  (string)           | your useridentifier for given username
     *      servicetoken        |  (string)           | servicetoken from ws
     *
     * ##Request
     *
     * **Request example JSON**
     *
     *       {
     *           "users" : [
     *               {
     *                   "key": {
     *                       "identifier": "user@domain.tld"
     *                   }
     *                   "email" : "user-updated@domain.tld",
     *                   "username" : "testuser-updated",
     *                   "firstname" : "Tester-Updated",
     *                   "lastname" : "Testuser-Updated",
     *                   "active" : 1,
     *                   "authId" : "24446666688888889",
     *                   "ldapSearchAttribute" : "",
     *                   "ldapSearchValue" : "",
     *                   "mfaEnabled" : "1",
     *                   "mfaRecreate" : "1"
     *               }
     *           ]
     *       }
     *
     * **Request example XML**
     *
     *       <userUpdate>
     *         <users>
     *           <userType>
     *            <key>
     *              <identifier>user@domain.tld<identifier>
     *            </key>
     *             <email><![CDATA[user@domain.tld]]></email>
     *             <username><![CDATA[testuser]]></username>
     *             <firstname><![CDATA[Tester]]></firstname>
     *             <lastname><![CDATA[Testuser]]></lastname>
     *             <active>1</active>
     *             <authId><![CDATA[24446666688888889]]></authId>
     *             <ldapSearchAttribute><![CDATA[]]></ldapSearchAttribute>
     *             <ldapSearchValue><![CDATA[]]></ldapSearchValue>
     *             <mfaEnabled><![CDATA[1]]></mfaEnabled>
     *             <mfaRecreate><![CDATA[1]]></mfaRecreate>
     *           </userType>
     *         </users>
     *       </userUpdate>
     *
     * ##Response success
     *
     * **Example JSON (body)**
     *
     *       {
     *           "code": 200,
     *           "status": "OK",
     *           "trackId": "a2b2634b2a",
     *           "date": "2016-10-31T14:50:15+01:00",
     *           "scriptTimeSec": 0.5391,
     *           "users": [
     *               {
     *                   "code": 200,
     *                   "status": "OK",
     *                   "guid": "d9005e4517a3c36e19d905b273ae2aea",
     *                   "username": "testuser",
     *                   "email": "user@domain.tld",
     *                   "firstname": "Tester",
     *                   "lastname": "Testuser",
     *                   "active": 1,
     *                   "deleted": 0,
     *                   "createdAt": "2016-10-31T14:54:18+01:00",
     *                   "updatedAt": "2016-10-31T14:54:18+01:00",
     *                   "authId": "24446666688888889",
     *                   "mfaEnabled": 1
     *               }
     *           ]
     *       }
     *
     * **Example XML (body)**
     *
     *       <?xml version="1.0" encoding="UTF-8"?>
     *       <response>
     *           <code>200</code>
     *           <status><![CDATA[OK]]></status>
     *           <trackId><![CDATA[e6e5c380a4]]></trackId>
     *           <date><![CDATA[2016-10-31T14:53:10+01:00]]></date>
     *           <scriptTimeSec>0.5568</scriptTimeSec>
     *           <users>
     *               <userType>
     *                   <code>200</code>
     *                   <status><![CDATA[OK]]></status>
     *                   <guid><![CDATA[f8f3996f8d2482c7f08da88c7f5b27eb]]></guid>
     *                   <username><![CDATA[testuser]]></username>
     *                   <email><![CDATA[user@domain.tld]]></email>
     *                   <firstname><![CDATA[Tester]]></firstname>
     *                   <lastname><![CDATA[Testuser]]></lastname>
     *                   <active>1</active>
     *                   <deleted>0</deleted>
     *                   <createdAt><![CDATA[2016-10-31T14:53:10+01:00]]></createdAt>
     *                   <updatedAt><![CDATA[2016-10-31T14:53:10+01:00]]></updatedAt>
     *                   <authId><![CDATA[24446666688888889]]></authId>
     *                   <mfaEnabled>1</mfaEnabled>
     *               </userType>
     *           </users>
     *       </response>
     *
     * ##Response error
     *
     * **Example JSON (body)**
     *
     *       {
     *           "code": 400,
     *           "status": "Bad Request",
     *           "trackId": "82d2846f7f",
     *           "date": "2016-10-31T14:54:18+01:00",
     *           "scriptTimeSec": 2.4736,
     *           "users": [
     *               {
     *                   "code": 400,
     *                   "status": "Bad Request",
     *                   "errors": [
     *                       {
     *                       "code": 400,
     *                       "message": "ErrorCode: UserAlreadyExists ErrorRef: u011 ShortMessage: Usernmae",
     *                       "exception": "external"
     *                       }
     *                   ]
     *               }
     *           ]
     *       }
     *
     * **Example XML (body)**
     *
     *       <?xml version="1.0" encoding="UTF-8"?>
     *       <response>
     *           <code>400</code>
     *           <status><![CDATA[Bad Request]]></status>
     *           <trackId><![CDATA[cf2fc5cf8d]]></trackId>
     *           <date><![CDATA[2016-10-31T14:55:28+01:00]]></date>
     *           <scriptTimeSec>2.255</scriptTimeSec>
     *           <users>
     *               <userType>
     *                   <code>400</code>
     *                   <status><![CDATA[Bad Request]]></status>
     *                   <errors>
     *                       <error>
     *                           <code>400</code>
     *                           <message><![CDATA[ErrorCode: UserAlreadyExists ErrorRef: u011 ShortMessage: Usernmae]]></message>
     *                           <exception><![CDATA[external]]></exception>
     *                       </error>
     *                   </errors>
     *               </userType>
     *           </users>
     *       </response>
     *
     * ##Response error status 401, 416 or 500
     *
     * **Example JSON (body)**
     *
     *       {
     *       "code": 416,
     *       "status": "Not Found",
     *       "trackId": "2d008cc4c5",
     *       "scriptTimeSec": 1.521,
     *       "count": 1,
     *       "errors": [
     *          {
     *          "code": 416,
     *          "status": "Not Found",
     *          "message": "no valid json string",
     *          "type": "external",
     *          "exception": "InvalidRequest",
     *          "debug": "not set"
     *          }
     *         ]
     *       }
     *
     * **Example XML (body)**
     *
     *      <?xml version="1.0" encoding="UTF-8"?>
     *      <response>
     *          <code>416</code>
     *          <status><![CDATA[Not Found]]></status>
     *          <trackId><![CDATA[16fe523df4]]></trackId>
     *          <scriptTimeSec>2.4459</scriptTimeSec>
     *          <count>1</count>
     *          <errors>
     *              <error>
     *                <code>416</code>
     *                <status><![CDATA[Not Found]]></status>
     *                <message><![CDATA[expected '>']]></message>
     *                <type><![CDATA[external]]></type>
     *                <exception><![CDATA[InvalidRequest]]></exception>
     *                <debug><![CDATA[not set]]></debug>
     *              </error>
     *          </errors>
     *      </response>
     *
     * @ApiDoc(
     *  resource="/api/{version}/sso-ws/user",
     *  description="secured: Updates an user in the sso-ws",
     *  headers={
     *      {
     *        "name"="username",
     *        "description"="sso username",
     *        "required"=true
     *       },
     *      {
     *        "name"="useridentifier",
     *        "description"="sso useridentifier",
     *        "required"=true
     *       },
     *      {
     *        "name"="servicetoken",
     *        "description"="servicetoken",
     *        "required"=true
     *       }
     *   },
     *  statusCodes={
     *         200="Returned when successful",
     *         204="Returned when no content was send",
     *         400="Returned when an error occurred",
     *         401="Unauthorized - Wrong credentials",
     *         403="Returned when the user is not authorized to do this request",
     *         416="Returned when the request is malformed",
     *         500="Returned on internal server errors"
     *  },
     *  tags={
     *      "stable" = "#000"
     *  },
     *  views = { "sso-ws" }
     * )
     * @Put("/v2/sso-ws/user/update", name="_sso-ws_v2", options={ "method_prefix" = true })
     * @param Request $request
     * @Secure(roles="ROLE_API")
     * @return View
     */
    public function userUpdateAction(Request $request)
    {
        return $this->woM->userWs()->version2()->user()->update()->init($request);
    }

    /**
     * *This service updates an users in the sso-ws. The service expects a JSON or XML in body and a header authentication.*
     *
     * #Request (header)
     *
     *      Key                 | Value               | Description
     *      ------------------- | ------------------- | --------------------------------------
     *      username            |  (string)           | your username
     *      useridentifier      |  (string)           | your useridentifier for given username
     *      servicetoken        |  (string)           | servicetoken from ws
     *
     * ##Request
     *
     * **Request example JSON**
     *
     *       {
     *           "users" : [
     *               {
     *                   "email" : "user-updated@domain.tld",
     *                   "username" : "testuser-updated",
     *                   "firstname" : "Tester-Updated",
     *                   "lastname" : "Testuser-Updated",
     *                   "active" : 1,
     *                   "authId" : "24446666688888889",
     *                   "ldapSearchAttribute" : "",
     *                   "ldapSearchValue" : "",
     *                   "mfaEnabled" : "1",
     *                   "mfaRecreate" : "1"
     *               }
     *           ]
     *       }
     *
     * **Request example XML**
     *
     *       <userUpdate>
     *         <users>
     *           <userType>
     *             <email><![CDATA[user@domain.tld]]></email>
     *             <username><![CDATA[testuser]]></username>
     *             <firstname><![CDATA[Tester]]></firstname>
     *             <lastname><![CDATA[Testuser]]></lastname>
     *             <active>1</active>
     *             <authId><![CDATA[24446666688888889]]></authId>
     *             <ldapSearchAttribute><![CDATA[]]></ldapSearchAttribute>
     *             <ldapSearchValue><![CDATA[]]></ldapSearchValue>
     *             <mfaEnabled><![CDATA[1]]></mfaEnabled>
     *             <mfaRecreate><![CDATA[1]]></mfaRecreate>
     *           </userType>
     *         </users>
     *       </userUpdate>
     *
     * ##Response success
     *
     * **Example JSON (body)**
     *
     *       {
     *           "code": 200,
     *           "status": "OK",
     *           "trackId": "a2b2634b2a",
     *           "date": "2016-10-31T14:50:15+01:00",
     *           "scriptTimeSec": 0.5391,
     *           "users": [
     *               {
     *                   "code": 200,
     *                   "status": "OK",
     *                   "guid": "d9005e4517a3c36e19d905b273ae2aea",
     *                   "username": "testuser",
     *                   "email": "user@domain.tld",
     *                   "firstname": "Tester",
     *                   "lastname": "Testuser",
     *                   "active": 1,
     *                   "deleted": 0,
     *                   "createdAt": "2016-10-31T14:54:18+01:00",
     *                   "updatedAt": "2016-10-31T14:54:18+01:00",
     *                   "authId": "24446666688888889",
     *                   "mfaEnabled": 1
     *               }
     *           ]
     *       }
     *
     * **Example XML (body)**
     *
     *       <?xml version="1.0" encoding="UTF-8"?>
     *       <response>
     *           <code>200</code>
     *           <status><![CDATA[OK]]></status>
     *           <trackId><![CDATA[e6e5c380a4]]></trackId>
     *           <date><![CDATA[2016-10-31T14:53:10+01:00]]></date>
     *           <scriptTimeSec>0.5568</scriptTimeSec>
     *           <users>
     *               <userType>
     *                   <code>200</code>
     *                   <status><![CDATA[OK]]></status>
     *                   <guid><![CDATA[f8f3996f8d2482c7f08da88c7f5b27eb]]></guid>
     *                   <username><![CDATA[testuser]]></username>
     *                   <email><![CDATA[user@domain.tld]]></email>
     *                   <firstname><![CDATA[Tester]]></firstname>
     *                   <lastname><![CDATA[Testuser]]></lastname>
     *                   <active>1</active>
     *                   <deleted>0</deleted>
     *                   <createdAt><![CDATA[2016-10-31T14:53:10+01:00]]></createdAt>
     *                   <updatedAt><![CDATA[2016-10-31T14:53:10+01:00]]></updatedAt>
     *                   <authId><![CDATA[24446666688888889]]></authId>
     *                   <mfaEnabled>1</mfaEnabled>
     *               </userType>
     *           </users>
     *       </response>
     *
     * ##Response error
     *
     * **Example JSON (body)**
     *
     *       {
     *           "code": 400,
     *           "status": "Bad Request",
     *           "trackId": "82d2846f7f",
     *           "date": "2016-10-31T14:54:18+01:00",
     *           "scriptTimeSec": 2.4736,
     *           "users": [
     *               {
     *                   "code": 400,
     *                   "status": "Bad Request",
     *                   "errors": [
     *                       {
     *                       "code": 400,
     *                       "message": "ErrorCode: UserAlreadyExists ErrorRef: u011 ShortMessage: Usernmae",
     *                       "exception": "external"
     *                       }
     *                   ]
     *               }
     *           ]
     *       }
     *
     * **Example XML (body)**
     *
     *       <?xml version="1.0" encoding="UTF-8"?>
     *       <response>
     *           <code>400</code>
     *           <status><![CDATA[Bad Request]]></status>
     *           <trackId><![CDATA[cf2fc5cf8d]]></trackId>
     *           <date><![CDATA[2016-10-31T14:55:28+01:00]]></date>
     *           <scriptTimeSec>2.255</scriptTimeSec>
     *           <users>
     *               <userType>
     *                   <code>400</code>
     *                   <status><![CDATA[Bad Request]]></status>
     *                   <errors>
     *                       <error>
     *                           <code>400</code>
     *                           <message><![CDATA[ErrorCode: UserAlreadyExists ErrorRef: u011 ShortMessage: Usernmae]]></message>
     *                           <exception><![CDATA[external]]></exception>
     *                       </error>
     *                   </errors>
     *               </userType>
     *           </users>
     *       </response>
     *
     * ##Response error status 401, 416 or 500
     *
     * **Example JSON (body)**
     *
     *       {
     *       "code": 416,
     *       "status": "Not Found",
     *       "trackId": "2d008cc4c5",
     *       "scriptTimeSec": 1.521,
     *       "count": 1,
     *       "errors": [
     *          {
     *          "code": 416,
     *          "status": "Not Found",
     *          "message": "no valid json string",
     *          "type": "external",
     *          "exception": "InvalidRequest",
     *          "debug": "not set"
     *          }
     *         ]
     *       }
     *
     * **Example XML (body)**
     *
     *      <?xml version="1.0" encoding="UTF-8"?>
     *      <response>
     *          <code>416</code>
     *          <status><![CDATA[Not Found]]></status>
     *          <trackId><![CDATA[16fe523df4]]></trackId>
     *          <scriptTimeSec>2.4459</scriptTimeSec>
     *          <count>1</count>
     *          <errors>
     *              <error>
     *                <code>416</code>
     *                <status><![CDATA[Not Found]]></status>
     *                <message><![CDATA[expected '>']]></message>
     *                <type><![CDATA[external]]></type>
     *                <exception><![CDATA[InvalidRequest]]></exception>
     *                <debug><![CDATA[not set]]></debug>
     *              </error>
     *          </errors>
     *      </response>
     *
     * @ApiDoc(
     *  resource="/api/{version}/sso-ws/user",
     *  description="secured: Updates an user in the sso-ws",
     *  headers={
     *      {
     *        "name"="username",
     *        "description"="sso username",
     *        "required"=true
     *       },
     *      {
     *        "name"="useridentifier",
     *        "description"="sso useridentifier",
     *        "required"=true
     *       },
     *      {
     *        "name"="servicetoken",
     *        "description"="servicetoken",
     *        "required"=true
     *       }
     *   },
     *  statusCodes={
     *         200="Returned when successful",
     *         204="Returned when no content was send",
     *         400="Returned when an error occurred",
     *         401="Unauthorized - Wrong credentials",
     *         403="Returned when the user is not authorized to do this request",
     *         416="Returned when the request is malformed",
     *         500="Returned on internal server errors"
     *  },
     *  tags={
     *      "stable" = "#000"
     *  },
     *  views = { "sso-ws" }
     * )
     * @Put("/v2/sso-ws/user/update/{guid}", name="_sso-ws_v2", options={ "method_prefix" = true })
     * @param Request $request
     * @Secure(roles="ROLE_API")
     * @return View
     */
    public function userUpdateByGuidAction(Request $request)
    {
        return $this->woM->userWs()->version2()->user()->update()->init($request);
    }
}
