<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2018 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\RestBundle\Controller\UserWs\Version2;

use FOS\RestBundle\Controller\Annotations\Get;
use FOS\RestBundle\Controller\Annotations\Post;
use FOS\RestBundle\View\View;
use JMS\SecurityExtraBundle\Annotation\Secure;
use Nelmio\ApiDocBundle\Annotation\ApiDoc;
use FOS\RestBundle\Controller\Annotations\NamePrefix;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Template;
use Symfony\Component\HttpFoundation\Request;
use FOS\RestBundle\Controller\FOSRestController;
use Sso\RestBundle\Worker\Worker as Worker;
use JMS\DiExtraBundle\Annotation as DI;
use LifeStyle\Tools\CachingBundle\Annotations\LsCache;

/**
 * @NamePrefix("api_rest_")
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package    Sso\RestBundle\Controller
 */
class UserSearchController extends FOSRestController
{
    /**
     * @DI\Inject("rest_worker_manager")
     * @var Worker
     */
    protected $woM;

    /**
     * *This service returns advanced search results. The service expects a JSON or XML in body and a header authentication.*
     *
     * #Request (header)
     *
     *      Key                 | Value               | Description
     *      ------------------- | ------------------- | --------------------------------------
     *      username            |  (string)           | your username
     *      useridentifier      |  (string)           | your useridentifier for given username
     *      servicetoken        |  (string)           | servicetoken from ws
     *
     * ##Request
     *
     * **Request example JSON**
     *
     *        {
     *          "advancedSearch": {
     *            "limit": "100",
     *            "offset": "0",
     *            "orderDir": "ASC",
     *            "fullResponse": "1",
     *            "filters": {
     *              "filter": [
     *                {
     *                  "strategy": "EQ",
     *                  "operator": "AND",
     *                  "userType": { "username": "m.hengstmann" }
     *                },
     *                {
     *                  "strategy": "EQ",
     *                  "operator": "AND",
     *                  "filters": {
     *                    "strategy": "EQ",
     *                    "operator": "AND",
     *                    "filter": [
     *                      {
     *                        "strategy": "EQ",
     *                        "operator": "AND",
     *                        "applicationType": {
     *                          "name": "SsoAdmin",
     *                          "attributeType": {
     *                            "name": "FIRSTNAME",
     *                            "value": "asdasd"
     *                          }
     *                        }
     *                      },
     *                      {
     *                        "strategy": "EQ",
     *                        "operator": "OR",
     *                        "applicationType": { "name": "FuselMusel" }
     *                      }
     *                    ]
     *                  }
     *                }
     *              ]
     *            }
     *          }
     *        }
     *
     *
     * **Request example XML**
     *
     *        <?xml version="1.0" encoding="UTF-8" ?>
     *        <userSearchAdvanced>
     *            <advancedSearch>
     *                <limit>100</limit>
     *                <offset>0</offset>
     *                <orderBy></orderBy>
     *                <orderDir>ASC</orderDir>
     *                <fullResponse>1</fullResponse>
     *                <filters>
     *                    <filter>
     *                        <strategy>EQ</strategy>
     *                        <operator>AND</operator>
     *                        <userType>
     *                            <username>m.hengstmann</username>
     *                        </userType>
     *                    </filter>
     *                    <filter>
     *                        <strategy>EQ</strategy>
     *                        <operator>AND</operator>
     *                        <filters>
     *                            <strategy>EQ</strategy>
     *                            <operator>AND</operator>
     *                            <filter>
     *                                <strategy>EQ</strategy>
     *                                <operator>AND</operator>
     *                                <applicationType>
     *                                    <name>SsoAdmin</name>
     *                                    <attributeType>
     *                                        <name>FIRSTNAME</name>
     *                                        <value>asdasd</value>
     *                                    </attributeType>
     *                                </applicationType>
     *                            </filter>
     *                            <filter>
     *                                <strategy>EQ</strategy>
     *                                <operator>OR</operator>
     *                                <applicationType>
     *                                    <name>FuselMusel</name>
     *                                </applicationType>
     *                            </filter>
     *                        </filters>
     *                    </filter>
     *                </filters>
     *            </advancedSearch>
     *        </userSearchAdvanced>
     *
     * ##Response success
     *
     * **Example JSON (body)**
     *
     *      {
     *          "code": 200,
     *          "trackId": "40b3e38fb4",
     *          "date": "2016-11-18T15:35:14+0100",
     *          "scriptTimeSec": 7.1507,
     *          "user": {
     *          "code": 200,
     *          "status": "OK",
     *          "users": [
     *              {
     *              "userType": {
     *                  "guid": "94efe14d29217fe315a0de552af87246",
     *              "username": "m.hengstmann",
     *                  "email": "m.hengstmann@life-style.de",
     *                  "firstname": "Michael",
     *                  "lastname": "Hengstmann",
     *                  "active": 1,
     *                  "deleted": 0,
     *                  "createdAt": "2016-09-06T14:06:45+0200",
     *                  "updatedAt": "2016-09-26T11:38:21+0200",
     *                  "lastLoginAt": "2016-11-18T11:16:58+0100",
     *                  "authId": "",
     *                  "ldapSearchAttributes": "",
     *                  "ldapSearchValue": "",
     *                  "mfaEnabled": 0
     *              },
     *              "applicationType": []
     *              }
     *          ]
     *          }
     *      }
     *
     * **Example XML (body)**
     *
     *      <?xml version="1.0" encoding="UTF-8" ?>
     *          <code>200</code>
     *          <trackId>40b3e38fb4</trackId>
     *          <date>2016-11-18T15:35:14+0100</date>
     *          <scriptTimeSec>7.1507</scriptTimeSec>
     *          <user>
     *              <code>200</code>
     *              <status>OK</status>
     *              <users>
     *                  <userType>
     *                      <guid>94efe14d29217fe315a0de552af87246</guid>
     *                        <username>m.hengstmann</username>
     *                      <email>m.hengstmann@life-style.de</email>
     *                      <firstname>Michael</firstname>
     *                      <lastname>Hengstmann</lastname>
     *                      <active>1</active>
     *                      <deleted>0</deleted>
     *                      <createdAt>2016-09-06T14:06:45+0200</createdAt>
     *                      <updatedAt>2016-09-26T11:38:21+0200</updatedAt>
     *                      <lastLoginAt>2016-11-18T11:16:58+0100</lastLoginAt>
     *                      <authId></authId>
     *                      <ldapSearchAttributes></ldapSearchAttributes>
     *                      <ldapSearchValue></ldapSearchValue>
     *                      <mfaEnabled>0</mfaEnabled>
     *                  </userType>
     *              </users>
     *          </user>
     *
     * ##Response error status 400
     *
     * **Example JSON (body)**
     *
     *        {
     *          "code": 400,
     *          "status": "Bad Request",
     *          "trackId": "4867901ca4",
     *          "date": "2016-11-23T16:54:00+01:00",
     *          "scriptTimeSec": 4.0317,
     *          "error": [
     *            {
     *              "code": 400,
     *              "message": "ErrorCode: usa ErrorRef:  ShortMessage: OrderDir must be ASC or DESC",
     *              "exception": "external"
     *            },
     *            {
     *              "code": 400,
     *              "message": "ErrorCode: usa ErrorRef:  ShortMessage: Strategy must be (EQ|NOT EQ|GT|GTE|LT|LTE|LIKE|NOT LIKE|REGEX)",
     *              "exception": "external"
     *            }
     *          ]
     *        }
     *
     ***Example XML (body)**
     *
     *        <?xml version="1.0" encoding="UTF-8"?>
     *        <userSearchAdvanced>
     *          <code>400</code>
     *          <status><![CDATA[Bad Request]]></status>
     *          <trackId><![CDATA[928914a733]]></trackId>
     *          <date><![CDATA[2016-11-23T17:03:25+01:00]]></date>
     *          <scriptTimeSec>2.7927</scriptTimeSec>
     *          <error>
     *            <error>
     *              <code>400</code>
     *              <message><![CDATA[ErrorCode: usa ErrorRef:  ShortMessage: OrderDir must be ASC or DESC]]></message>
     *              <exception><![CDATA[external]]></exception>
     *            </error>
     *            <error>
     *              <code>400</code>
     *              <message><![CDATA[ErrorCode: usa ErrorRef:  ShortMessage: Strategy must be (EQ|NOT EQ|GT|GTE|LT|LTE|LIKE|NOT LIKE|REGEX)]]></message>
     *              <exception><![CDATA[external]]></exception>
     *            </error>
     *          </error>
     *        </userSearchAdvanced>
     *
     * ##Response error status 401, 416 or 500
     *
     * **Example JSON (body)**
     *
     *       {
     *       "code": 416,
     *       "status": "Not Found",
     *       "trackId": "2d008cc4c5",
     *       "scriptTimeSec": 1.521,
     *       "count": 1,
     *       "errors": [
     *          {
     *          "code": 416,
     *          "status": "Not Found",
     *          "message": "no valid json string",
     *          "type": "external",
     *          "exception": "InvalidRequest",
     *          "debug": "not set"
     *          }
     *         ]
     *       }
     *
     * **Example XML (body)**
     *
     *      <?xml version="1.0" encoding="UTF-8"?>
     *      <response>
     *          <code>416</code>
     *          <status><![CDATA[Not Found]]></status>
     *          <trackId><![CDATA[16fe523df4]]></trackId>
     *          <scriptTimeSec>2.4459</scriptTimeSec>
     *          <count>1</count>
     *          <errors>
     *              <error>
     *                <code>416</code>
     *                <status><![CDATA[Not Found]]></status>
     *                <message><![CDATA[expected '>']]></message>
     *                <type><![CDATA[external]]></type>
     *                <exception><![CDATA[InvalidRequest]]></exception>
     *                <debug><![CDATA[not set]]></debug>
     *              </error>
     *          </errors>
     *      </response>
     *
     * @ApiDoc(
     *  resource="/api/{version}/sso-ws/userSearch",
     *  description="secured: Advanced search",
     *  headers={
     *      {
     *        "name"="username",
     *        "description"="sso username",
     *        "required" = true
     *       },
     *      {
     *        "name"="useridentifier",
     *        "description"="sso useridentifier",
     *        "required" = true
     *       },
     *      {
     *        "name"="servicetoken",
     *        "description"="servicetoken",
     *        "required" = true
     *       }
     *   },
     *  statusCodes={
     *         200="Returned when successful",
     *         204="Returned when no content was send",
     *         400="Returned when an error occurred",
     *         401="Unauthorized - Wrong credentials",
     *         403="Returned when the user is not authorized to do this request",
     *         500="Returned on internal server errors"
     *  },
     *  tags={
     *      "stable" = "#000",
     *  },
     *  output="Sso\RestBundle\ModelRest\Response\UserWs\Version2\UserSearch\Advanced\Response",
     *  views = { "sso-ws" }
     * )
     * @Post("/v2/sso-ws/userSearch/advanced", name="_sso-ws_v2", options={ "method_prefix" = true })
     * @param Request $request
     * @Secure(roles="ROLE_API")
     * @LsCache(cacheTTL=300, keyHeader=true)
     * @return View
     * @Template()
     */
    public function userSearchAdvancedAction(Request $request)
    {
        return $this->woM->userWs()->version2()->userSearch()->advanced()->init($request);
    }

    /**
     *
     * *This service returns search results by application. The service expects a JSON or XML in body and a header authentication.*
     *
     * #Request (header)
     *
     *      Key                 | Value               | Description
     *      ------------------- | ------------------- | --------------------------------------
     *      username            |  (string)           | your username
     *      useridentifier      |  (string)           | your useridentifier for given username
     *      servicetoken        |  (string)           | servicetoken from ws
     *
     * #Response success
     *
     * **Example JSON (body)**
     *
     *       {
     *           "code": 200,
     *           "status": "OK",
     *           "trackId": "e05a74dbd6",
     *           "date": "2016-11-28T08:37:31+01:00",
     *           "scriptTimeSec": 1.2358,
     *           "userByApplication": {
     *              "userType": [
     *                   {
     *                       "code": 200,
     *                       "status": "OK",
     *                       "guid": "701413c416c43be173aa39e467168a4f",
     *                       "email": "test@test.com",
     *                       "username": "U0MX101460",
     *                       "active": 0
     *                   }
     *               ]
     *           }
     *       }
     *
     * **Example XML (body)**
     *
     *       <?xml version="1.0" encoding="UTF-8"?>
     *       <userSearchByApplication>
     *           <code>200</code>
     *           <status><![CDATA[OK]]></status>
     *           <trackId><![CDATA[b5d3c04bef]]></trackId>
     *           <date><![CDATA[2016-11-28T08:37:09+01:00]]></date>
     *           <scriptTimeSec>2.4816</scriptTimeSec>
     *           <userByApplication>
     *               <userType>
     *                   <code>200</code>
     *                   <status><![CDATA[OK]]></status>
     *                   <guid><![CDATA[701413c416c43be173aa39e467168a4f]]></guid>
     *                   <email><![CDATA[test@test.com]]></email>
     *                   <username><![CDATA[U0MX101460]]></username>
     *                   <active>0</active>
     *               </userType>
     *           </userByApplication>
     *       </userSearchByApplication>
     *
     * #Response error
     *
     * **Example JSON (body)**
     *
     *       {
     *           "code": 400,
     *           "status": "Bad Request",
     *           "trackId": "697a04fa7f",
     *           "date": "2016-11-28T08:38:49+01:00",
     *           "scriptTimeSec": 3.0308,
     *           "userByAttribute": {
     *               "userType": [
     *                   {
     *                       "code": 400,
     *                       "status": "Bad Request",
     *                       "errors": [
     *                           {
     *                               "code": 400,
     *                               "message": "ErrorCode: XXX ErrorRef: app ShortMessage: Invalid Application",
     *                               "exception": "external"
     *                           }
     *                       ]
     *                   }
     *               ]
     *           }
     *       }
     *
     * **Example XML (body)**
     *
     *       <?xml version="1.0" encoding="UTF-8"?>
     *       <userSearchByApplication>
     *           <code>400</code>
     *           <status><![CDATA[Bad Request]]></status>
     *           <trackId><![CDATA[5c9aa2cf37]]></trackId>
     *           <date><![CDATA[2016-11-28T08:38:49+01:00]]></date>
     *           <scriptTimeSec>7.3158</scriptTimeSec>
     *           <userByApplication>
     *               <userType>
     *                   <code>400</code>
     *                   <status><![CDATA[Bad Request]]></status>
     *                   <errors>
     *                       <error>
     *                           <code>400</code>
     *                           <message><![CDATA[ErrorCode: XXX ErrorRef: app ShortMessage: Invalid Application]]></message>
     *                           <exception><![CDATA[external]]></exception>
     *                       </error>
     *                   </errors>
     *               </userType>
     *           </userByApplication>
     *       </userSearchByApplication>
     *
     * ##Response error status 401, 416 or 500
     *
     * **Example JSON (body)**
     *
     *       {
     *       "code": 416,
     *       "status": "Not Found",
     *       "trackId": "2d008cc4c5",
     *       "scriptTimeSec": 1.521,
     *       "count": 1,
     *       "errors": [
     *          {
     *          "code": 416,
     *          "status": "Not Found",
     *          "message": "no valid json string",
     *          "type": "external",
     *          "exception": "InvalidRequest",
     *          "debug": "not set"
     *          }
     *         ]
     *       }
     *
     * **Example XML (body)**
     *
     *      <?xml version="1.0" encoding="UTF-8"?>
     *      <response>
     *          <code>416</code>
     *          <status><![CDATA[Not Found]]></status>
     *          <trackId><![CDATA[16fe523df4]]></trackId>
     *          <scriptTimeSec>2.4459</scriptTimeSec>
     *          <count>1</count>
     *          <errors>
     *              <error>
     *                <code>416</code>
     *                <status><![CDATA[Not Found]]></status>
     *                <message><![CDATA[expected '>']]></message>
     *                <type><![CDATA[external]]></type>
     *                <exception><![CDATA[InvalidRequest]]></exception>
     *                <debug><![CDATA[not set]]></debug>
     *              </error>
     *          </errors>
     *      </response>
     *
     * @ApiDoc(
     *  resource="/api/{version}/sso-ws/userSearch",
     *  description="secured: Searches by application",
     *  statusCodes={
     *         200="Returned when successful",
     *         204="Returned when no content was send",
     *         400="Returned when an error occurred",
     *         401="Unauthorized - Wrong credentials",
     *         403="Returned when the user is not authorized to do this request",
     *         416="Returned when the request is malformed",
     *         500="Returned on internal server errors"
     *  },
     *  headers={
     *      {
     *        "name"="username",
     *        "description"="sso username",
     *        "required"=true
     *       },
     *      {
     *        "name"="useridentifier",
     *        "description"="sso useridentifier",
     *        "required"=true
     *       },
     *      {
     *        "name"="servicetoken",
     *        "description"="servicetoken",
     *        "required"=true
     *       }
     *  },
     *  parameters={
     *      {"name"="applicationName", "dataType"="string", "required"=true, "description"="Application name"},
     *      {"name"="applicationActive", "dataType"="integer", "required"=true, "description"="Application active"},
     *      {"name"="modifiedAfter", "dataType"="dateTime", "required"=false, "description"="Modified after datetime"}
     *  },
     *  tags={
     *      "stable" = "#000"
     *  },
     *  output="Sso\RestBundle\ModelRest\Response\UserWs\Version2\UserSearch\FindByApplication\Response",
     *  views = { "sso-ws" }
     * )
     *
     * @Get("/v2/sso-ws/userSearch/findByApplication", name="_sso-ws_v2", options={ "method_prefix" = true })
     * @Secure(roles="ROLE_API")
     * @LsCache(cacheTTL=300, keyHeader=true)
     * @param Request $request
     * @return View
     */
    public function getFindByApplicationAction(Request $request)
    {
        return $this->woM->userWs()->version2()->userSearch()->findByApplication()->init($request);
    }

    /**
     *
     * *This service returns search results by application attribute. The service expects a JSON or XML in body and a header authentication.*
     *
     * #Request (header)
     *
     *      Key                 | Value               | Description
     *      ------------------- | ------------------- | --------------------------------------
     *      username            |  (string)           | your username
     *      useridentifier      |  (string)           | your useridentifier for given username
     *      servicetoken        |  (string)           | servicetoken from ws
     *
     * #Response success
     *
     * **Example JSON (body)**
     *
     *       {
     *           "code": 200,
     *           "status": "OK",
     *           "trackId": "7ad0d51472",
     *           "date": "2016-11-23T10:38:04+01:00",
     *           "scriptTimeSec": 3.8362,
     *           "userByAttribute": {
     *              "userType": [
     *                   {
     *                       "code": 200,
     *                       "status": "OK",
     *                       "guid": "701413c416c43be173aa39e467168a4f",
     *                       "email": "test@test.com",
     *                       "username": "U0MX101460",
     *                       "active": 0
     *                   }
     *               ]
     *           }
     *       }
     *
     * **Example XML (body)**
     *
     *       <?xml version="1.0" encoding="UTF-8"?>
     *       <userSearchByAttribute>
     *           <code>200</code>
     *           <status><![CDATA[OK]]></status>
     *           <trackId><![CDATA[6393946ca7]]></trackId>
     *           <date><![CDATA[2016-11-23T12:54:47+01:00]]></date>
     *           <scriptTimeSec>7.2904</scriptTimeSec>
     *           <userByAttribute>
     *               <userType>
     *                   <code>200</code>
     *                   <status><![CDATA[OK]]></status>
     *                   <guid><![CDATA[701413c416c43be173aa39e467168a4f]]></guid>
     *                   <email><![CDATA[test@test.com]]></email>
     *                   <username><![CDATA[U0MX101460]]></username>
     *                   <active>0</active>
     *               </userType>
     *           </userByAttribute>
     *       </userSearchByAttribute>
     *
     * #Response error
     *
     * **Example JSON (body)**
     *
     *       {
     *           "code": 400,
     *           "status": "Bad Request",
     *           "trackId": "a705fd05d9",
     *           "date": "2016-11-23T12:57:36+01:00",
     *           "scriptTimeSec": 1.3046,
     *           "userByAttribute": {
     *               "userType": [
     *                   {
     *                       "code": 400,
     *                       "status": "Bad Request",
     *                       "errors": [
     *                           {
     *                               "code": 400,
     *                               "message": "ErrorCode: XXX ErrorRef: app ShortMessage: Invalid Application Attribute",
     *                               "exception": "external"
     *                           }
     *                       ]
     *                   }
     *               ]
     *           }
     *       }
     *
     * **Example XML (body)**
     *
     *       <?xml version="1.0" encoding="UTF-8"?>
     *       <userSearchByAttribute>
     *           <code>400</code>
     *           <status><![CDATA[Bad Request]]></status>
     *           <trackId><![CDATA[4a0c69cce3]]></trackId>
     *           <date><![CDATA[2016-11-23T12:58:43+01:00]]></date>
     *           <scriptTimeSec>5.011</scriptTimeSec>
     *           <userByAttribute>
     *               <userType>
     *                   <code>400</code>
     *                   <status><![CDATA[Bad Request]]></status>
     *                   <errors>
     *                       <error>
     *                           <code>400</code>
     *                           <message><![CDATA[ErrorCode: XXX ErrorRef: app ShortMessage: Invalid Application Attribute]]></message>
     *                           <exception><![CDATA[external]]></exception>
     *                       </error>
     *                   </errors>
     *               </userType>
     *           </userByAttribute>
     *       </userSearchByAttribute>
     *
     * ##Response error status 401, 416 or 500
     *
     * **Example JSON (body)**
     *
     *       {
     *       "code": 416,
     *       "status": "Not Found",
     *       "trackId": "2d008cc4c5",
     *       "scriptTimeSec": 1.521,
     *       "count": 1,
     *       "errors": [
     *          {
     *          "code": 416,
     *          "status": "Not Found",
     *          "message": "no valid json string",
     *          "type": "external",
     *          "exception": "InvalidRequest",
     *          "debug": "not set"
     *          }
     *         ]
     *       }
     *
     * **Example XML (body)**
     *
     *      <?xml version="1.0" encoding="UTF-8"?>
     *      <response>
     *          <code>416</code>
     *          <status><![CDATA[Not Found]]></status>
     *          <trackId><![CDATA[16fe523df4]]></trackId>
     *          <scriptTimeSec>2.4459</scriptTimeSec>
     *          <count>1</count>
     *          <errors>
     *              <error>
     *                <code>416</code>
     *                <status><![CDATA[Not Found]]></status>
     *                <message><![CDATA[expected '>']]></message>
     *                <type><![CDATA[external]]></type>
     *                <exception><![CDATA[InvalidRequest]]></exception>
     *                <debug><![CDATA[not set]]></debug>
     *              </error>
     *          </errors>
     *      </response>
     *
     * @ApiDoc(
     *  resource="/api/{version}/sso-ws/userSearch",
     *  description="secured: Searches by application attribute",
     *  statusCodes={
     *         200="Returned when successful",
     *         204="Returned when no content was send",
     *         400="Returned when an error occurred",
     *         401="Unauthorized - Wrong credentials",
     *         403="Returned when the user is not authorized to do this request",
     *         416="Returned when the request is malformed",
     *         500="Returned on internal server errors"
     *  },
     *  headers={
     *      {
     *        "name"="username",
     *        "description"="sso username",
     *        "required"=true
     *       },
     *      {
     *        "name"="useridentifier",
     *        "description"="sso useridentifier",
     *        "required"=true
     *       },
     *      {
     *        "name"="servicetoken",
     *        "description"="servicetoken",
     *        "required"=true
     *       }
     *  },
     *  parameters={
     *      {"name"="applicationName", "dataType"="string", "required"=true, "description"="Application name"},
     *      {"name"="applicationActive", "dataType"="integer", "required"=true, "description"="Application active"},
     *      {"name"="attributeName", "dataType"="string", "required"=true, "description"="Attribute name"},
     *      {"name"="attributeValue", "dataType"="string", "required"=true, "description"="Attribute value"},
     *      {"name"="modifiedAfter", "dataType"="dateTime", "required"=false, "description"="Modified after datetime"}
     *  },
     *  tags={
     *      "stable" = "#000"
     *  },
     *  output="Sso\RestBundle\ModelRest\Response\UserWs\Version2\UserSearch\FindByApplicationAttribute\Response",
     *  views = { "sso-ws" }
     * )
     *
     * @Get("/v2/sso-ws/userSearch/findByApplicationAttribute", name="_sso-ws_v2", options={ "method_prefix" = true })
     * @Secure(roles="ROLE_API")
     * @LsCache(cacheTTL=300, keyHeader=true)
     * @param Request $request
     * @return View
     */
    public function getFindByApplicationAttributeAction(Request $request)
    {
        return $this->woM->userWs()->version2()->userSearch()->findByApplicationAttribute()->init($request);
    }

    /**
     *
     * *This service returns search results by application role. The service expects a JSON or XML in body and a header authentication.*
     *
     * #Request (header)
     *
     *      Key                 | Value               | Description
     *      ------------------- | ------------------- | --------------------------------------
     *      username            |  (string)           | your username
     *      useridentifier      |  (string)           | your useridentifier for given username
     *      servicetoken        |  (string)           | servicetoken from ws
     *
     * #Response success
     *
     * **Example JSON (body)**
     *
     *       {
     *           "code": 200,
     *           "status": "OK",
     *           "trackId": "aee98b824c",
     *           "date": "2016-11-24T11:05:13+01:00",
     *           "scriptTimeSec": 2.5605,
     *           "userByRole": {
     *              "userType": [
     *                   {
     *                       "code": 200,
     *                       "status": "OK",
     *                       "guid": "8113fd04e027b0b47751d7636145c965",
     *                       "email": "andrew@teammate-exec.com",
     *                       "username": "agsqwe",
     *                       "active": 1
     *                   }
     *               ]
     *           }
     *       }
     *
     * **Example XML (body)**
     *
     *       <?xml version="1.0" encoding="UTF-8"?>
     *       <userSearchByRole>
     *           <code>200</code>
     *           <status><![CDATA[OK]]></status>
     *           <trackId><![CDATA[a036e0620e]]></trackId>
     *           <date><![CDATA[2016-11-24T11:04:01+01:00]]></date>
     *           <scriptTimeSec>2.8368</scriptTimeSec>
     *           <userByRole>
     *               <userType>
     *                   <code>200</code>
     *                   <status><![CDATA[OK]]></status>
     *                   <guid><![CDATA[8113fd04e027b0b47751d7636145c965]]></guid>
     *                   <email><![CDATA[andrew@teammate-exec.com]]></email>
     *                   <username><![CDATA[agsqwe]]></username>
     *                   <active>1</active>
     *               </userType>
     *           </userByRole>
     *       </userSearchByRole>
     *
     * #Response error
     *
     * **Example JSON (body)**
     *
     *       {
     *           "code": 400,
     *           "status": "Bad Request",
     *           "trackId": "f8507e66c4",
     *           "date": "2016-11-24T11:07:58+01:00",
     *           "scriptTimeSec": 5.1017,
     *           "userByAttribute": {
     *               "userType": [
     *                   {
     *                       "code": 400,
     *                       "status": "Bad Request",
     *                       "errors": [
     *                           {
     *                               "code": 400,
     *                               "message": "ErrorCode: usfbr2 ErrorRef: usfbr2002 ShortMessage: Invalid Application Role",
     *                               "exception": "external"
     *                           }
     *                       ]
     *                   }
     *               ]
     *           }
     *       }
     *
     * **Example XML (body)**
     *
     *       <?xml version="1.0" encoding="UTF-8"?>
     *       <userSearchByRole>
     *           <code>400</code>
     *           <status><![CDATA[Bad Request]]></status>
     *           <trackId><![CDATA[7d360c573e]]></trackId>
     *           <date><![CDATA[2016-11-24T11:07:56+01:00]]></date>
     *           <scriptTimeSec>7.2328</scriptTimeSec>
     *           <userByRole>
     *               <userType>
     *                   <code>400</code>
     *                   <status><![CDATA[Bad Request]]></status>
     *                   <errors>
     *                       <error>
     *                           <code>400</code>
     *                           <message><![CDATA[ErrorCode: usfbr2 ErrorRef: usfbr2002 ShortMessage: Invalid Application Role]]></message>
     *                           <exception><![CDATA[external]]></exception>
     *                       </error>
     *                   </errors>
     *               </userType>
     *           </userByRole>
     *       </userSearchByRole>
     *
     * ##Response error status 401, 416 or 500
     *
     * **Example JSON (body)**
     *
     *       {
     *       "code": 416,
     *       "status": "Not Found",
     *       "trackId": "2d008cc4c5",
     *       "scriptTimeSec": 1.521,
     *       "count": 1,
     *       "errors": [
     *          {
     *          "code": 416,
     *          "status": "Not Found",
     *          "message": "no valid json string",
     *          "type": "external",
     *          "exception": "InvalidRequest",
     *          "debug": "not set"
     *          }
     *         ]
     *       }
     *
     * **Example XML (body)**
     *
     *      <?xml version="1.0" encoding="UTF-8"?>
     *      <response>
     *          <code>416</code>
     *          <status><![CDATA[Not Found]]></status>
     *          <trackId><![CDATA[16fe523df4]]></trackId>
     *          <scriptTimeSec>2.4459</scriptTimeSec>
     *          <count>1</count>
     *          <errors>
     *              <error>
     *                <code>416</code>
     *                <status><![CDATA[Not Found]]></status>
     *                <message><![CDATA[expected '>']]></message>
     *                <type><![CDATA[external]]></type>
     *                <exception><![CDATA[InvalidRequest]]></exception>
     *                <debug><![CDATA[not set]]></debug>
     *              </error>
     *          </errors>
     *      </response>
     *
     * @ApiDoc(
     *  resource="/api/{version}/sso-ws/userSearch",
     *  description="secured: Searches by application role",
     *  statusCodes={
     *         200="Returned when successful",
     *         204="Returned when no content was send",
     *         400="Returned when an error occurred",
     *         401="Unauthorized - Wrong credentials",
     *         403="Returned when the user is not authorized to do this request",
     *         416="Returned when the request is malformed",
     *         500="Returned on internal server errors"
     *  },
     *  headers={
     *      {
     *        "name"="username",
     *        "description"="sso username",
     *        "required"=true
     *       },
     *      {
     *        "name"="useridentifier",
     *        "description"="sso useridentifier",
     *        "required"=true
     *       },
     *      {
     *        "name"="servicetoken",
     *        "description"="servicetoken",
     *        "required"=true
     *       }
     *  },
     *  parameters={
     *      {"name"="applicationName", "dataType"="string", "required"=true, "description"="Application name"},
     *      {"name"="applicationActive", "dataType"="integer", "required"=true, "description"="Application active"},
     *      {"name"="roleName", "dataType"="string", "required"=true, "description"="Role name"},
     *      {"name"="roleActive", "dataType"="integer", "required"=true, "description"="Role active"},
     *      {"name"="modifiedAfter", "dataType"="dateTime", "required"=false, "description"="Modified after datetime"}
     *  },
     *  tags={
     *      "stable" = "#000"
     *  },
     *  output="Sso\RestBundle\ModelRest\Response\UserWs\Version2\UserSearch\FindByApplicationRole\Response",
     *  views = { "sso-ws" }
     * )
     *
     * @Get("/v2/sso-ws/userSearch/findByApplicationRole", name="_sso-ws_v2", options={ "method_prefix" = true })
     * @Secure(roles="ROLE_API")
     * @LsCache(cacheTTL=300, keyHeader=true)
     * @param Request $request
     * @return View
     */
    public function getFindByApplicationRoleAction(Request $request)
    {
        return $this->woM->userWs()->version2()->userSearch()->findByApplicationRole()->init($request);
    }
}
