<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\RestBundle\Tests\ModelRest\Request\UserWs\Version1\User\Add;

use Sso\RestBundle\ModelRest\Request\UserWs\Version1\User\Add\Request;
use Sso\RestBundle\ModelRest\Request\UserWs\Version1\User\Add\UserType;
use Symfony\Bundle\FrameworkBundle\Test\WebTestCase;
use Sso\RestBundle\Api\Manager as ApiManager;
use JMS\Serializer\SerializerInterface;
use Faker\Factory as FakerFactory;
use Faker\Generator as FakerGenerator;

/**
 * Class RequestTest
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package Sso\RestBundle\Tests\ModelRest
 */
class RequestTest extends WebTestCase
{
    /**
     * @var SerializerInterface
     */
    private $serializer;

    /**
     * @var ApiManager
     */
    private $apiManager;

    /**
     * @var FakerGenerator
     */
    private $faker;

    /**
     * @var string
     */
    private $email;

    /**
     * @var string
     */
    private $username;

    /**
     * @var string
     */
    private $firstName;

    /**
     * @var string
     */
    private $lastName;

    /**
     * @var string
     */
    private $password;

    /**
     * @var boolean
     */
    private $active;

    /**
     * @var string
     */
    private $authId;

    protected function setUp()
    {
        parent::setUp();
        static::bootKernel();
        $this->apiManager = new ApiManager(static::$kernel->getContainer());
        $this->serializer = $this->apiManager->serializer();
        $this->faker = FakerFactory::create();

        // Set same random data
        $this->email = $this->faker->email;
        $this->username = $this->faker->userName;
        $this->firstName = $this->faker->firstName;
        $this->lastName = $this->faker->lastName;
        $this->password = 'd5zJwgnzU2uTtv97';
        $this->active = $this->faker->boolean ? 1 : 0;
        $this->authId = $this->faker->randomNumber();
    }

    public function testJsonSerialize()
    {
        $this->assertJsonStringEqualsJsonString($this->getRequestBody('json'), $this->setRequestBody('json'));
    }

    public function testXmlSerialize()
    {
        $this->assertXmlStringEqualsXmlString($this->getRequestBody('xml'), $this->setRequestBody('xml'));
    }

    /**
     * @param string $type (xml | json)
     * @return string
     */
    private function setRequestBody($type = 'xml')
    {
        $request = new Request();
        $userType = new UserType();
        $userType->setEmail($this->email);
        $userType->setUsername($this->username);
        $userType->setFirstName($this->firstName);
        $userType->setLastName($this->lastName);
        $userType->setPassword($this->password);
        $userType->setActive($this->active);
        $userType->setAuthId($this->authId);
        $userType->setLdapSearchAttributes('');
        $userType->setLdapSearchValue('');

        $request->addUserType($userType);
        return $this->serializer->serialize($request, ($type !== 'json' ? 'xml' : 'json'));
    }

    /**
     * @param string $type (xml | json)
     * @return string
     */
    private function getRequestBody($type = 'xml')
    {
        if ($type === 'json') {
            return '{
                "users" : [
                    {
                        "email" : "' . $this->email . '",
                        "username" : "' . $this->username . '",
                        "firstname" : "' . $this->firstName . '",
                        "lastname" : "' . $this->lastName . '",
                        "password" : "' . $this->password . '",
                        "active" : ' . $this->active . ',
                        "authId" : "' . $this->authId . '",
                        "ldapSearchAttribute" : "",
                        "ldapSearchValue" : ""
                    }
                ]
            }';
        } else {
            return '<userAdd>
                <users>
                    <userType>
                        <email><![CDATA[' . $this->email . ']]></email>
                        <username><![CDATA[' . $this->username . ']]></username>
                        <firstname><![CDATA[' . $this->firstName . ']]></firstname>
                        <lastname><![CDATA[' . $this->lastName . ']]></lastname>
                        <password><![CDATA[' . $this->password . ']]></password>
                        <active>' . $this->active . '</active>
                        <authId><![CDATA[' . $this->authId . ']]></authId>
                        <ldapSearchAttribute></ldapSearchAttribute>
                        <ldapSearchValue></ldapSearchValue>
                    </userType>
                </users>
            </userAdd>';
        }
    }

    public function testGetterUserType()
    {
        /** @var  Request $requestModel */
        $requestModel = $this->serializer->deserialize(
            $this->setRequestBody('json'),
            Request::class,
            'json'
        );

        $userType = $requestModel->getUserType()->get(0);

        $this->assertEquals($this->email, $userType->getEmail());
        $this->assertEquals($this->username, $userType->getUsername());
        $this->assertEquals($this->firstName, $userType->getFirstName());
        $this->assertEquals($this->lastName, $userType->getLastName());
        $this->assertEquals($this->password, $userType->getPassword());
        $this->assertEquals($this->active, $userType->getActive());
        $this->assertEquals($this->authId, $userType->getAuthId());
    }
}
