<?php

/**
 * Class RequestTest
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\RestBundle\Tests\ModelRest\Request\UserWs\Version1\UserApplicationRole\Add;

use Sso\RestBundle\Api\Manager as ApiManager;
use Sso\RestBundle\ModelRest\Request\UserWs\Version1\UserApplicationRole\Add;
use Faker\Factory as FakerFactory;
use Faker\Generator as FakerGenerator;
use JMS\Serializer\SerializerInterface;
use Symfony\Bundle\FrameworkBundle\Test\WebTestCase;

/**
 * Class RequestTest
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package Sso\RestBundle\Tests\ModelRest\Request\UserWs\Version1\UserApplicationRole\Add
 */
class RequestTest extends WebTestCase
{
    /**
     * @var SerializerInterface
     */
    private $serializer;

    /**
     * @var ApiManager
     */
    private $apiManager;

    /**
     * @var FakerGenerator
     */
    private $faker;

    /**
     * @var Add\Request
     */
    private $request;

    protected function setUp()
    {
        parent::setUp();
        static::bootKernel();
        $this->apiManager = new ApiManager(static::$kernel->getContainer());
        $this->serializer = $this->apiManager->serializer();
        $this->faker = FakerFactory::create();

        $this->setUpRequest();
    }

    protected function setUpRequest()
    {
        $request = new Add\Request();
        $roleType = new Add\RoleType();
        $userType = new Add\UserType();
        $applicationType = new Add\ApplicationType();

        $applicationType->setName($this->faker->name);
        $userType->setIdentifier($this->faker->email);
        $roleType->setName($this->faker->name)
            ->setActive($this->faker->numberBetween(0, 1))
            ->setUserType($userType)
            ->setApplicationType($applicationType);
        $request->addRoleType($roleType);

        $this->request = $request;
    }

    public function testGetAndAddRoleTypes()
    {
        $request = new Add\Request();
        $roleType1 = new Add\RoleType();
        $roleType2 = new Add\RoleType();

        $roleType1Name = $this->faker->name;
        $roleType1Active = $this->faker->numberBetween(0, 1);
        $roleType1->setName($roleType1Name)
            ->setActive($roleType1Active);

        $roleType2Name = $this->faker->name;
        $roleType2Active = $this->faker->numberBetween(0, 1);
        $roleType2->setName($roleType2Name)
            ->setActive($roleType2Active);

        $request->addRoleType($roleType1)
            ->addRoleType($roleType2);

        static::assertEquals($request->getRoleTypes()->count(), 2);
        static::assertEquals($request->getRoleTypes()->get(0)->getName(), $roleType1Name);
        static::assertEquals($request->getRoleTypes()->get(1)->getName(), $roleType2Name);
    }

    /**
     * @param string $type (xml | json)
     * @return string
     */
    private function getSerializedRequestContent($type = 'xml')
    {
        switch ($type) {
            case 'json':
                return <<< JSON
{
	"userRoles": [
		{
			"name": "{$this->request->getRoleTypes()->get(0)->getName()}",
			"active": "{$this->request->getRoleTypes()->get(0)->getActive()}",
			"userType": {
				"identifier": "{$this->request->getRoleTypes()->get(0)->getUserType()->getIdentifier()}"
			},
			"applicationType": {
				"name": "{$this->request->getRoleTypes()->get(0)->getApplicationType()->getName()}"
			}
		}
	]
}
JSON;
            case 'xml':
                return <<< XML
<userApplicationRoleAdd>
    <userRoles>
        <roleType>
            <name>{$this->request->getRoleTypes()->get(0)->getName()}</name>
            <active>{$this->request->getRoleTypes()->get(0)->getActive()}</active>
            <userType>
                <identifier>{$this->request->getRoleTypes()->get(0)->getUserType()->getIdentifier()}</identifier>
            </userType>
            <applicationType>
                <name>{$this->request->getRoleTypes()->get(0)->getApplicationType()->getName()}</name>
            </applicationType>
        </roleType>
    </userRoles>
</userApplicationRoleAdd>
XML;
            default:
                throw new \PHPUnit_Framework_Exception('Invalid request type (hint: not json/xml)');
        }
    }
}