<?php

/**
 * Class RequestTest
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\RestBundle\Tests\ModelRest\Request\UserWs\Version1\UserApplicationRole\Delete;

use Sso\RestBundle\Api\Manager as ApiManager;
use Sso\RestBundle\ModelRest\Request\UserWs\Version1\UserApplicationRole\Delete\ApplicationType;
use Sso\RestBundle\ModelRest\Request\UserWs\Version1\UserApplicationRole\Delete\Request;
use Sso\RestBundle\ModelRest\Request\UserWs\Version1\UserApplicationRole\Delete\UserApplicationRoleType;
use Sso\RestBundle\ModelRest\Request\UserWs\Version1\UserApplicationRole\Delete\UserType;
use Faker\Factory as FakerFactory;
use Faker\Generator as FakerGenerator;
use JMS\Serializer\SerializerInterface;
use Symfony\Bundle\FrameworkBundle\Test\WebTestCase;

/**
 * Class RequestTest
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package Sso\RestBundle\Tests\ModelRest\Request\UserWs\Version1\UserApplicationRole\Delete
 */
class RequestTest extends WebTestCase
{
    /**
     * @var SerializerInterface
     */
    private $serializer;

    /**
     * @var ApiManager
     */
    private $apiManager;

    /**
     * @var FakerGenerator
     */
    private $faker;

    /**
     * @var string
     */
    private $userIdentifier;

    /**
     * @var string
     */
    private $applicationName;

    /**
     * @var string
     */
    private $roleName;

    protected function setUp()
    {
        parent::setUp();
        static::bootKernel();
        $this->apiManager = new ApiManager(static::$kernel->getContainer());
        $this->serializer = $this->apiManager->serializer();
        $this->faker = FakerFactory::create();

        // Set same random data
        $this->userIdentifier = $this->faker->email;
        $this->applicationName = $this->faker->name;
        $this->roleName = $this->faker->name;
    }

    public function testAddUserType()
    {
        $request = new Request();
        $userType = new UserType();
        $applicationType = new ApplicationType();
        $userApplicationRoleType = new UserApplicationRoleType();

        $applicationType->setName($this->applicationName);

        $userType
            ->setIdentifier($this->userIdentifier);

        $userApplicationRoleType
            ->setApplicationType($applicationType)
            ->setUserType($userType)
            ->setName($this->roleName);

        $request->addUserApplicationRoleType($userApplicationRoleType);

        static::assertNotEmpty($request->getUserApplicationRoleTypes());
        static::assertEquals($request->getUserApplicationRoleTypes()->get(0), $userApplicationRoleType);
    }

    public function testJSONSerialize()
    {
        $request = new Request();
        $userType = new UserType();
        $applicationType = new ApplicationType();
        $userApplicationRoleType = new UserApplicationRoleType();

        $applicationType->setName($this->applicationName);
        $userType->setIdentifier($this->userIdentifier);
        $userApplicationRoleType
            ->setName($this->roleName)
            ->setUserType($userType)
            ->setApplicationType($applicationType);

        $request->addUserApplicationRoleType($userApplicationRoleType);

        static::assertJsonStringEqualsJsonString($this->serializer->serialize($request, 'json'),
            $this->getSerializedRequestContent('json'));
    }

    public function testXMLSerialize()
    {
        $request = new Request();
        $userType = new UserType();
        $applicationType = new ApplicationType();
        $userApplicationRoleType = new UserApplicationRoleType();

        $applicationType->setName($this->applicationName);
        $userType->setIdentifier($this->userIdentifier);
        $userApplicationRoleType
            ->setName($this->roleName)
            ->setUserType($userType)
            ->setApplicationType($applicationType);

        $request->addUserApplicationRoleType($userApplicationRoleType);

        static::assertXmlStringEqualsXmlString($this->serializer->serialize($request, 'xml'),
            $this->getSerializedRequestContent('xml'));
    }

    /**
     * @param string $type (xml | json)
     * @return string
     */
    private function getSerializedRequestContent($type = 'xml')
    {
        switch ($type) {
            case 'json':
                return <<< JSON
                {
                    "userApplicationRoles": [
                        {
                            "name": "$this->roleName",
                            "userType": {
                                "identifier": "$this->userIdentifier"
                            },
                            "applicationType": {
                                "name": "$this->applicationName"
                            }
                        }
                    ]
                }
JSON;
            case 'xml':
                return <<< XML
                <userApplicationRoleDelete>
                    <userApplicationRoles>
                        <userApplicationRoleType>
                            <name><![CDATA[$this->roleName]]></name>
                            <userType>
                                <identifier><![CDATA[$this->userIdentifier]]></identifier>
                            </userType>
                            <applicationType>
                                <name><![CDATA[$this->applicationName]]></name>
                            </applicationType>
                        </userApplicationRoleType>
                    </userApplicationRoles>
                </userApplicationRoleDelete>
XML;
            default:
                throw new \PHPUnit_Framework_Exception('Invalid request type (hint: not json/xml)');
        }
    }
}