<?php

/**
 * Class RequestTest
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\RestBundle\Tests\ModelRest\Request\UserWs\Version1\UserService\ActivateConfirm;

use Sso\RestBundle\Api\Manager as ApiManager;
use Sso\RestBundle\ModelRest\Request\UserWs\Version1\UserService\ActivateConfirm;
use Faker\Factory as FakerFactory;
use Faker\Generator as FakerGenerator;
use JMS\Serializer\SerializerInterface;
use Symfony\Bundle\FrameworkBundle\Test\WebTestCase;

/**
 * Class RequestTest
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package Sso\RestBundle\Tests\ModelRest\Request\UserWs\Version1\UserService\ActivateConfirm
 */
class RequestTest extends WebTestCase
{
    /**
     * @var SerializerInterface
     */
    private $serializer;

    /**
     * @var ApiManager
     */
    private $apiManager;

    /**
     * @var FakerGenerator
     */
    private $faker;

    /**
     * @var ActivateConfirm\Request
     */
    private $request;

    protected function setUp()
    {
        parent::setUp();
        static::bootKernel();
        $this->apiManager = new ApiManager(static::$kernel->getContainer());
        $this->serializer = $this->apiManager->serializer();
        $this->faker = FakerFactory::create();

        $this->setUpRequest();
    }

    protected function setUpRequest()
    {
        $request = new ActivateConfirm\Request();
        $userType1 = new ActivateConfirm\UserType();
        $userType2 = new ActivateConfirm\UserType();

        $userType1
            ->setEmail($this->faker->email)
            ->setToken($this->faker->uuid);
        $userType2
            ->setEmail($this->faker->email)
            ->setToken($this->faker->uuid);

        $request
            ->addUserType($userType1)
            ->addUserType($userType2);

        $this->request = $request;
    }

    public function testGetAndAddRoleTypes()
    {
        $request = new ActivateConfirm\Request();
        $userType1 = new ActivateConfirm\UserType();
        $userType2 = new ActivateConfirm\UserType();

        $userType1Email = $this->faker->email;
        $userType1->setEmail($userType1Email);

        $userType2Email = $this->faker->email;
        $userType2->setEmail($userType2Email);

        $request->addUserType($userType1)
            ->addUserType($userType2);

        static::assertEquals($request->getUserTypes()->count(), 2);
        static::assertEquals($request->getUserTypes()->get(0)->getEmail(), $userType1Email);
        static::assertEquals($request->getUserTypes()->get(1)->getEmail(), $userType2Email);
    }

    public function testJSONSerialize()
    {
        static::assertJsonStringEqualsJsonString($this->serializer->serialize($this->request, 'json'),
            $this->getSerializedRequestContent('json'));
    }

    public function testXMLSerialize()
    {
        static::assertXmlStringEqualsXmlString($this->serializer->serialize($this->request, 'xml'),
            $this->getSerializedRequestContent('xml'));
    }

    /**
     * @param string $type (xml | json)
     * @return string
     */
    private function getSerializedRequestContent($type = 'xml')
    {
        switch ($type) {
            case 'json':
                return <<< JSON
{
	"userActivateConfirm": [
		{
			"email": "{$this->request->getUserTypes()->get(0)->getEmail()}",
			"token": "{$this->request->getUserTypes()->get(0)->getToken()}"
		},
		{
			"email": "{$this->request->getUserTypes()->get(1)->getEmail()}",
			"token": "{$this->request->getUserTypes()->get(1)->getToken()}"
		}
	]
}
JSON;
            case 'xml':
                return <<< XML
<userServiceActivateConfirm>
    <userActivateConfirm>
        <userType>
            <email>{$this->request->getUserTypes()->get(0)->getEmail()}</email>
            <token>{$this->request->getUserTypes()->get(0)->getToken()}</token>
        </userType>
        <userType>
            <email>{$this->request->getUserTypes()->get(1)->getEmail()}</email>
            <token>{$this->request->getUserTypes()->get(1)->getToken()}</token>
        </userType>
    </userActivateConfirm>
</userServiceActivateConfirm>
XML;
            default:
                throw new \PHPUnit_Framework_Exception('Invalid request type (hint: not json/xml)');
        }
    }
}