<?php

/**
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\RestBundle\Tests\ModelRest\Response\ObjectWs\Version1\Attribute\Update;

use Sso\RestBundle\Api;
use Sso\RestBundle\ModelRest\Response\ObjectWs\Version1\Attribute\Update\Response as RestResponse;
use Sso\RestBundle\ModelRest\Response\ObjectWs\Version1\Attribute\Update\Attribute;
use Sso\RestBundle\ModelRest\Common\Response\UserWs\ErrorCollector;
use Doctrine\Common\Collections\ArrayCollection;
use GuzzleHttp\Client;
use Symfony\Bundle\FrameworkBundle\Test\WebTestCase;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Faker\Factory as FakerFactory;
use Faker\Generator as FakerGenerator;

/**
 * ResponseTest
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */
class ResponseTest extends WebTestCase
{
    /**
     * @var Api\Manager
     */
    public $apiM;

    /**
     * @var Client
     */
    protected $client;

    /**
     * @var ContainerInterface
     */
    protected $container;

    /**
     * @var \JMS\Serializer\Serializer
     */
    protected $serializer;

    /**
     * @var FakerGenerator
     */
    private $faker;

    /**
     * @var string
     */
    private $trackId;

    /**
     * @var string
     */
    private $scriptTimeSec;

    /**
     * @var string
     */
    private $date;

    /**
     * @var string
     */
    private $message;

    /**
     * @var string
     */
    private $id;

    /**
     * @var string
     */
    private $name;

    /**
     * @var int
     */
    private $mandatory;

    /**
     * @var string
     */
    private $possibleValue;

    /**
     * @var string
     */
    private $defaultValue;

    /**
     * @var int
     */
    private $translatable;

    /**
     * @var string
     */
    private $groupId;

    protected function setUp()
    {
        $this->client = static::createClient();
        $this->container = $this->client->getContainer();
        $this->apiM = new Api\Manager($this->container);
        $this->serializer = $this->apiM->serializer();
        $this->faker = FakerFactory::create();

        // Set same random data
        $this->trackId = $this->faker->md5;
        $this->scriptTimeSec = $this->faker->randomFloat(2);
        $this->date = '2099-01-01T00:00:00+00:00';
        $this->message = $this->faker->sentence(2);
        $this->id = $this->faker->sha1;
        $this->name = $this->faker->name;
        $this->mandatory = $this->faker->boolean ? 1 : 0;
        $this->possibleValue = $this->faker->word;
        $this->defaultValue = $this->faker->word;
        $this->translatable = $this->faker->boolean ? 1 : 0;
        $this->groupId = $this->faker->sha1;
    }

    public function testResponseJSONModel()
    {
        $this->assertJsonStringEqualsJsonString(
            $this->getSuccessResponse('json'),
            $this->getResponseModel('json')
        );
    }

    public function testResponseXMLModel()
    {
        $this->assertXmlStringEqualsXmlString(
            $this->getSuccessResponse('xml'),
            $this->getResponseModel('xml')
        );
    }

    public function testGetterSuccess()
    {
        /** @var  RestResponse $responseModel */
        $responseModel = $this->serializer->deserialize(
            $this->getSuccessResponse('json'),
            RestResponse::class,
            'json'
        );

        $this->assertEquals(200, $responseModel->getCode());
        $this->assertEquals('OK', $responseModel->getStatus());
        $this->assertEquals($this->scriptTimeSec, $responseModel->getScriptTimeSec());
        $this->assertEquals($this->trackId, $responseModel->getTrackId());
        $this->assertEquals(new \DateTime($this->date), $responseModel->getDate());

        foreach ($responseModel->getAttributes() as $attribute) {
            $this->assertEquals(200, $attribute->getCode());
            $this->assertEquals('OK', $attribute->getStatus());
            $this->assertEquals($this->id, $attribute->getId());
            $this->assertEquals($this->name, $attribute->getName());
            $this->assertEquals($this->mandatory, $attribute->getMandatory());
            $this->assertEquals($this->possibleValue, $attribute->getPossibleValues()->get(0));
            $this->assertEquals($this->defaultValue, $attribute->getDefaultValue());
            $this->assertEquals($this->translatable, $attribute->getTranslatable());
            $this->assertEquals($this->groupId, $attribute->getGroupId());
            break;
        }
    }

    public function testSetterError()
    {
        $responseErrorModel = new RestResponse();
        $responseErrorModel
            ->setCode(400)
            ->setScriptTimeSec($this->scriptTimeSec)
            ->setDate(new \DateTime($this->date))
            ->setTrackId($this->trackId);

        $attribute = new Attribute();
        $attribute->setCode(400);

        $error = new ErrorCollector\Error();
        $error->setCode(400);
        $error->setMessage($this->message);
        $error->setException('external');

        $attribute->setError($error);
        $responseErrorModel->setAttribute($attribute);

        $this->assertJsonStringEqualsJsonString(
            $this->serializer->serialize($responseErrorModel, "json"),
            $this->getErrorResponse()
        );
    }

    public function testGetterError()
    {
        /** @var  RestResponse $responseModel */
        $responseModel = $this->serializer->deserialize(
            $this->getErrorResponse('json'),
            RestResponse::class,
            'json'
        );

        $this->assertEquals(400, $responseModel->getCode());
        $this->assertEquals('Bad Request', $responseModel->getStatus());
        $this->assertEquals($this->scriptTimeSec, $responseModel->getScriptTimeSec());
        $this->assertEquals($this->trackId, $responseModel->getTrackId());
        $this->assertEquals(new \DateTime($this->date), $responseModel->getDate());

        foreach ($responseModel->getAttributes() as $attribute) {
            $this->assertEquals(400, $attribute->getCode());
            $this->assertEquals('Bad Request', $attribute->getStatus());
            foreach ($attribute->getErrors() as $error) {
                $this->assertEquals(400, $error->getCode());
                $this->assertEquals($this->message, $error->getMessage());
                $this->assertEquals('external', $error->getException());
            }
            break;
        }
    }

    /**
     * @param string $type
     * @return string
     */
    private function getResponseModel($type = 'json')
    {
        $response = new RestResponse();
        $response
            ->setCode(200)
            ->setStatus('OK')
            ->setTrackId($this->trackId)
            ->setDate(new \DateTime($this->date))
            ->setScriptTimeSec($this->scriptTimeSec);

        $response->setAttribute($this->getAttribute());
        $responseModel = $this->serializer->serialize($response, $type);

        return $responseModel;
    }

    /**
     * @return Attribute
     */
    private function getAttribute()
    {
        $attribute = new Attribute();
        $attribute
            ->setCode(200)
            ->setStatus('OK');
        $attribute
            ->setId($this->id)
            ->setName($this->name)
            ->setMandatory($this->mandatory)
            ->setPossibleValues([$this->possibleValue])
            ->setDefaultValue($this->defaultValue)
            ->setTranslatable($this->translatable)
            ->setGroupId($this->groupId);

        return $attribute;
    }

    /**
     * @param string $type
     * @return string
     */
    private function getSuccessResponse($type = 'json')
    {
        if ($type === 'xml') {
            return '<?xml version="1.0" encoding="UTF-8"?>
                <attributeUpdate>
                  <code>200</code>
                  <status><![CDATA[OK]]></status>
                  <trackId><![CDATA[' . $this->trackId . ']]></trackId>
                  <date><![CDATA[' . $this->date . ']]></date>
                  <scriptTimeSec>' . $this->scriptTimeSec . '</scriptTimeSec>
                  <attributes>
                    <attribute>
                        <code>200</code>
                        <status><![CDATA[OK]]></status>
                        <id><![CDATA[' . $this->id . ']]></id>
                        <name><![CDATA[' . $this->name . ']]></name>
                        <mandatory>' . $this->mandatory . '</mandatory>
                        <possibleValues>
                            <value><![CDATA[' . $this->possibleValue . ']]></value>                        
                        </possibleValues>
                        <defaultValue><![CDATA[' . $this->defaultValue . ']]></defaultValue>
                        <translatable>' . $this->translatable . '</translatable>
                        <groupId><![CDATA[' . $this->groupId . ']]></groupId>
                    </attribute>
                  </attributes>
                </attributeUpdate>';
        } else {
            return '{
              "code": 200,
              "status": "OK",
              "trackId": "' . $this->trackId . '",
              "date": "' . $this->date . '",
              "scriptTimeSec": ' . $this->scriptTimeSec . ',
              "attributes": [
                {
                  "code": 200,
                  "status": "OK",
                  "id": "' . $this->id . '",
                  "name": "' . $this->name . '",
                  "mandatory": ' . $this->mandatory . ',
                  "possibleValues": [
                    "' . $this->possibleValue . '"
                  ],
                  "defaultValue": "' . $this->defaultValue . '",
                  "translatable": "' . $this->translatable . '",
                  "groupId": "' . $this->groupId . '"
                }
              ]
            }';
        }
    }

    /**
     * @param string $type
     * @return string
     */
    private function getErrorResponse($type = 'json')
    {
        if ($type === 'xml') {
            return '<?xml version="1.0" encoding="UTF-8"?>
                      <attributeUpdate>
                          <code>400</code>
                          <status><![CDATA[Bad Request]]></status>
                          <trackId><![CDATA[' . $this->trackId . ']]></trackId>
                          <date><![CDATA[' . $this->date . ']]></date>
                          <scriptTimeSec>' . $this->scriptTimeSec . '</scriptTimeSec>
                          <attributes>
                              <attribute>
                                  <code>400</code>
                                  <status><![CDATA[Bad Request]]></status>
                                  <errors>
                                      <error>
                                          <code>400</code>
                                          <message><![CDATA[' . $this->message . ']]></message>
                                          <exception><![CDATA[external]]></exception>
                                      </error>
                                  </errors>
                              </attribute>
                          </attributes>
                      </attributeUpdate>';
        } else {
            return '{
                      "code": 400,
                      "status": "Bad Request",
                      "trackId": "' . $this->trackId . '",
                      "date": "' . $this->date . '",
                      "scriptTimeSec": ' . $this->scriptTimeSec . ',
                      "attributes": [
                          {
                              "code": 400,
                              "status": "Bad Request",
                              "errors": [
                                  {
                                  "code": 400,
                                  "message": "' . $this->message . '",
                                  "exception": "external"
                                  }
                              ]
                          }
                      ]
                  }';
        }
    }
}
