<?php
/**
 * Class ResponseTest
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\RestBundle\Tests\ModelRest\Response\UserWs\Version1\UserApplicationRole\Add;

use Sso\RestBundle\Api\Manager as ApiManager;
use Sso\RestBundle\ModelRest\Common\Response\UserWs\ErrorCollector;
use Sso\RestBundle\ModelRest\Response\UserWs\Version1\UserApplicationRole\Add;
use Faker\Factory as FakerFactory;
use Faker\Generator as FakerGenerator;
use JMS\Serializer\Serializer;
use Symfony\Bundle\FrameworkBundle\Test\WebTestCase;

/**
 * Class ResponseTest
 * @package Sso\RestBundle\Tests\ModelRest\Response\UserWs\Version1\UserApplicationRole\Add
 */
class ResponseTest extends WebTestCase
{
    /**
     * @var Serializer
     */
    private $serializer;

    /**
     * @var ApiManager
     */
    private $apiManager;

    /**
     * @var FakerGenerator
     */
    private $faker;

    /**
     * @var Add\Response
     */
    private $response;

    protected function setUp()
    {
        parent::setUp();
        static::bootKernel();
        $this->apiManager = new ApiManager(static::$kernel->getContainer());
        $this->serializer = $this->apiManager->serializer();
        $this->faker = FakerFactory::create();

        $this->setUpResponse();
    }

    private function setUpResponse()
    {
        $response = new Add\Response();
        $roleType = new Add\RoleType();
        $error = new ErrorCollector\Error();

        $roleType->setCode($this->faker->numberBetween(100, 400))
            ->setStatus($this->faker->word)
            ->addError($error);

        $response
            ->setCode($this->faker->numberBetween(100, 400))
            ->setStatus($this->faker->word)
            ->setTrackId($this->faker->word)
            ->setDate($this->faker->dateTime)
            ->setScriptTimeSec(round($this->faker->randomFloat(), 3))
            ->addRoleType($roleType);

        $error
            ->setCode($this->faker->numberBetween(100, 400))
            ->setMessage($this->faker->text)
            ->setException($this->faker->word);

        $this->response = $response;
    }

    public function testGetAndAddRoleType()
    {
        $response = new Add\Response();
        $roleType = new Add\RoleType();

        $response->addRoleType($roleType);

        static::assertNotEmpty($response->getRoleTypes());
        static::assertEquals($response->getRoleTypes()->get(0), $roleType);
    }

    public function testJSONSerialize()
    {
        static::assertJsonStringEqualsJsonString($this->serializer->serialize($this->response, 'json'),
            $this->getSerializedResponseContent('json'));
    }

    public function testXMLSerialize()
    {
        static::assertXmlStringEqualsXmlString($this->serializer->serialize($this->response, 'xml'),
            $this->getSerializedResponseContent('xml'));
    }

    /**
     * @param string $type (xml | json)
     * @return string
     */
    private function getSerializedResponseContent($type = 'xml')
    {
        switch ($type) {
            case 'json':
                return <<< JSON
{
    "code": {$this->response->getCode()},
    "status": "{$this->response->getStatus()}",
    "trackId": "{$this->response->getTrackId()}",
    "date": "{$this->response->getDate()->format(\DateTime::ATOM)}",
    "scriptTimeSec": {$this->response->getScriptTimeSec()},
    "userRoles": [
        {
            "code": {$this->response->getRoleTypes()->get(0)->getCode()},
            "status": "{$this->response->getRoleTypes()->get(0)->getStatus()}",
            "errors": [
                {
                    "code": {$this->response->getRoleTypes()->get(0)->getErrors()->get(0)->getCode()},
                    "message": "{$this->response->getRoleTypes()->get(0)->getErrors()->get(0)->getMessage()}",
                    "exception": "{$this->response->getRoleTypes()->get(0)->getErrors()->get(0)->getException()}"
                }
            ]
        }
    ]
}
JSON;
            case 'xml':
                return <<< XML
<?xml version="1.0" encoding="UTF-8"?>
<userApplicationRoleAdd>
    <code>{$this->response->getCode()}</code>
    <status>
        <![CDATA[{$this->response->getStatus()}]]>
    </status>
    <trackId>
        <![CDATA[{$this->response->getTrackId()}]]>
    </trackId>
    <date>
        <![CDATA[{$this->response->getDate()->format(\DateTime::ATOM)}]]>
    </date>
    <scriptTimeSec>{$this->response->getScriptTimeSec()}</scriptTimeSec>
    <userRoles>
        <roleType>
            <code>{$this->response->getRoleTypes()->get(0)->getCode()}</code>
            <status>
                <![CDATA[{$this->response->getRoleTypes()->get(0)->getStatus()}]]>
            </status>
            <errors>
                <error>
                    <code>{$this->response->getRoleTypes()->get(0)->getErrors()->get(0)->getCode()}</code>
                    <message>
                        <![CDATA[{$this->response->getRoleTypes()->get(0)->getErrors()->get(0)->getMessage()}]]>
                    </message>
                    <exception>
                        <![CDATA[{$this->response->getRoleTypes()->get(0)->getErrors()->get(0)->getException()}]]>
                    </exception>
                </error>
            </errors>
        </roleType>
    </userRoles>
</userApplicationRoleAdd>
XML;
            default:
                throw new \PHPUnit_Framework_Exception('Invalid request type (hint: not json/xml)');
        }
    }
}
