<?php

/**
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\RestBundle\Tests\ModelRest\Response\UserWs\Version1\UserApplicationRole\Delete;

use Sso\RestBundle\Api\Manager as ApiManager;
use Sso\RestBundle\ModelRest\Common\Response\UserWs\ErrorCollector\Error;
use Sso\RestBundle\ModelRest\Response\UserWs\Version1\UserApplicationRole\Delete\Response;
use Sso\RestBundle\ModelRest\Response\UserWs\Version1\UserApplicationRole\Delete\UserApplicationRoleType;
use Faker\Factory as FakerFactory;
use Faker\Generator as FakerGenerator;
use JMS\Serializer\Serializer;
use Symfony\Bundle\FrameworkBundle\Test\WebTestCase;

/**
 * Class ResponseTest
 * @package Sso\RestBundle\Tests\ModelRest\Response\UserWs\Version1\UserApplication
 */
class ResponseTest extends WebTestCase
{
    /**
     * @var Serializer
     */
    private $serializer;

    /**
     * @var ApiManager
     */
    private $apiManager;

    /**
     * @var FakerGenerator
     */
    private $faker;

    /**
     * @var \stdClass
     */
    private $error;

    protected function setUp()
    {
        parent::setUp();
        static::bootKernel();
        $this->apiManager = new ApiManager(static::$kernel->getContainer());
        $this->serializer = $this->apiManager->serializer();
        $this->faker = FakerFactory::create();

        $this->error = new \stdClass();
        $this->error->code = 400;
        $this->error->status = 'Bad request';
        $this->error->trackId = 'b4cea79f2a';
        $this->error->date = new \DateTime('2016-11-09T09:16:04+01:00');
        $this->error->message = 'ErrorCode: u002 ErrorRef: ua033 ShortMessage: Invalid user application';
        $this->error->exception = 'external';
        $this->error->scriptTimeSec = 2.1917;
    }

    public function testAddUserApplicationRoleType()
    {
        $response = new Response();
        $userApplicationRoleType = new UserApplicationRoleType();

        $response->addUserApplicationRoleType($userApplicationRoleType);

        static::assertNotEmpty($response->getUserApplicationRoleType());
        static::assertEquals($response->getUserApplicationRoleType()->get(0), $userApplicationRoleType);
    }

    public function testJSONSerialize()
    {
        $response = new Response();
        $userApplicationRoleType = new UserApplicationRoleType();
        $error = new Error();

        $response
            ->setCode($this->error->code)
            ->setStatus($this->error->status)
            ->setTrackId($this->error->trackId)
            ->setDate($this->error->date)
            ->setScriptTimeSec($this->error->scriptTimeSec);

        $userApplicationRoleType
            ->setCode($this->error->code)
            ->setStatus($this->error->status);

        $error
            ->setCode($this->error->code)
            ->setMessage($this->error->message)
            ->setException($this->error->exception);

        $userApplicationRoleType->setError($error);
        $response->addUserApplicationRoleType($userApplicationRoleType);

        static::assertJsonStringEqualsJsonString($this->serializer->serialize($response, 'json'),
            $this->getSerializedResponseContent('json'));
    }

    public function testXMLSerialize()
    {
        $response = new Response();
        $userApplicationRoleType = new UserApplicationRoleType();
        $error = new Error();

        $response
            ->setCode($this->error->code)
            ->setStatus($this->error->status)
            ->setTrackId($this->error->trackId)
            ->setDate($this->error->date)
            ->setScriptTimeSec($this->error->scriptTimeSec);

        $userApplicationRoleType
            ->setCode($this->error->code)
            ->setStatus($this->error->status);

        $error
            ->setCode($this->error->code)
            ->setMessage($this->error->message)
            ->setException($this->error->exception);

        $userApplicationRoleType->setError($error);
        $response->addUserApplicationRoleType($userApplicationRoleType);

        static::assertXmlStringEqualsXmlString($this->serializer->serialize($response, 'xml'),
            $this->getSerializedResponseContent('xml'));
    }

    /**
     * @param string $type (xml | json)
     * @return string
     */
    private function getSerializedResponseContent($type = 'xml')
    {
        switch ($type) {
            case 'json':
                return <<< JSON
                    {
                        "code": {$this->error->code},
                        "status": "{$this->error->status}",
                        "trackId": "{$this->error->trackId}",
                        "date": "{$this->error->date->format(\DateTime::ATOM)}",
                        "scriptTimeSec": {$this->error->scriptTimeSec},
                        "userApplicationRoles": [
                            {
                                "code": {$this->error->code},
                                "status": "{$this->error->status}",
                                "errors": [
                                    {
                                        "code": {$this->error->code},
                                        "message": "{$this->error->message}",
                                        "exception": "{$this->error->exception}"
                                    }
                                ]
                            }
                        ]
                    }
JSON;
            case 'xml':
                return <<< XML
<?xml version="1.0" encoding="UTF-8"?>
<userApplicationRoleDelete>
    <code>{$this->error->code}</code>
    <status>
        <![CDATA[{$this->error->status}]]>
    </status>
    <trackId>
        <![CDATA[{$this->error->trackId}]]>
    </trackId>
    <date>
        <![CDATA[{$this->error->date->format(\DateTime::ATOM)}]]>
    </date>
    <scriptTimeSec>{$this->error->scriptTimeSec}</scriptTimeSec>
    <userApplicationRoles>
        <userApplicationRoleType>
         <code>{$this->error->code}</code>
         <status>
             <![CDATA[{$this->error->status}]]>
         </status>
         <errors>
             <error>
                 <code>{$this->error->code}</code>
                 <message>
                     <![CDATA[{$this->error->message}]]>
                 </message>
                 <exception>
                     <![CDATA[{$this->error->exception}]]>
                 </exception>
             </error>
         </errors>
        </userApplicationRoleType>
    </userApplicationRoles>
</userApplicationRoleDelete>
XML;
            default:
                throw new \PHPUnit_Framework_Exception('Invalid request type (hint: not json/xml)');
        }
    }
}
