<?php

/**
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\RestBundle\Tests\ModelRest\Response\UserWs\Version2\UserSearch\Advanced;

use Sso\RestBundle\Api;
use Sso\RestBundle\ModelRest\Response\UserWs\Version2\UserSearch\Advanced\Response as RestResponse;
use Sso\RestBundle\ModelShared\Response\UserWs\Version2\UserSearch\Advanced\Success\ApplicationRoleType;
use Sso\RestBundle\ModelShared\Response\UserWs\Version2\UserSearch\Advanced\Success\ApplicationType;
use Sso\RestBundle\ModelShared\Response\UserWs\Version2\UserSearch\Advanced\Success\AttributeType;
use Sso\RestBundle\ModelShared\Response\UserWs\Version2\UserSearch\Advanced\Success\User;
use Sso\RestBundle\ModelShared\Response\UserWs\Version2\UserSearch\Advanced\Success\UserType;
use Doctrine\Common\Collections\ArrayCollection;
use GuzzleHttp\Client;
use Symfony\Bundle\FrameworkBundle\Test\WebTestCase;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Sso\RestBundle\ModelRest\Common\Response\UserWs\ErrorCollector;
use Faker\Factory as FakerFactory;
use Faker\Generator as FakerGenerator;

/**
 * ResponseTest
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */
class ResponseTest extends WebTestCase
{
    /**
     * @var Api\Manager
     */
    public $apiM;

    /**
     * @var Client
     */
    protected $client;

    /**
     * @var ContainerInterface
     */
    protected $container;

    /**
     * @var \JMS\Serializer\Serializer
     */
    protected $serializer;

    /**
     * @var FakerGenerator
     */
    private $faker;

    /**
     * @var string
     */
    private $trackId;

    /**
     * @var string
     */
    private $scriptTimeSec;

    /**
     * @var string
     */
    private $date;

    /**
     * @var string
     */
    private $name;

    /**
     * @var string
     */
    private $description;

    /**
     * @var string
     */
    private $message;

    /**
     * @var string
     */
    private $guid;

    /**
     * @var string
     */
    private $email;

    /**
     * @var string
     */
    private $username;

    /**
     * @var string
     */
    private $firstName;

    /**
     * @var string
     */
    private $lastName;

    /**
     * @var boolean
     */
    private $active;

    /**
     * @var boolean
     */
    private $deleted;

    /**
     * @var string
     */
    private $authId;

    /**
     * @var integer
     */
    private $mfaEnabled;

    /**
     * @var integer
     */
    private $count;

    /**
     * @var integer
     */
    private $offset;

    /**
     * @var integer
     */
    private $limit;

    /**
     * @var integer
     */
    private $fullResponse;

    /**
     * @var string
     */
    private $attrValue;

    protected function setUp()
    {
        $this->client = static::createClient();
        $this->container = $this->client->getContainer();
        $this->apiM = new Api\Manager($this->container);
        $this->serializer = $this->apiM->serializer();
        $this->faker = FakerFactory::create();

        // Set same random data
        $this->trackId = $this->faker->md5;
        $this->date = '2099-01-01T00:00:00+00:00';
        $this->scriptTimeSec = $this->faker->randomFloat(2);
        $this->count = $this->faker->numberBetween(1, 10);
        $this->offset = $this->faker->numberBetween(0, 10);
        $this->limit = $this->faker->numberBetween(0, 1000);
        $this->fullResponse = $this->faker->boolean ? 1 : 0;
        $this->name = $this->faker->word;
        $this->description = $this->faker->sentence();
        $this->message = $this->faker->sentence();
        $this->guid = $this->faker->uuid;
        $this->username = $this->faker->userName;
        $this->email = $this->faker->email;
        $this->firstName = $this->faker->firstName;
        $this->lastName = $this->faker->lastName;
        $this->active = $this->faker->boolean ? 1 : 0;
        $this->deleted = $this->faker->boolean ? 1 : 0;
        $this->authId = $this->faker->md5;
        $this->mfaEnabled = $this->faker->boolean ? 1 : 0;
        $this->attrValue = $this->faker->sentence(2);
    }

    public function testResponseJSONModel()
    {
        $this->assertJsonStringEqualsJsonString(
            $this->getSuccessResponse('json'),
            $this->getResponseModel('json')
        );
    }

    public function testResponseXMLModel()
    {
        $this->assertXmlStringEqualsXmlString(
            $this->getSuccessResponse('xml'),
            $this->getResponseModel('xml')
        );
    }

    public function testGetterSuccess()
    {
        /** @var  RestResponse $responseModel */
        $responseModel = $this->serializer->deserialize(
            $this->getSuccessResponse('json'),
            RestResponse::class,
            'json'
        );

        $this->assertEquals(200, $responseModel->getCode());
        $this->assertEquals('OK', $responseModel->getStatus());
        $this->assertEquals($this->scriptTimeSec, $responseModel->getScriptTimeSec());
        $this->assertEquals($this->trackId, $responseModel->getTrackId());
        $this->assertEquals(new \DateTime($this->date), $responseModel->getDate());
        $this->assertEquals($this->count, $responseModel->getCount());
        $this->assertEquals($this->offset, $responseModel->getOffset());
        $this->assertEquals($this->limit, $responseModel->getLimit());
        $this->assertEquals($this->fullResponse, $responseModel->getFullResponse());

        $userType = $responseModel->getUsers()[0]->getUserType();
        $applicationType = $responseModel->getUsers()[0]->getApplicationType()[0];
        $attributeType = $applicationType->getAttributeType()[0];
        $applicationRoleType = $applicationType->getApplicationRoleType()[0];

        $this->assertEquals($this->guid, $userType->getGuid());
        $this->assertEquals($this->username, $userType->getUsername());
        $this->assertEquals($this->email, $userType->getEmail());
        $this->assertEquals($this->firstName, $userType->getFirstname());
        $this->assertEquals($this->lastName, $userType->getLastname());
        $this->assertEquals($this->active, $userType->isActive() ? 1 : 0);
        $this->assertEquals($this->deleted, $userType->isDeleted() ? 1 : 0);
        $this->assertEquals(new \DateTime($this->date), $userType->getCreatedAt());
        $this->assertEquals(new \DateTime($this->date), $userType->getUpdatedAt());
        $this->assertEquals(new \DateTime($this->date), $userType->getLastLoginAt());
        $this->assertEquals($this->authId, $userType->getAuthId());
        $this->assertEquals('', $userType->getLdapSearchAttributes());
        $this->assertEquals('', $userType->getLdapSearchValue());
        $this->assertEquals($this->mfaEnabled, $userType->isMfaEnabled() ? 1 : 0);
        $this->assertEquals($this->mfaEnabled, $userType->getMfaEnabledAsInteger());

        $this->assertEquals($this->name, $applicationType->getName());
        $this->assertEquals($this->active, $applicationType->isActive() ? 1 : 0);

        $this->assertEquals('one', $attributeType->getType());
        $this->assertEquals($this->name, $attributeType->getName());
        $this->assertEquals($this->attrValue, $attributeType->getValue());

        $this->assertEquals($this->name, $applicationRoleType->getName());
        $this->assertEquals($this->description, $applicationRoleType->getDescription());
        $this->assertEquals($this->active, $applicationRoleType->getActive());
    }

    public function testSetterError()
    {
        $responseErrorModel = new RestResponse();
        $responseErrorModel
            ->setCode(400)
            ->setStatus('Bad Request')
            ->setTrackId($this->trackId)
            ->setDate(new \DateTime($this->date))
            ->setScriptTimeSec($this->scriptTimeSec);

        $error = new ErrorCollector\Error();
        $error
            ->setCode(400)
            ->setMessage($this->message)
            ->setException('external');

        $responseErrorModel->setError($error);

        $this->assertJsonStringEqualsJsonString(
            $this->serializer->serialize($responseErrorModel, "json"),
            $this->getErrorResponse()
        );
    }

    public function testGetterError()
    {
        /** @var  RestResponse $responseModel */
        $responseModel = $this->serializer->deserialize(
            $this->getErrorResponse('json'),
            RestResponse::class,
            'json'
        );

        $this->assertEquals(400, $responseModel->getCode());
        $this->assertEquals('Bad Request', $responseModel->getStatus());
        $this->assertEquals($this->trackId, $responseModel->getTrackId());
        $this->assertEquals(new \DateTime($this->date), $responseModel->getDate());
        $this->assertEquals($this->scriptTimeSec, $responseModel->getScriptTimeSec());

        foreach ($responseModel->getError() as $error) {
            $this->assertEquals(400, $error->getCode());
            $this->assertEquals($this->message, $error->getMessage());
            $this->assertEquals('external', $error->getException());

            break;
        }
    }

    /**
     * @param string $type
     * @return string
     */
    private function getResponseModel($type = 'json')
    {
        $response = new RestResponse();

        $response
            ->setCode(200)
            ->setStatus('OK')
            ->setTrackId($this->trackId)
            ->setDate(new \DateTime($this->date))
            ->setScriptTimeSec($this->scriptTimeSec);

        $response
            ->setCount($this->count)
            ->setOffset($this->offset)
            ->setLimit($this->limit)
            ->setFullResponse($this->fullResponse);

        $user = new ArrayCollection();
        $user->add($this->getUser());

        $response->addUser($this->getUser());
        $response->setUser($user);
        $responseModel = $this->serializer->serialize($response, $type);

        return $responseModel;
    }

    /**
     * @return User
     */
    private function getUser()
    {
        $user = new User();
        $userType = new UserType();
        $applicationType = new ApplicationType();
        $attributeType = new AttributeType();
        $applicationRoleType = new ApplicationRoleType();

        $userType
            ->setGuid($this->guid)
            ->setUsername($this->username)
            ->setEmail($this->email)
            ->setFirstname($this->firstName)
            ->setLastname($this->lastName)
            ->setActive($this->active === 1)
            ->setDeleted($this->deleted === 1)
            ->setCreatedAt(new \DateTime($this->date))
            ->setUpdatedAt(new \DateTime($this->date))
            ->setLastLoginAt(new \DateTime($this->date))
            ->setAuthId($this->authId)
            ->setLdapSearchAttributes('')
            ->setLdapSearchValue('')
            ->setMfaEnabled($this->mfaEnabled === 1);

        $attributeType
            ->setType('one')
            ->setName($this->name)
            ->setValue($this->attrValue);

        $applicationRoleType
            ->setName($this->name)
            ->setDescription($this->description)
            ->setActive($this->active);

        $applicationType
            ->setName($this->name)
            ->setActive($this->active)
            ->addAttributeType($attributeType)
            ->addApplicationRoleType($applicationRoleType);

        $user
            ->setUserType($userType)
            ->addApplicationType($applicationType);

        return $user;
    }

    /**
     * @param string $type
     * @return string
     */
    private function getSuccessResponse($type = 'json')
    {
        if ($type === 'xml') {
            return '<?xml version="1.0" encoding="UTF-8"?>
                <userSearchAdvanced>
                  <code>200</code>
                  <status><![CDATA[OK]]></status>
                  <trackId><![CDATA[' . $this->trackId . ']]></trackId>
                  <date><![CDATA[' . $this->date . ']]></date>
                  <scriptTimeSec>' . $this->scriptTimeSec . '</scriptTimeSec>
                  <count>' . $this->count . '</count>
                  <offset>' . $this->offset . '</offset>
                  <limit>' . $this->limit . '</limit>
                  <fullResponse>' . $this->fullResponse . '</fullResponse>
                  <users>
                    <UserType>
                        <Guid><![CDATA[' . $this->guid . ']]></Guid>
                        <Username><![CDATA[' . $this->username . ']]></Username>
                        <Email><![CDATA[' . $this->email . ']]></Email>
                        <Firstname><![CDATA[' . $this->firstName . ']]></Firstname>
                        <Lastname><![CDATA[' . $this->lastName . ']]></Lastname>
                        <Active>' . $this->active . '</Active>
                        <Deleted>' . $this->deleted . '</Deleted>
                        <CreatedAt><![CDATA[' . $this->date . ']]></CreatedAt>
                        <UpdatedAt><![CDATA[' . $this->date . ']]></UpdatedAt>
                        <LastLoginAt><![CDATA[' . $this->date . ']]></LastLoginAt>
                        <AuthId><![CDATA[' . $this->authId . ']]></AuthId>
                        <LdapSearchAttributes><![CDATA[]]></LdapSearchAttributes>
                        <LdapSearchValue><![CDATA[]]></LdapSearchValue>
                        <MfaEnabled>' . $this->mfaEnabled . '</MfaEnabled>
                    </UserType>
                    <ApplicationType>
                        <Name><![CDATA[' . $this->name . ']]></Name>
                        <Active>' . $this->active . '</Active>
                        <AttributeType>
                             <Type><![CDATA[one]]></Type>
                             <Name><![CDATA[' . $this->name . ']]></Name>
                             <Value><![CDATA[' . $this->attrValue . ']]></Value>
                        </AttributeType>
                        <ApplicationRoleType>
                             <Name><![CDATA[' . $this->name . ']]></Name>
                             <Description><![CDATA[' . $this->description . ']]></Description>
                             <Active>' . $this->active . '</Active>
                        </ApplicationRoleType>
                    </ApplicationType>
                  </users>
                </userSearchAdvanced>';
        } else {
            return '{
              "code": 200,
              "status": "OK",
              "trackId": "' . $this->trackId . '",
              "date": "' . $this->date . '",
              "scriptTimeSec": ' . $this->scriptTimeSec . ',
              "count": ' . $this->count . ',
              "offset": ' . $this->offset . ',
              "limit": ' . $this->limit . ',
              "fullResponse": ' . $this->fullResponse . ',
              "users": [
                  {
                    "UserType": {
                      "Guid": "' . $this->guid . '",
                      "Username": "' . $this->username . '",
                      "Email": "' . $this->email . '",
                      "Firstname": "' . $this->firstName . '",
                      "Lastname": "' . $this->lastName . '",
                      "Active": ' . $this->active . ',
                      "Deleted": ' . $this->deleted . ',
                      "CreatedAt": "' . $this->date . '",
                      "UpdatedAt": "' . $this->date . '",
                      "LastLoginAt": "' . $this->date . '",
                      "AuthId": "' . $this->authId . '",
                      "LdapSearchAttributes": "",
                      "LdapSearchValue": "",
                      "MfaEnabled": ' . $this->mfaEnabled . '
                    },
                    "ApplicationType": [
                        {
                            "Name": "' . $this->name . '",
                            "Active": ' . $this->active . ',
                            "AttributeType": [
                                {
                                    "Type": "one",
                                    "Name": "' . $this->name . '",
                                    "Value": "' . $this->attrValue . '"
                                }
                            ],
                            "ApplicationRoleType": [
                                {
                                    "Name": "' . $this->name . '",
                                    "Description": "' . $this->description . '",
                                    "Active": ' . $this->active . '
                                }
                            ]
                        }
                    ]
                  }
              ]
            }';
        }
    }

    /**
     * @param string $type
     * @return string
     */
    private function getErrorResponse($type = 'json')
    {
        if ($type === 'xml') {
            return '<?xml version="1.0" encoding="UTF-8"?>
                      <userSearchAdvanced>
                          <code>400</code>
                          <status><![CDATA[Bad Request]]></status>
                          <trackId><![CDATA[' . $this->trackId . ']]></trackId>
                          <date><![CDATA[' . $this->date . ']]></date>
                          <scriptTimeSec>' . $this->scriptTimeSec . '</scriptTimeSec>
                          <errors>
                              <error>
                                  <code>400</code>
                                  <message><![CDATA[' . $this->message . ']]></message>
                                  <exception><![CDATA[external]]></exception>
                              </error>
                          </error>
                      </userSearchAdvanced>';
        } else {
            return '{
                      "code": 400,
                      "status": "Bad Request",
                      "trackId": "' . $this->trackId . '",
                      "date": "' . $this->date . '",
                      "scriptTimeSec": ' . $this->scriptTimeSec . ',
                      "errors": [
                          {
                              "code": 400,
                              "message": "' . $this->message . '",
                              "exception": "external"
                          }
                      ]
                  }';
        }
    }
}
