<?php

/**
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\RestBundle\Tests\ModelXml\Request\ObjectWs\Version1\Attribute\Update;

use Sso\RestBundle\ModelXml\Common\Request\Credentials\ObjectWs\Credentials;
use Sso\RestBundle\ModelXml\Request\ObjectWs\Version1\Attribute\Update\Request;
use Sso\RestBundle\ModelXml\Request\ObjectWs\Version1\Attribute\Update\Attribute;
use Doctrine\Common\Collections\ArrayCollection;
use Symfony\Bundle\FrameworkBundle\Test\WebTestCase;
use Sso\RestBundle\Api\Manager as ApiManager;
use JMS\Serializer\SerializerInterface;
use Faker\Factory as FakerFactory;
use Faker\Generator as FakerGenerator;

/**
 * Class RequestTest
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package Sso\RestBundle\Tests\ModelRest
 */
class RequestTest extends WebTestCase
{
    /**
     * @var SerializerInterface
     */
    private $serializer;

    /**
     * @var ApiManager
     */
    private $apiManager;

    /**
     * @var FakerGenerator
     */
    private $faker;

    /**
     * @var string
     */
    private $serviceToken;

    /**
     * @var string
     */
    private $serviceName;

    /**
     * @var string
     */
    private $userIdentifier;

    /**
     * @var string
     */
    private $username;

    /**
     * @var string
     */
    private $langCode;

    /**
     * @var string
     */
    private $id;

    /**
     * @var string
     */
    private $name;

    /**
     * @var int
     */
    private $mandatory;

    /**
     * @var string
     */
    private $possibleValue;

    /**
     * @var string
     */
    private $defaultValue;

    /**
     * @var int
     */
    private $translatable;

    /**
     * @var string
     */
    private $groupId;

    protected function setUp()
    {
        parent::setUp();
        static::bootKernel();
        $this->apiManager = new ApiManager(static::$kernel->getContainer());
        $this->serializer = $this->apiManager->serializer();
        $this->faker = FakerFactory::create();

        // Set same random data
        $this->serviceToken = $this->faker->sha256;
        $this->serviceName = $this->faker->word;
        $this->userIdentifier = $this->faker->md5;
        $this->username = $this->faker->userName;
        $this->langCode = $this->faker->languageCode;
        $this->id = $this->faker->sha1;
        $this->name = $this->faker->name;
        $this->mandatory = $this->faker->boolean ? 1 : 0;
        $this->possibleValue = $this->faker->word;
        $this->defaultValue = $this->faker->word;
        $this->translatable = $this->faker->boolean ? 1 : 0;
        $this->groupId = $this->faker->sha1;
    }

    public function testSerialize()
    {
        $this->assertXmlStringEqualsXmlString($this->getRequestBody(), $this->setRequestBody());
    }

    /**
     * @return string
     */
    private function setRequestBody()
    {
        $request = new Request();
        $credentials = new Credentials();
        $attribute = new Attribute();

        $credentials->setCredentialsData(
            $this->serviceToken,
            $this->serviceName,
            $this->userIdentifier,
            $this->username,
            $this->langCode
        );

        $attribute->setAttribute(
            $this->id,
            $this->name,
            $this->mandatory,
            new ArrayCollection([$this->possibleValue]),
            $this->defaultValue,
            $this->translatable,
            $this->groupId
        );

        $request->setCredentials($credentials);
        $request->setAttribute($attribute);

        return $this->serializer->serialize($request, 'xml');
    }

    /**
     * @return string
     */
    private function getRequestBody()
    {
        return '<SsoRequest>
            <Credentials>
                <ServiceProvider>
                    <ServiceToken>' . $this->serviceToken . '</ServiceToken>
                    <ServiceName>' . $this->serviceName . '</ServiceName>
                </ServiceProvider>
                <ServiceTrigger>
                    <UserIdentifier>' . $this->userIdentifier . '</UserIdentifier>
                    <Username>' . $this->username . '</Username>
                </ServiceTrigger>
                <ServiceCountry>
                    <Code>' . $this->langCode . '</Code>
                </ServiceCountry>
            </Credentials>
            <Attribute>
                <Update>
                    <AttributeType>
                        <Id>' . $this->id . '</Id>
                        <Name>' . $this->name . '</Name>
                        <Mandatory>' . $this->mandatory . '</Mandatory>
                        <PossibleValues>
                            <Value><![CDATA[' . $this->possibleValue . ']]></Value>                        
                        </PossibleValues>
                        <DefaultValue><![CDATA[' . $this->defaultValue . ']]></DefaultValue>
                        <Translatable>' . $this->translatable . '</Translatable>
                        <GroupId>' . $this->groupId . '</GroupId>
                    </AttributeType>
                </Update>
            </Attribute>
        </SsoRequest>';
    }

    public function testGetters()
    {
        /** @var  Request $requestModel */
        $requestModel = $this->serializer->deserialize(
            $this->setRequestBody(),
            Request::class,
            'xml'
        );

        $attributeType = $requestModel->getAttribute()->getUpdate()->getAttributeType();

        $credentials = $requestModel->getCredentials();

        $this->assertEquals($this->id, $attributeType->getId());
        $this->assertEquals($this->name, $attributeType->getName());
        $this->assertEquals($this->mandatory, $attributeType->getMandatory());
        $this->assertEquals($this->possibleValue, $attributeType->getPossibleValues()->get(0));
        $this->assertEquals($this->defaultValue, $attributeType->getDefaultValue());
        $this->assertEquals($this->translatable, $attributeType->getTranslatable());
        $this->assertEquals($this->groupId, $attributeType->getGroupId());
        $this->assertEquals($this->serviceToken, $credentials->getServiceProvider()->getServiceToken());
    }
}
