<?php

/**
 * Class RequestTest
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\RestBundle\Tests\ModelXml\Request\ObjectWs\Version1\AttributeGroup\Update;

use Sso\RestBundle\Api\Manager as ApiManager;
use Sso\RestBundle\ModelXml\Common\Request\Credentials\ObjectWs\Credentials;
use Sso\RestBundle\ModelXml\Request\ObjectWs\Version1\AttributeGroup\Update;
use Doctrine\Common\Collections\ArrayCollection;
use Faker\Factory as FakerFactory;
use Faker\Generator as FakerGenerator;
use JMS\Serializer\SerializerInterface;
use Symfony\Bundle\FrameworkBundle\Test\WebTestCase;

/**
 * Class RequestTest
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package Sso\RestBundle\Tests\ModelXml\Request\ObjectWs\Version1\AttributeGroup\Update
 */
class RequestTest extends WebTestCase
{
    /**
     * @var SerializerInterface
     */
    private $serializer;

    /**
     * @var ApiManager
     */
    private $apiManager;

    /**
     * @var FakerGenerator
     */
    private $faker;

    /**
     * @var Update\Request
     */
    private $request;

    protected function setUp()
    {
        parent::setUp();
        static::bootKernel();
        $this->apiManager = new ApiManager(static::$kernel->getContainer());
        $this->serializer = $this->apiManager->serializer();
        $this->faker = FakerFactory::create();

        $this->setUpRequest();
    }

    private function setUpRequest()
    {
        $request = new Update\Request();
        $credentials = new Credentials();
        $attributeGroup = new Update\AttributeGroup();

        $addTypeIds = new ArrayCollection([$this->faker->uuid, $this->faker->uuid]);
        $deleteTypeIds = new ArrayCollection([$this->faker->uuid, $this->faker->uuid]);

        $attributeGroup->setAttributeGroupData($this->faker->uuid, $this->faker->name, $addTypeIds, $deleteTypeIds);
        $request->setAttributeGroup($attributeGroup);
        $credentials->setCredentialsData($this->faker->uuid, $this->faker->name, $this->faker->userName,
            $this->faker->userName, $this->faker->text);
        $request->setCredentials($credentials)
            ->setAttributeGroup($attributeGroup);

        $this->request = $request;
    }

    public function testGetAndSetCredentials()
    {
        $request = new Update\Request();
        $credentials = new Credentials();

        $request->setCredentials($credentials);

        static::assertEquals($request->getCredentials(), $credentials);
    }

    public function testGetAndSetAttributeGroup()
    {
        $request = new Update\Request();
        $attributeGroup = new Update\AttributeGroup();

        $request->setAttributeGroup($attributeGroup);

        static::assertEquals($request->getAttributeGroup(), $attributeGroup);
    }

    public function testXMLSerialize()
    {
        static::assertXmlStringEqualsXmlString($this->serializer->serialize($this->request, 'xml'),
            $this->getExpectedResponseContent());
    }

    /**
     * @return string
     */
    private function getExpectedResponseContent()
    {
        return <<< XML
<?xml version="1.0"?>
<SsoRequest>
    <Credentials>
        <ServiceProvider>
            <ServiceToken><![CDATA[{$this->request->getCredentials()->getServiceProvider()->getServiceToken()}]]></ServiceToken>
            <ServiceName><![CDATA[{$this->request->getCredentials()->getServiceProvider()->getServiceName()}]]></ServiceName>
        </ServiceProvider>
        <ServiceTrigger>
            <UserIdentifier><![CDATA[{$this->request->getCredentials()->getServiceTrigger()->getUserIdentifier()}]]></UserIdentifier>
            <Username><![CDATA[{$this->request->getCredentials()->getServiceTrigger()->getUsername()}]]></Username>
        </ServiceTrigger>
        <ServiceCountry>
            <Code>{$this->request->getCredentials()->getServiceCountry()->getCode()}</Code>
        </ServiceCountry>
    </Credentials>
    <AttributeGroup>
        <Update>
            <AttributeGroupType>
                <Id><![CDATA[{$this->request->getAttributeGroup()->getUpdate()->getAttributeGroupType()->getId()}]]></Id>
                <Name><![CDATA[{$this->request->getAttributeGroup()->getUpdate()->getAttributeGroupType()->getName()}]]></Name>
                <AddTypes>
                    <TypeId><![CDATA[{$this->request->getAttributeGroup()->getUpdate()->getAttributeGroupType()->getAddType()->getTypeIds()->get(0)}]]></TypeId>
                    <TypeId><![CDATA[{$this->request->getAttributeGroup()->getUpdate()->getAttributeGroupType()->getAddType()->getTypeIds()->get(1)}]]></TypeId>
                </AddTypes>
                <DeleteTypes>
                    <TypeId><![CDATA[{$this->request->getAttributeGroup()->getUpdate()->getAttributeGroupType()->getDeleteType()->getTypeIds()->get(0)}]]></TypeId>
                    <TypeId><![CDATA[{$this->request->getAttributeGroup()->getUpdate()->getAttributeGroupType()->getDeleteType()->getTypeIds()->get(1)}]]></TypeId>
                </DeleteTypes>
            </AttributeGroupType>
        </Update>
    </AttributeGroup>
</SsoRequest>       
XML;
    }
}