<?php

/**
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\RestBundle\Tests\ModelXml\Request\UserWs\Version1\ApplicationAttribute\Show;

use Sso\RestBundle\ModelXml\Common\Request\Credentials\UserWs\Credentials;
use Sso\RestBundle\ModelXml\Request\UserWs\Version1\ApplicationAttribute\Show\Request;
use Sso\RestBundle\ModelXml\Request\UserWs\Version1\ApplicationAttribute\Show\ApplicationAttribute;
use Symfony\Bundle\FrameworkBundle\Test\WebTestCase;
use Sso\RestBundle\Api\Manager as ApiManager;
use JMS\Serializer\SerializerInterface;
use Faker\Factory as FakerFactory;
use Faker\Generator as FakerGenerator;

/**
 * Class RequestTest
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package Sso\RestBundle\Tests\ModelRest
 */
class RequestTest extends WebTestCase
{
    /**
     * @var SerializerInterface
     */
    private $serializer;

    /**
     * @var ApiManager
     */
    private $apiManager;

    /**
     * @var FakerGenerator
     */
    private $faker;

    /**
     * @var string
     */
    private $serviceToken;

    /**
     * @var string
     */
    private $serviceName;

    /**
     * @var string
     */
    private $userIdentifier;

    /**
     * @var string
     */
    private $username;

    /**
     * @var string
     */
    private $langCode;

    /**
     * @var string
     */
    private $attributeName;

    /**
     * @var string
     */
    private $applicationName;

    protected function setUp()
    {
        parent::setUp();
        static::bootKernel();
        $this->apiManager = new ApiManager(static::$kernel->getContainer());
        $this->serializer = $this->apiManager->serializer();
        $this->faker = FakerFactory::create();

        // Set same random data
        $this->serviceToken = $this->faker->sha256;
        $this->serviceName = $this->faker->word;
        $this->userIdentifier = $this->faker->md5;
        $this->username = $this->faker->userName;
        $this->langCode = $this->faker->languageCode;
        $this->attributeName = $this->faker->word;
        $this->applicationName = $this->faker->word;
    }

    public function testSerialize()
    {
        $this->assertXmlStringEqualsXmlString($this->getRequestBody(), $this->setRequestBody());
    }

    /**
     * @return string
     */
    private function setRequestBody()
    {
        $request = new Request();
        $credentials = new Credentials();
        $applicationAttribute = new ApplicationAttribute();

        $credentials->setCredentialsData(
            $this->serviceToken,
            $this->serviceName,
            $this->userIdentifier,
            $this->username,
            $this->langCode
        );

        $applicationAttribute->setApplicationAttribute(
            $this->applicationName,
            $this->attributeName
        );

        $request->setCredentials($credentials);
        $request->setApplicationAttribute($applicationAttribute);

        return $this->serializer->serialize($request, 'xml');
    }

    /**
     * @return string
     */
    private function getRequestBody()
    {
        return '<SsoRequest>
            <Credentials>
                <ServiceProvider>
                    <ServiceToken>' . $this->serviceToken . '</ServiceToken>
                    <ServiceName>' . $this->serviceName . '</ServiceName>
                </ServiceProvider>
                <ServiceTrigger>
                    <UserIdentifier>' . $this->userIdentifier . '</UserIdentifier>
                    <Username>' . $this->username . '</Username>
                </ServiceTrigger>
                <ServiceCountry>
                    <Code>' . $this->langCode . '</Code>
                </ServiceCountry>
            </Credentials>
            <ApplicationAttribute>
                <Show>
                    <Key>
                        <ApplicationType>
                            <Name>' . $this->applicationName . '</Name>
                        </ApplicationType>
                        <AttributeType>
                            <Name>' . $this->attributeName . '</Name>
                        </AttributeType>
                    </Key>
                </Show>
            </ApplicationAttribute>
        </SsoRequest>';
    }

    public function testGetterUserType()
    {
        /** @var  Request $requestModel */
        $requestModel = $this->serializer->deserialize(
            $this->setRequestBody(),
            Request::class,
            'xml'
        );

        $applicationAttribute = $requestModel->getApplicationAttribute()->getShow();
        $applicationType = $applicationAttribute->getKey()->getApplicationType();
        $attributeType = $applicationAttribute->getKey()->getAttributeType();

        $credentials = $requestModel->getCredentials();

        $this->assertEquals($this->applicationName, $applicationType->getName());
        $this->assertEquals($this->attributeName, $attributeType->getName());
        $this->assertEquals($this->serviceToken, $credentials->getServiceProvider()->getServiceToken());
    }
}
