<?php

/**
 * Class RequestTest
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\RestBundle\Tests\ModelXml\Request\UserWs\Version1\UserApplication\Delete;

use Sso\RestBundle\Api\Manager as ApiManager;
use Sso\RestBundle\ModelXml\Common\Request\Credentials\UserWs\Credentials;
use Sso\RestBundle\ModelXml\Request\UserWs\Version1\UserApplication\Delete;
use Faker\Factory as FakerFactory;
use Faker\Generator as FakerGenerator;
use JMS\Serializer\SerializerInterface;
use Symfony\Bundle\FrameworkBundle\Test\WebTestCase;

/**
 * Class RequestTest
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package Sso\RestBundle\Tests\ModelRest
 */
class RequestTest extends WebTestCase
{
    /**
     * @var SerializerInterface
     */
    private $serializer;

    /**
     * @var ApiManager
     */
    private $apiManager;

    /**
     * @var FakerGenerator
     */
    private $faker;

    /**
     * @var \stdClass
     */
    private $credentials;

    /**
     * @var string
     */
    private $userIdentifier;

    /**
     * @var string
     */
    private $applicationName;

    protected function setUp()
    {
        parent::setUp();
        static::bootKernel();
        $this->apiManager = new ApiManager(static::$kernel->getContainer());
        $this->serializer = $this->apiManager->serializer();
        $this->faker = FakerFactory::create();

        // Set same random data
        $this->credentials = new \stdClass();
        $this->credentials->serviceToken = $this->faker->uuid;
        $this->credentials->serviceName = $this->faker->name;
        $this->credentials->userIdentifier = $this->faker->uuid;
        $this->credentials->userName = $this->faker->userName;
        $this->credentials->code = $this->faker->randomNumber();

        $this->userIdentifier = $this->faker->uuid;
        $this->applicationName = $this->faker->name;
    }

    public function testGetAndSetCredentials()
    {
        $request = new Delete\Request();
        $credentials = new Credentials();

        $request->setCredentials($credentials);

        static::assertEquals($request->getCredentials(), $credentials);
    }

    public function testGetAndSetUserApplication()
    {
        $request = new Delete\Request();
        $userApplication = new Delete\UserApplication();

        $request->setUserApplication($userApplication);

        static::assertEquals($request->getUserApplication(), $userApplication);
    }

    public function testXMLSerialize()
    {
        $request = new Delete\Request();
        $userApplication = new Delete\UserApplication();
        $credentials = new Credentials();

        $credentials->setCredentialsData($this->credentials->serviceToken, $this->credentials->serviceName,
            $this->credentials->userIdentifier, $this->credentials->userName, $this->credentials->code);
        $userApplication->setUserApplicationTypeData($this->userIdentifier, $this->applicationName);

        $request->setCredentials($credentials);
        $request->setUserApplication($userApplication);

        static::assertXmlStringEqualsXmlString($this->serializer->serialize($request, 'xml'),
            $this->getExpectedResponseContent());
    }

    /**
     * @return string
     */
    private function getExpectedResponseContent()
    {
        return <<< XML
<?xml version="1.0" encoding="UTF-8"?>
<SsoRequest>
    <Credentials>
        <ServiceProvider>
            <ServiceToken><![CDATA[{$this->credentials->serviceToken}]]></ServiceToken>
            <ServiceName><![CDATA[{$this->credentials->serviceName}]]></ServiceName>
        </ServiceProvider>
        <ServiceTrigger>
            <UserIdentifier><![CDATA[{$this->credentials->userIdentifier}]]></UserIdentifier>
            <Username><![CDATA[{$this->credentials->userName}]]></Username>
        </ServiceTrigger>
        <ServiceCountry>
            <Code>{$this->credentials->code}</Code>
        </ServiceCountry>
    </Credentials>
    <UserApplication>
        <Delete>
            <Key>
                <UserType>
                    <Identifier><![CDATA[{$this->userIdentifier}]]></Identifier>
                </UserType>
                <ApplicationType>
                    <Name><![CDATA[{$this->applicationName}]]></Name>
                </ApplicationType>
            </Key>
        </Delete>
    </UserApplication>
</SsoRequest>
XML;
    }
}