<?php

/**
 * Class RequestTest
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\RestBundle\Tests\ModelXml\Request\UserWs\Version1\UserService\ActivateRequest;

use Sso\RestBundle\Api\Manager as ApiManager;
use Sso\RestBundle\ModelXml\Common\Request\Credentials\UserWs\Credentials;
use Sso\RestBundle\ModelXml\Request\UserWs\Version1\UserService\ActivateRequest;
use Faker\Factory as FakerFactory;
use Faker\Generator as FakerGenerator;
use JMS\Serializer\SerializerInterface;
use Symfony\Bundle\FrameworkBundle\Test\WebTestCase;

/**
 * Class RequestTest
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package Sso\RestBundle\Tests\ModelXml\Request\UserWs\Version1\UserService\ActivateRequest
 */
class RequestTest extends WebTestCase
{
    /**
     * @var SerializerInterface
     */
    private $serializer;

    /**
     * @var ApiManager
     */
    private $apiManager;

    /**
     * @var FakerGenerator
     */
    private $faker;

    /**
     * @var ActivateRequest\Request
     */
    private $request;

    protected function setUp()
    {
        parent::setUp();
        static::bootKernel();
        $this->apiManager = new ApiManager(static::$kernel->getContainer());
        $this->serializer = $this->apiManager->serializer();
        $this->faker = FakerFactory::create();

        $this->setUpRequest();
    }

    private function setUpRequest()
    {
        $request = new ActivateRequest\Request();
        $credentials = new Credentials();
        $userService = new ActivateRequest\UserService();

        $credentials->setCredentialsData($this->faker->uuid, $this->faker->name, $this->faker->userName,
            $this->faker->userName, $this->faker->text);
        $request->setCredentials($credentials)
            ->setUserService($userService);
        $userService->setUserServiceData($this->faker->userName);

        $this->request = $request;
    }

    public function testGetAndSetCredentials()
    {
        $request = new ActivateRequest\Request();
        $credentials = new Credentials();

        $request->setCredentials($credentials);

        static::assertEquals($request->getCredentials(), $credentials);
    }

    public function testGetAndSetUserServiceRole()
    {
        $request = new ActivateRequest\Request();
        $userService = new ActivateRequest\UserService();

        $request->setUserService($userService);

        static::assertEquals($request->getUserService(), $userService);
    }

    public function testXMLSerialize()
    {
        static::assertXmlStringEqualsXmlString($this->serializer->serialize($this->request, 'xml'),
            $this->getExpectedResponseContent());
    }

    /**
     * @return string
     */
    private function getExpectedResponseContent()
    {
        return <<< XML
<?xml version="1.0"?>
<SsoRequest>
    <Credentials>
        <ServiceProvider>
            <ServiceToken><![CDATA[{$this->request->getCredentials()->getServiceProvider()->getServiceToken()}]]></ServiceToken>
            <ServiceName><![CDATA[{$this->request->getCredentials()->getServiceProvider()->getServiceName()}]]></ServiceName>
        </ServiceProvider>
        <ServiceTrigger>
            <UserIdentifier><![CDATA[{$this->request->getCredentials()->getServiceTrigger()->getUserIdentifier()}]]></UserIdentifier>
            <Username><![CDATA[{$this->request->getCredentials()->getServiceTrigger()->getUsername()}]]></Username>
        </ServiceTrigger>
        <ServiceCountry>
            <Code>{$this->request->getCredentials()->getServiceCountry()->getCode()}</Code>
        </ServiceCountry>
    </Credentials>
    <UserService>
        <ActivateRequest>
            <Key>
                <UserType>
                    <Identifier><![CDATA[{$this->request->getUserService()->getActivateRequest()->getKey()->getUserType()->getIdentifier()}]]></Identifier>
                </UserType>
            </Key>
        </ActivateRequest>
    </UserService>
</SsoRequest>
XML;
    }
}