<?php

/**
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\RestBundle\Tests\ModelXml\Request\UserWs\Version2\UserApplication\Show;

use Sso\RestBundle\ModelXml\Common\Request\Credentials\UserWs\Credentials;
use Sso\RestBundle\ModelXml\Request\UserWs\Version2\UserApplication\Show\Request;
use Sso\RestBundle\ModelXml\Request\UserWs\Version2\UserApplication\Show\UserApplication;
use Symfony\Bundle\FrameworkBundle\Test\WebTestCase;
use Sso\RestBundle\Api\Manager as ApiManager;
use JMS\Serializer\SerializerInterface;
use Faker\Factory as FakerFactory;
use Faker\Generator as FakerGenerator;

/**
 * Class RequestTest
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package Sso\RestBundle\Tests\ModelRest
 */
class RequestTest extends WebTestCase
{
    /**
     * @var SerializerInterface
     */
    private $serializer;

    /**
     * @var ApiManager
     */
    private $apiManager;

    /**
     * @var FakerGenerator
     */
    private $faker;

    /**
     * @var string
     */
    private $serviceToken;

    /**
     * @var string
     */
    private $serviceName;

    /**
     * @var string
     */
    private $userIdentifier;

    /**
     * @var string
     */
    private $username;

    /**
     * @var string
     */
    private $langCode;

    /**
     * @var string
     */
    private $identifier;

    /**
     * @var integer
     */
    private $applicationActive;

    protected function setUp()
    {
        parent::setUp();
        static::bootKernel();
        $this->apiManager = new ApiManager(static::$kernel->getContainer());
        $this->serializer = $this->apiManager->serializer();
        $this->faker = FakerFactory::create();

        // Set same random data
        $this->serviceToken = $this->faker->sha256;
        $this->serviceName = $this->faker->word;
        $this->userIdentifier = $this->faker->md5;
        $this->username = $this->faker->userName;
        $this->langCode = $this->faker->languageCode;
        $this->identifier = $this->faker->uuid;
        $this->applicationActive = $this->faker->boolean ? 1 : 0;
    }

    public function testSerialize()
    {
        $this->assertXmlStringEqualsXmlString($this->getRequestBody(), $this->setRequestBody());
    }

    /**
     * @return string
     */
    private function setRequestBody()
    {
        $request = new Request();
        $credentials = new Credentials();
        $userApplication = new UserApplication();

        $credentials->setCredentialsData(
            $this->serviceToken,
            $this->serviceName,
            $this->userIdentifier,
            $this->username,
            $this->langCode
        );

        $userApplication->setUserApplication(
            $this->identifier,
            $this->applicationActive
        );

        $request->setCredentials($credentials);
        $request->setUserApplication($userApplication);

        return $this->serializer->serialize($request, 'xml');
    }

    /**
     * @return string
     */
    private function getRequestBody()
    {
        return '<SsoRequest>
            <Credentials>
                <ServiceProvider>
                    <ServiceToken>' . $this->serviceToken . '</ServiceToken>
                    <ServiceName>' . $this->serviceName . '</ServiceName>
                </ServiceProvider>
                <ServiceTrigger>
                    <UserIdentifier>' . $this->userIdentifier . '</UserIdentifier>
                    <Username>' . $this->username . '</Username>
                </ServiceTrigger>
                <ServiceCountry>
                    <Code>' . $this->langCode . '</Code>
                </ServiceCountry>
            </Credentials>
            <UserApplication>
                <Show>
                    <Key>
                        <UserType>
                            <Identifier>' . $this->identifier . '</Identifier>
                        </UserType>
                        <ApplicationType>
                            <Active>' . $this->applicationActive . '</Active>
                        </ApplicationType>
                    </Key>
                </Show>
            </UserApplication>
        </SsoRequest>';
    }

    public function testGetterUserType()
    {
        /** @var  Request $requestModel */
        $requestModel = $this->serializer->deserialize(
            $this->setRequestBody(),
            Request::class,
            'xml'
        );

        $userApplication = $requestModel->getUserApplication()->getShow();
        $userType = $userApplication->getKey()->getUserType();
        $applicationType = $userApplication->getKey()->getApplicationType();

        $credentials = $requestModel->getCredentials();

        $this->assertEquals($this->identifier, $userType->getIdentifier());
        $this->assertEquals($this->applicationActive, $applicationType->getActive());
        $this->assertEquals($this->serviceToken, $credentials->getServiceProvider()->getServiceToken());
    }
}
