<?php

/**
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\RestBundle\Tests\ModelXml\Request\UserWs\Version2\UserSearch\FindByApplication;

use Sso\RestBundle\ModelXml\Common\Request\Credentials\UserWs\Credentials;
use Sso\RestBundle\ModelXml\Request\UserWs\Version2\UserSearch\FindByApplication\Request;
use Sso\RestBundle\ModelXml\Request\UserWs\Version2\UserSearch\FindByApplication\UserSearch;
use Symfony\Bundle\FrameworkBundle\Test\WebTestCase;
use Sso\RestBundle\Api\Manager as ApiManager;
use JMS\Serializer\SerializerInterface;
use Faker\Factory as FakerFactory;
use Faker\Generator as FakerGenerator;

/**
 * Class RequestTest
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package Sso\RestBundle\Tests\ModelRest
 */
class RequestTest extends WebTestCase
{
    /**
     * @var SerializerInterface
     */
    private $serializer;

    /**
     * @var ApiManager
     */
    private $apiManager;

    /**
     * @var FakerGenerator
     */
    private $faker;

    /**
     * @var string
     */
    private $serviceToken;

    /**
     * @var string
     */
    private $serviceName;

    /**
     * @var string
     */
    private $userIdentifier;

    /**
     * @var string
     */
    private $username;

    /**
     * @var string
     */
    private $langCode;

    /**
     * @var string
     */
    private $modified;

    /**
     * @var string
     */
    private $appName;

    /**
     * @var integer
     */
    private $appActive;

    protected function setUp()
    {
        parent::setUp();
        static::bootKernel();
        $this->apiManager = new ApiManager(static::$kernel->getContainer());
        $this->serializer = $this->apiManager->serializer();
        $this->faker = FakerFactory::create();

        // Set same random data
        $this->serviceToken = $this->faker->sha256;
        $this->serviceName = $this->faker->word;
        $this->userIdentifier = $this->faker->md5;
        $this->username = $this->faker->userName;
        $this->langCode = $this->faker->languageCode;
        $this->modified = '2099-01-01T00:00:00+00:00';
        $this->appName = $this->faker->word;
        $this->appActive = $this->faker->boolean ? 1 : 0;
    }

    public function testSerialize()
    {
        $this->assertXmlStringEqualsXmlString($this->getRequestBody(), $this->setRequestBody());
    }

    /**
     * @return string
     */
    private function setRequestBody()
    {
        $request = new Request();
        $credentials = new Credentials();
        $userSearch = new UserSearch();

        $credentials->setCredentialsData(
            $this->serviceToken,
            $this->serviceName,
            $this->userIdentifier,
            $this->username,
            $this->langCode
        );

        $userSearch->setUserSearch(
            $this->appName,
            $this->appActive,
            $this->modified
        );

        $request->setCredentials($credentials);
        $request->setUserSearch($userSearch);

        return $this->serializer->serialize($request, 'xml');
    }

    /**
     * @return string
     */
    private function getRequestBody()
    {
        return '<SsoRequest>
            <Credentials>
                <ServiceProvider>
                    <ServiceToken>' . $this->serviceToken . '</ServiceToken>
                    <ServiceName>' . $this->serviceName . '</ServiceName>
                </ServiceProvider>
                <ServiceTrigger>
                    <UserIdentifier>' . $this->userIdentifier . '</UserIdentifier>
                    <Username>' . $this->username . '</Username>
                </ServiceTrigger>
                <ServiceCountry>
                    <Code>' . $this->langCode . '</Code>
                </ServiceCountry>
            </Credentials>
            <UserSearch>
                <FindByApplication>
                    <Key>
                        <UserType>
                            <ModifiedAfter>' . $this->modified . '</ModifiedAfter>
                        </UserType>
                        <ApplicationType>
                            <Name>' . $this->appName . '</Name>
                            <Active>' . $this->appActive . '</Active>
                        </ApplicationType>
                    </Key>
                </FindByApplication>
            </UserSearch>
        </SsoRequest>';
    }

    public function testGetterUserSearch()
    {
        /** @var  Request $requestModel */
        $requestModel = $this->serializer->deserialize(
            $this->setRequestBody(),
            Request::class,
            'xml'
        );

        $applicationAttribute = $requestModel->getUserSearch()->getFindByApplication();
        $userType = $applicationAttribute->getKey()->getUserType();
        $applicationType = $applicationAttribute->getKey()->getApplicationType();

        $credentials = $requestModel->getCredentials();

        $this->assertEquals(new \DateTime($this->modified), $userType->getModifiedAfter());
        $this->assertEquals($this->appName, $applicationType->getName());
        $this->assertEquals($this->appActive, $applicationType->getActive());
        $this->assertEquals($this->serviceToken, $credentials->getServiceProvider()->getServiceToken());
    }
}
