<?php

/**
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\RestBundle\Tests\ModelXml\Response\ObjectWs\Version1\AttributeGroup\Add;

use Sso\RestBundle\Api;
use Sso\RestBundle\ModelXml\Response\ObjectWs\Version1\AttributeGroup\Add\Error\Error;
use Sso\RestBundle\ModelXml\Response\ObjectWs\Version1\AttributeGroup\Add\Success\Add as SuccessAdd;
use Sso\RestBundle\ModelXml\Response\ObjectWs\Version1\AttributeGroup\Add\Success\AttributeGroup as SuccessAttributeGroup;
use Sso\RestBundle\ModelXml\Response\ObjectWs\Version1\AttributeGroup\Add\Success\AttributeGroupType;
use Sso\RestBundle\ModelXml\Response\ObjectWs\Version1\AttributeGroup\Add\Success\Response as SuccessXmlResponse;
use Sso\RestBundle\ModelXml\Response\ObjectWs\Version1\AttributeGroup\Add\Success\Type;
use Sso\RestBundle\ModelXml\Response\ObjectWs\Version1\AttributeGroup\Add\Error\Add as ErrorAdd;
use Sso\RestBundle\ModelXml\Response\ObjectWs\Version1\AttributeGroup\Add\Error\AttributeGroup as ErrorAttributeGroup;
use Sso\RestBundle\ModelXml\Response\ObjectWs\Version1\AttributeGroup\Add\Error\Response as ErrorXmlResponse;
use GuzzleHttp\Client;
use Symfony\Bundle\FrameworkBundle\Test\WebTestCase;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Faker\Factory as FakerFactory;
use Faker\Generator as FakerGenerator;

/**
 * Class ResponseTest
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */
class ResponseTest extends WebTestCase
{
    /**
     * @var Api\Manager
     */
    public $apiM;

    /**
     * @var Client
     */
    protected $client;

    /**
     * @var ContainerInterface
     */
    protected $container;

    /**
     * @var \JMS\Serializer\Serializer
     */
    protected $serializer;

    /**
     * @var FakerGenerator
     */
    private $faker;

    /**
     * @var float
     */
    private $scriptTimeSec;

    /**
     * @var string
     */
    private $trackId;

    /**
     * @var string
     */
    private $date;

    /**
     * @var string
     */
    private $message;

    /**
     * @var string
     */
    private $id;

    /**
     * @var string
     */
    private $name;

    /**
     * @var string
     */
    private $languageCode;

    protected function setUp()
    {
        $this->client = static::createClient();
        $this->container = $this->client->getContainer();
        $this->apiM = new Api\Manager($this->container);
        $this->serializer = $this->apiM->serializer();
        $this->faker = FakerFactory::create();

        // Set same random data
        $this->scriptTimeSec = $this->faker->randomFloat(2);
        $this->trackId = $this->faker->md5;
        $this->date = '2099-01-01T00:00:00+00:00';
        $this->message = $this->faker->sentence(2);
        $this->id = $this->faker->sha1;
        $this->name = $this->faker->name;
    }

    public function testSuccessResponseModel()
    {
        $this->assertXmlStringEqualsXmlString($this->getSuccessResponse(), $this->getSuccessResponseModel());
    }

    public function testErrorResponseModel()
    {
        $this->assertXmlStringEqualsXmlString($this->getErrorResponse(), $this->getErrorResponseModel());
    }

    private function getSuccessResponseModel()
    {
        $response = new SuccessXmlResponse();
        $response->setStatus('OK');
        $response->setTrackId($this->trackId);
        $response->setDate(new \DateTime($this->date));

        $attributeGroup = new SuccessAttributeGroup();
        $add = new SuccessAdd();
        $attributeGroupType = new AttributeGroupType();
        $type = new Type();

        $type->setId($this->id);

        $attributeGroupType
            ->setId($this->id)
            ->setName($this->name)
            ->setType($type);

        $add
            ->setStatus('OK')
            ->setAttributeGroupType($attributeGroupType);

        $attributeGroup->setAdd($add);

        $response->setAttributeGroup($attributeGroup);

        $responseModel = $this->serializer->serialize($response, 'xml');

        return $responseModel;
    }

    private function getErrorResponseModel()
    {
        $response = new ErrorXmlResponse();
        $response->setStatus('Failure');
        $response->setTrackId($this->trackId);
        $response->setDate(new \DateTime($this->date));

        $attributeGroup = new ErrorAttributeGroup();
        $add = new ErrorAdd();
        $error = new Error();

        $error->setErrorCode('fw003');
        $error->setErrorRef('hd007');
        $error->setShortMessage('InvalidXml');
        $error->setLongMessage($this->message);

        $add->setStatus('Failure')->setError($error);
        $attributeGroup->setAdd($add);

        $response->setAttributeGroup($attributeGroup);

        $responseModel = $this->serializer->serialize($response, 'xml');

        return $responseModel;
    }

    /**
     * @return string
     */
    private function getSuccessResponse()
    {
        return '<?xml version="1.0" encoding="utf-8"?>
            <SsoResponse>
                <Status><![CDATA[OK]]></Status>
                <Trackid><![CDATA[' . $this->trackId . ']]></Trackid>
                <Date><![CDATA[' . $this->date . ']]></Date>
                <AttributeGroup>
                    <Add>
                        <Status><![CDATA[OK]]></Status>
                        <AttributeGroupType>
                            <Id><![CDATA[' . $this->id . ']]></Id>
                            <Name><![CDATA[' . $this->name . ']]></Name>
                            <Types>
                                <Type>
                                    <Id><![CDATA[' . $this->id . ']]></Id>
                                </Type>
                            </Types>
                        </AttributeGroupType>
                    </Add>
                </AttributeGroup>
            </SsoResponse>';
    }

    /**
     * @return string
     */
    private function getErrorResponse()
    {
        return '<?xml version="1.0" encoding="utf-8"?>
            <SsoResponse>
                <Status><![CDATA[Failure]]></Status>
                <Trackid><![CDATA[' . $this->trackId . ']]></Trackid>
                <Date><![CDATA[' . $this->date . ']]></Date>
                <AttributeGroup>
                    <Add>
                        <Status><![CDATA[Failure]]></Status>
                        <Errors>
                            <Error>
                                <ErrorCode><![CDATA[fw003]]></ErrorCode>
                                <ErrorRef><![CDATA[hd007]]></ErrorRef>
                                <ShortMessage><![CDATA[InvalidXml]]></ShortMessage>
                                <LongMessage><![CDATA[' . $this->message . ']]></LongMessage>    
                            </Error>
                        </Errors>
                    </Add>
                </AttributeGroup>
            </SsoResponse>';
    }

    public function testSuccessGetters()
    {
        /** @var SuccessXmlResponse $responseModel */
        $responseModel = $this->serializer->deserialize(
            $this->getSuccessResponseModel(),
            SuccessXmlResponse::class,
            'xml'
        );

        $type = $responseModel->getAttributeGroup()->getAdd()->getAttributeGroupType()->getTypes()->get(0);

        $this->assertEquals('OK', $responseModel->getStatus());
        $this->assertEquals($this->trackId, $responseModel->getTrackId());
        $this->assertEquals(new \DateTime($this->date), $responseModel->getDate());
        $this->assertEquals('OK', $responseModel->getAttributeGroup()->getAdd()->getStatus());
        $this->assertEquals($this->id, $responseModel->getAttributeGroup()->getAdd()->getAttributeGroupType()->getId());
        $this->assertEquals($this->name,
            $responseModel->getAttributeGroup()->getAdd()->getAttributeGroupType()->getName());
        $this->assertEquals($this->id, $type->getId());
    }

    public function testErrorGetters()
    {
        /** @var ErrorXmlResponse $responseModel */
        $responseModel = $this->serializer->deserialize(
            $this->getErrorResponseModel(),
            ErrorXmlResponse::class,
            'xml'
        );

        $error = $responseModel->getAttributeGroup()->getAdd()->getErrors()->get(0);

        $this->assertEquals('Failure', $responseModel->getStatus());
        $this->assertEquals($this->trackId, $responseModel->getTrackId());
        $this->assertEquals(new \DateTime($this->date), $responseModel->getDate());
        $this->assertEquals('Failure', $responseModel->getAttributeGroup()->getAdd()->getStatus());
        $this->assertEquals('fw003', $error->getErrorCode());
        $this->assertEquals('hd007', $error->getErrorRef());
        $this->assertEquals('InvalidXml', $error->getShortMessage());
        $this->assertEquals($this->message, $error->getLongMessage());
    }
}
