<?php

/**
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\RestBundle\Tests\ModelXml\Response\UserWs\Version1\UserService\PasswordRequest;

use Sso\RestBundle\Api;
use Sso\RestBundle\ModelXml\Response\UserWs\Version1\UserService\PasswordRequest\Error\Error;
use Sso\RestBundle\ModelXml\Response\UserWs\Version1\UserService\PasswordRequest\Success\PasswordRequest as SuccessPasswordRequest;
use Sso\RestBundle\ModelXml\Response\UserWs\Version1\UserService\PasswordRequest\Success\Response as SuccessXmlResponse;
use Sso\RestBundle\ModelXml\Response\UserWs\Version1\UserService\PasswordRequest\Success\UserService as SuccessUserService;
use Sso\RestBundle\ModelXml\Response\UserWs\Version1\UserService\PasswordRequest\Success\UserType as SuccessUserType;
use Sso\RestBundle\ModelXml\Response\UserWs\Version1\UserService\PasswordRequest\Error\PasswordRequest as ErrorPasswordRequest;
use Sso\RestBundle\ModelXml\Response\UserWs\Version1\UserService\PasswordRequest\Error\Response as ErrorXmlResponse;
use Sso\RestBundle\ModelXml\Response\UserWs\Version1\UserService\PasswordRequest\Error\UserService as ErrorUserService;
use GuzzleHttp\Client;
use Symfony\Bundle\FrameworkBundle\Test\WebTestCase;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Faker\Factory as FakerFactory;
use Faker\Generator as FakerGenerator;

/**
 * Class ResponseTest
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */
class ResponseTest extends WebTestCase
{
    /**
     * @var Api\Manager
     */
    public $apiM;

    /**
     * @var Client
     */
    protected $client;

    /**
     * @var ContainerInterface
     */
    protected $container;

    /**
     * @var \JMS\Serializer\Serializer
     */
    protected $serializer;

    /**
     * @var FakerGenerator
     */
    private $faker;

    /**
     * @var float
     */
    private $scriptTimeSec;

    /**
     * @var string
     */
    private $trackId;

    /**
     * @var string
     */
    private $date;

    /**
     * @var string
     */
    private $message;

    /**
     * @var string
     */
    private $username;

    /**
     * @var string
     */
    private $email;

    /**
     * @var string
     */
    private $passwordToken;

    protected function setUp()
    {
        $this->client = static::createClient();
        $this->container = $this->client->getContainer();
        $this->apiM = new Api\Manager($this->container);
        $this->serializer = $this->apiM->serializer();
        $this->faker = FakerFactory::create();

        // Set same random data
        $this->scriptTimeSec = $this->faker->randomFloat(2);
        $this->trackId = $this->faker->md5;
        $this->date = '2099-01-01T00:00:00+00:00';
        $this->message = $this->faker->sentence(2);
        $this->username = $this->faker->userName;
        $this->email = $this->faker->email;
        $this->passwordToken = $this->faker->sha256;
    }

    public function testSuccessResponseModel()
    {
        $this->assertXmlStringEqualsXmlString($this->getSuccessResponse(), $this->getSuccessResponseModel());
    }

    public function testErrorResponseModel()
    {
        $this->assertXmlStringEqualsXmlString($this->getErrorResponse(), $this->getErrorResponseModel());
    }

    private function getSuccessResponseModel()
    {
        $response = new SuccessXmlResponse();
        $userService = new SuccessUserService();
        $passwordRequest = new SuccessPasswordRequest();
        $userType = new SuccessUserType();

        $response->setStatus('OK');
        $response->setTrackId($this->trackId);
        $response->setDate($this->date);

        $userType
            ->setUsername($this->username)
            ->setEmail($this->email)
            ->setPasswordToken($this->passwordToken)
            ->setPasswordTokenExpire($this->date);

        $passwordRequest
            ->setStatus('OK')
            ->setUserType($userType);

        $userService->setPasswordRequest($passwordRequest);

        $response->setUserService($userService);

        $responseModel = $this->serializer->serialize($response, 'xml');

        return $responseModel;
    }

    private function getErrorResponseModel()
    {
        $response = new ErrorXmlResponse();
        $response->setStatus('Failure');
        $response->setTrackId($this->trackId);
        $response->setDate($this->date);

        $userService = new ErrorUserService();
        $passwordRequest = new ErrorPasswordRequest();
        $error = new Error();

        $error->setErrorCode('fw003');
        $error->setErrorRef('hd007');
        $error->setShortMessage('InvalidXml');
        $error->setLongMessage($this->message);

        $passwordRequest->setStatus('Failure')->setError($error);
        $userService->setPasswordRequest($passwordRequest);

        $response->setUserService($userService);

        $responseModel = $this->serializer->serialize($response, 'xml');

        return $responseModel;
    }

    /**
     * @return string
     */
    private function getSuccessResponse()
    {
        return '<?xml version="1.0" encoding="utf-8"?>
            <SsoResponse>
                <Status><![CDATA[OK]]></Status>
                <Trackid><![CDATA[' . $this->trackId . ']]></Trackid>
                <Date><![CDATA[' . $this->date . ']]></Date>
                <UserService>
                    <PasswordRequest>
                        <Status><![CDATA[OK]]></Status>
                        <UserType>
                            <Username><![CDATA[' . $this->username . ']]></Username>
                            <Email><![CDATA[' . $this->email . ']]></Email>
                            <PasswordToken><![CDATA[' . $this->passwordToken . ']]></PasswordToken>
                            <PasswordTokenExpire><![CDATA[' . $this->date . ']]></PasswordTokenExpire>
                        </UserType>
                    </PasswordRequest>
                </UserService>
            </SsoResponse>';
    }

    /**
     * @return string
     */
    private function getErrorResponse()
    {
        return '<?xml version="1.0" encoding="utf-8"?>
            <SsoResponse>
                <Status><![CDATA[Failure]]></Status>
                <Trackid><![CDATA[' . $this->trackId . ']]></Trackid>
                <Date><![CDATA[' . $this->date . ']]></Date>
                <UserService>
                    <PasswordRequest>
                        <Status><![CDATA[Failure]]></Status>
                        <Errors>
                            <Error>
                                <ErrorCode><![CDATA[fw003]]></ErrorCode>
                                <ErrorRef><![CDATA[hd007]]></ErrorRef>
                                <ShortMessage><![CDATA[InvalidXml]]></ShortMessage>
                                <LongMessage><![CDATA[' . $this->message . ']]></LongMessage>    
                            </Error>
                        </Errors>
                    </PasswordRequest>
                </UserService>
            </SsoResponse>';
    }

    public function testSuccessGetters()
    {
        /** @var SuccessXmlResponse $responseModel */
        $responseModel = $this->serializer->deserialize(
            $this->getSuccessResponseModel(),
            SuccessXmlResponse::class,
            'xml'
        );

        $this->assertEquals('OK', $responseModel->getStatus());
        $this->assertEquals($this->trackId, $responseModel->getTrackId());
        $this->assertEquals(new \DateTime($this->date), $responseModel->getDate());
        $this->assertEquals('OK', $responseModel->getUserService()->getPasswordRequest()->getStatus());
        $this->assertEquals($this->username,
            $responseModel->getUserService()->getPasswordRequest()->getUserType()->getUsername());
        $this->assertEquals($this->email,
            $responseModel->getUserService()->getPasswordRequest()->getUserType()->getEmail());
        $this->assertEquals($this->passwordToken,
            $responseModel->getUserService()->getPasswordRequest()->getUserType()->getPasswordToken());
        $this->assertEquals($this->date,
            $responseModel->getUserService()->getPasswordRequest()->getUserType()->getPasswordTokenExpire());
    }

    public function testErrorGetters()
    {
        /** @var ErrorXmlResponse $responseModel */
        $responseModel = $this->serializer->deserialize(
            $this->getErrorResponseModel(),
            ErrorXmlResponse::class,
            'xml'
        );

        $error = $responseModel->getUserService()->getPasswordRequest()->getErrors()->get(0);

        $this->assertEquals('Failure', $responseModel->getStatus());
        $this->assertEquals($this->trackId, $responseModel->getTrackId());
        $this->assertEquals(new \DateTime($this->date), $responseModel->getDate());
        $this->assertEquals('Failure', $responseModel->getUserService()->getPasswordRequest()->getStatus());
        $this->assertEquals('fw003', $error->getErrorCode());
        $this->assertEquals('hd007', $error->getErrorRef());
        $this->assertEquals('InvalidXml', $error->getShortMessage());
        $this->assertEquals($this->message, $error->getLongMessage());
    }
}
