<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2018 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\RestBundle\Worker\ObjectWs\Version1\Object\Show\Multi;

use FOS\RestBundle\View\View as RestView;
use Sso\RestBundle\Api\Manager as ApiManager;
use LifeStyle\Tools\RestErrorBundle\Api\Manager as ErrorApi;
use LifeStyle\Tools\RestErrorBundle\Api\Error\Errors\Index as Errors;
use Sso\RestBundle\ModelRest\Request\ObjectWs\Version1\Object\Show\Multi\Request as RequestModel;
use LifeStyle\Tools\RestAuthBundle\Security\Authentication\Token\UidentifyToken;
use Symfony\Component\HttpFoundation\Request;
use GuzzleHttp\Exception\RequestException;

/**
 * Class Handler
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package    Sso\RestBundle\Worker\ObjectWs\Version1\Object\Show\Multi
 */
class Handler
{

    /**
     * @var UidentifyToken
     */
    protected $userToken;

    /**
     * @var ApiManager
     */
    protected $apiM;

    /**
     * @var ErrorApi
     */
    protected $errorApi;

    /**
     * @var Errors
     */
    protected $errors;

    /**
     * @var \Sso\RestBundle\ModelXml\Response\ObjectWs\Version1\Object\SHow\Multi\Error\Response
     */
    protected $errorXmlObject;

    /**
     * @var \Sso\RestBundle\ModelXml\Response\ObjectWs\Version1\Object\Show\Multi\Success\Response
     */
    protected $successXmlObject;

    /**
     * @var string
     */
    protected $format;

    /**
     * @var string
     */
    protected $requestedLangCode;

    /**
     * @param ApiManager $apiM
     */
    public function __construct(ApiManager $apiM)
    {
        $this->apiM = $apiM;
        $this->errorApi = $this->apiM->errorManager();
        $this->errors = $this->errorApi->error()->errors();
        $this->format = 'json';
        $this->userToken = $this->apiM->getToken();
    }

    /**
     *
     * @param Request $request
     * @return RestView
     */
    public function init(Request $request)
    {
        // first: validate input data
        $requestModelRest = $this->validateRestRequest($request);
        if ($this->errors->hasErrors()) {
            return $this->errorApi->error()->view()->getErrorsView($this->errors->getErrors());
        }

        $this->requestedLangCode = $requestModelRest->getLanguageCode();

        // second: send data to object ws ObjectGet
        $resModelXmlArray = $this->send($requestModelRest);

        // third: build response
        if ($resModelXmlArray['status']) {
            //ok
            $view = $this->buildSuccessView($resModelXmlArray['xml']);
        } else {
            //error
            $view = $this->buildErrorView($resModelXmlArray['xml']);
        }

        return $view;
    }

    /**
     * @param Request $request
     * @return RequestModel
     */
    private function validateRestRequest(Request $request)
    {
        $this->format = preg_match('/\.xml/i', $request->getUri()) ? 'xml' : 'json';

        $restRequestModel = $this->apiM->modelRest()->request()->objectWs()->version1()->object()->show()->multi();

        $restRequestModel
            ->setGuid($request->query->get('guid'))
            ->setReferenceId($request->query->get('referenceId'))
            ->setOffset($request->query->get('offset'))
            ->setLimit($request->query->get('limit'))
            ->setLanguageCode($request->query->get('languageCode'));

        $validationErrors = $this->apiM->validator()->validate($restRequestModel);

        if (count($validationErrors) > 0) {
            foreach ($validationErrors as $error) {
                $message = $error->getPropertyPath() . ": " . $error->getMessage();
                $this->errors->addError(400, $message, 'external')->setCode(400);
            }
        }

        return $restRequestModel;
    }

    /**
     *
     * @param string $xmlString
     * @return RestView
     */
    private function buildSuccessView($xmlString)
    {
        $this->successXmlObject = $this->apiM->serializer()->deserialize($xmlString,
            'Sso\RestBundle\ModelXml\Response\ObjectWs\Version1\Object\Show\Multi\Success\Response', 'xml');
        $modelRestResponse = $this->apiM->modelRest()->response()->objectWs()->version1()->object()->show()->multi()->success()->response($this->apiM->helper()->mapper());

        $modelRestResponse = $this->apiM->helper()->mapper()->modelInToModelOut($this->successXmlObject,
            $modelRestResponse);
        $modelRestResponse->setCode(200);
        $modelRestResponse->setScriptTimeSec($this->apiM->scriptTimeSeconds());
        $modelRestResponse->setCount($this->successXmlObject->getObject()->getShow()->getCount());
        $modelRestResponse->setOffset($this->successXmlObject->getObject()->getShow()->getOffset());
        $modelRestResponse->setLimit($this->successXmlObject->getObject()->getShow()->getLimit());


        // map and set object type models
        $xmlObjectTypes = $this->successXmlObject->getObject()->getShow()->getObjectType();
        foreach ($xmlObjectTypes as $objectType) {
            $mappedModel = $this->apiM->helper()->mapper()->modelInToModelOut($objectType,
                $this->apiM->modelRest()->response()->objectWs()->version1()->object()->show()->multi()->success()->objectType($this->apiM->helper()->mapper(),
                    $this->requestedLangCode));
            $modelRestResponse->addObject($mappedModel);
        }

        $view = RestView::create();
        $view->setData($modelRestResponse);
        $view->setStatusCode(200);

        return $view;
    }

    /**
     *
     * @param string $xmlString
     * @return RestView
     */
    private function buildErrorView($xmlString)
    {
        $responseItem = $this->apiM->modelRest()->response()->objectWs()->version1()->object()->show()->multi()->error()->response();

        $this->errorXmlObject = $this->apiM->serializer()->deserialize($xmlString,
            'Sso\RestBundle\ModelXml\Response\ObjectWs\Version1\Object\Show\Multi\Error\Response', 'xml');

        $responseItem->setTrackId($this->errorXmlObject->getTrackid());
        $responseItem->setScriptTimeSec($this->apiM->scriptTimeSeconds());
        $responseItem->setCode(400);

        $errors = $this->errorXmlObject->getObject()->getShow()->getError();

        foreach ($errors as $error) {
            $errorItem = $this->apiM->modelRest()->response()->objectWs()->version1()->object()->show()->multi()->error()->error();
            $message = 'ErrorCode: ' . $error->getErrorCode() . ' ErrorRef: ' . $error->getErrorRef() . ' ShortMessage: ' . $error->getShortMessage();
            $errorItem->setMessage($message);
            $errorItem->setCode(400);
            $errorItem->setException('external');
            $responseItem->addError($errorItem);
        }

        $view = RestView::create();
        $view->setData($responseItem);
        $view->setStatusCode(400);

        return $view;
    }

    /**
     * @param RequestModel $request
     * @return array
     */
    public function send(RequestModel $request)
    {
        $requestModelXml = $this->apiM->modelXml()->request()->objectWs()->version1()->object()->show()->multi()->request();

        // add credentials
        $credentialsModelXml = $this->apiM->modelXml()->request()->objectWs()->version1()->object()->show()->multi()->credentials();
        $credentialsModelXml->setCredentialsData($this->userToken->servicetoken, 'MyServiceName',
            $this->userToken->useridentifier, $this->userToken->username, 'en');
        $requestModelXml->setCredentials($credentialsModelXml);

        // add objecttype
        $objectModelXml = $this->apiM->modelXml()->request()->objectWs()->version1()->object()->show()->multi()->object();
        $objectModelXml->setObjectTypeData($request->getGuid(), $request->getReferenceId(), $request->getOffset(),
            $request->getLimit());
        $requestModelXml->setObject($objectModelXml);

        $options = array(
            'headers' => array(
                'API-VERSION' => '1',
                'API-CONTROLLER' => 'Object',
                'API-ACTION' => 'ShowMulti',
                'Content-Type' => 'text/xml; charset=UTF-8',
                'Accept' => 'text/xml,application/xml;q=0.9,*/*;q=0.8',
            ),
            'body' => $this->apiM->serializer()->serialize($requestModelXml, 'xml'),
        );

        try {
            $clientResponse = $this->apiM->client()->post($this->apiM->configuration()->objectWs()->getBaseUrl() . $this->apiM->configuration()->objectWs()->getServiceAction(),
                $options);
        } catch (RequestException $ex) {
            $errorResponse = $ex->getResponse()->getBody()->getContents();
            $this->apiM->logger()->error($errorResponse);

            return ['status' => false, 'xml' => $errorResponse];
        }

        return ['status' => true, 'xml' => $clientResponse->getBody()->getContents()];
    }

}
