<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2018 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\RestBundle\Worker\ObjectWs\Version1\ObjectValue\Update;

use Sso\RestBundle\Api\ExceptionCollection\MainValidationException;
use FOS\RestBundle\View\View as RestView;
use Sso\RestBundle\Api\Manager as ApiManager;
use GuzzleHttp\Exception\RequestException;
use LifeStyle\Tools\RestErrorBundle\Api\Manager as ErrorApi;
use LifeStyle\Tools\RestErrorBundle\Api\Error\Errors\Index as Errors;
use LifeStyle\Tools\RestAuthBundle\Security\Authentication\Token\UidentifyToken;
use Symfony\Component\HttpFoundation\Request;
use Sso\RestBundle\ModelRest\Request\ObjectWs\Version1\ObjectValue\Update\Request as RequestModel;
use Sso\RestBundle\ModelRest\Request\ObjectWs\Version1\ObjectValue\Update\ObjectValueType as RequestObjectValueType;
use Sso\RestBundle\ModelRest\Response\ObjectWs\Version1\ObjectValue\Update\ObjectValueType as ResponseObjectValueType;
use Sso\RestBundle\ModelRest\Response\ObjectWs\Version1\ObjectValue\Update\Response as RestResponseModel;
use Sso\RestBundle\ModelXml\Response\ObjectWs\Version1\ObjectValue\Update\Success\Response as SuccessXMLResponse;
use Sso\RestBundle\ModelXml\Response\ObjectWs\Version1\ObjectValue\Update\Error\Response as ErrorXMLResponse;

/**
 * Class Handler
 * @package Sso\RestBundle\Worker\ObjectWs\Version1\ObjectValue\Update
 */
class Handler
{
    const STATUS = "status";
    const ERROR_EXTERNAL = "external";

    /**
     * @var UidentifyToken
     */
    private $userToken;

    /**
     * @var ApiManager
     */
    private $apiM;

    /**
     * @var ErrorApi
     */
    private $errorApi;

    /**
     * @var Errors
     */
    private $errors;

    /**
     * @var ErrorXMLResponse
     */
    private $errorXmlEntity;

    /**
     * @var SuccessXMLResponse
     */
    private $successXmlEntity;

    /**
     * @var RestResponseModel
     */
    private $restResponseModel;

    /**
     * @var bool
     */
    private $oneResponseSuccess = false;

    /**
     * @param ApiManager $apiM
     */
    public function __construct(ApiManager $apiM)
    {
        $this->apiM = $apiM;
        $this->errorApi = $this->apiM->errorManager();
        $this->errors = $this->errorApi->error()->errors();
        $this->restResponseModel = $this->apiM->modelRest()->response()->objectWs()->version1()->objectValue()->update()->response();
        $this->userToken = $this->apiM->getToken();
    }

    /**
     * @param Request $request
     * @return RestView
     */
    public function init(Request $request)
    {
        // first: validate main if there is any error it is allowed to use the default error response
        try {
            list($requestContent, $format) = $this->apiM->mainValidator()->validateRestRequest($request);
        } catch (MainValidationException $exc) {
            $this->errorApi->error()->errors()->setScriptSeconds($this->apiM->scriptTimeSeconds());

            return $this->errorApi->error()->view()->getErrorsView($this->errors->getErrors(), 416);
        }

        // second: setting the request model
        $requestModelRest = $this->apiM->serializer()->deserialize($requestContent, RequestModel::class, $format);

        // loop the objectValueType
        foreach ($requestModelRest->getObjectValueType() as $objectValueType) {
            $responseObjectValueType = $this->sendXmlRequest($objectValueType);
            $this->restResponseModel->addObjectValueType($responseObjectValueType);
        }

        // reset the main code status
        $this->restResponseModel->setCode($this->oneResponseSuccess ? 200 : 400);

        return $this->apiM->restView()->setData($this->restResponseModel)->setStatusCode($this->oneResponseSuccess ? 200 : 400);
    }

    /**
     * @param RequestObjectValueType $requestObjectValueType
     * @return ResponseObjectValueType
     */
    private function sendXmlRequest(RequestObjectValueType $requestObjectValueType)
    {
        $requestModel = $this->apiM->modelXml()->request()->objectWs()->version1()->objectValue()->update()->request();

        // add credentials
        $credentialsModelXml = $this->apiM->modelXml()->request()->objectWs()->version1()->objectValue()->update()->credentials();
        $credentialsModelXml->setCredentialsData(
            $this->userToken->servicetoken,
            'ObjectValue',
            $this->userToken->useridentifier,
            $this->userToken->username,
            'en'
        );
        $requestModel->setCredentials($credentialsModelXml);

        // add objectValue
        $objectModelXml = $this->apiM->modelXml()->request()->objectWs()->version1()->objectValue()->update()->objectValue();
        $objectModelXml->setObjectValue(
            $requestObjectValueType->getId(),
            $requestObjectValueType->getValue(),
            $requestObjectValueType->getLanguageCode()
        );
        $requestModel->setObjectValue($objectModelXml);

        // send request
        try {
            $clientResponse = $this->apiM->client()->post(
                $this->apiM->configuration()->objectWs()->getBaseUrl() . $this->apiM->configuration()->objectWs()->getServiceAction(),
                $this->apiM->clientHelper()->options()->getOptions(1, 'ObjectValue', 'Update',
                    $this->apiM->serializer()->serialize($requestModel, 'xml'))
            );
        } catch (RequestException $ex) {
            $errorResponse = $ex->getResponse()->getBody()->getContents();
            $this->apiM->logger()->error($errorResponse);

            return $this->buildItem([static::STATUS => false, 'xml' => $errorResponse]);
        }

        return $this->buildItem([static::STATUS => true, 'xml' => $clientResponse->getBody()->getContents()]);
    }

    /**
     * @param string $xmlResponse
     * @return ResponseObjectValueType
     */
    private function buildItem($xmlResponse)
    {
        $responseItem = $this->apiM->modelRest()->response()->objectWs()->version1()->objectValue()->update()->objectValueType();

        if (!$xmlResponse[static::STATUS]) {
            // build error response for a objectValueType
            $view = $this->buildError($xmlResponse['xml'], $responseItem);
        } else {
            // fill the success response for a objectValueType
            $view = $this->buildSuccess($xmlResponse['xml'], $responseItem);
        }

        return $view;
    }

    /**
     * @param string $xmlString
     * @param ResponseObjectValueType $responseItem
     * @return ResponseObjectValueType
     */
    private function buildSuccess($xmlString, ResponseObjectValueType $responseItem)
    {
        $this->successXmlEntity = $this->apiM->serializer()->deserialize($xmlString, SuccessXMLResponse::class, 'xml');

        if (!$this->successXmlEntity->getObjectValue()
            || ($this->successXmlEntity->getObjectValue()
                && $this->successXmlEntity->getObjectValue()->getUpdate()->getStatus() != 'Success')
        ) {
            return $this->buildError($xmlString, $responseItem);
        }

        $this->oneResponseSuccess = true;

        /**
         * @var ResponseObjectValueType $responseItem
         */
        $responseItem = $this->apiM->helper()->mapper()->modelInToModelOut(
            $this->successXmlEntity->getObjectValue()->getUpdate()->getObjectValueType(),
            $responseItem
        );

        // Update additional fields to response
        $this->restResponseModel
            ->setCode(200)
            ->setTrackId($this->successXmlEntity->getTrackId())
            ->setDate($this->successXmlEntity->getDate())
            ->setScriptTimeSec($this->apiM->scriptTimeSeconds());

        // Update additional fields to success response
        $responseItem->setCode(200);

        return $responseItem;
    }

    /**
     * @param ResponseObjectValueType $responseItem
     * @param string $xmlString
     * @return ResponseObjectValueType
     */
    private function buildError($xmlString, ResponseObjectValueType $responseItem)
    {
        $this->errorXmlEntity = $this->apiM->serializer()->deserialize($xmlString, ErrorXMLResponse::class, 'xml');
        $errors = $this->errorXmlEntity->getObjectValue()->getUpdate()->getErrors();

        foreach ($errors as $error) {
            $errorItem = $this->apiM->modelRest()->response()->objectWs()->version1()->objectValue()->update()->error();
            $message = 'ErrorCode: ' . $error->getErrorCode() . ' ErrorRef: ' . $error->getErrorRef() . ' ShortMessage: ' . $error->getShortMessage();
            $errorItem->setMessage($message);
            $errorItem->setCode(400);
            $errorItem->setException(static::ERROR_EXTERNAL);
            $responseItem->setError($errorItem);
        }

        // Update additional fields to response
        $this->restResponseModel
            ->setCode(400)
            ->setTrackId($this->errorXmlEntity->getTrackId())
            ->setDate($this->errorXmlEntity->getDate())
            ->setScriptTimeSec($this->apiM->scriptTimeSeconds());

        // Update additional fields to error response
        $responseItem->setCode(400);

        return $responseItem;
    }
}
