<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2018 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\RestBundle\Worker\UserWs\Version2\UserSearch\FindByApplication;

use Sso\RestBundle\ModelRest\Request\UserWs\Version2\UserSearch\FindByApplication\ApplicationType;
use Sso\RestBundle\ModelRest\Request\UserWs\Version2\UserSearch\FindByApplication\UserByApplication;
use Sso\RestBundle\ModelRest\Request\UserWs\Version2\UserSearch\FindByApplication\UserType;
use Sso\RestBundle\ModelRest\Response\UserWs\Version2\UserSearch\FindByApplication\UserType as ResponseUserType;
use Doctrine\Common\Collections\ArrayCollection;
use FOS\RestBundle\View\View as RestView;
use Sso\RestBundle\Api\Manager as ApiManager;
use GuzzleHttp\Exception\RequestException;
use LifeStyle\Tools\RestErrorBundle\Api\Manager as ErrorApi;
use LifeStyle\Tools\RestErrorBundle\Api\Error\Errors\Index as Errors;
use LifeStyle\Tools\RestAuthBundle\Security\Authentication\Token\UidentifyToken;
use Symfony\Component\HttpFoundation\Request;
use Sso\RestBundle\ModelRest\Request\UserWs\Version2\UserSearch\FindByApplication\Request as RequestModel;
use Sso\RestBundle\ModelRest\Response\UserWs\Version2\UserSearch\FindByApplication\UserByApplication as ResponseUserByApplication;
use Sso\RestBundle\ModelRest\Response\UserWs\Version2\UserSearch\FindByApplication\Response as RestResponseModel;
use Sso\RestBundle\ModelXml\Response\UserWs\Version2\UserSearch\FindByApplication\Success\Response as SuccessXMLResponse;
use Sso\RestBundle\ModelXml\Response\UserWs\Version2\UserSearch\FindByApplication\Error\Response as ErrorXMLResponse;

/**
 * Class Handler
 * @package Sso\RestBundle\Worker\UserWs\Version2\UserSearch\FindByApplication
 */
class Handler
{
    const STATUS = "status";
    const ERROR_EXTERNAL = "external";

    /**
     * @var string
     */
    private $format = 'json';

    /**
     * @var UidentifyToken
     */
    private $userToken;

    /**
     * @var ApiManager
     */
    private $apiM;

    /**
     * @var ErrorApi
     */
    private $errorApi;

    /**
     * @var Errors
     */
    private $errors;

    /**
     * @var ErrorXMLResponse
     */
    private $errorXmlEntity;

    /**
     * @var SuccessXMLResponse
     */
    private $successXmlEntity;

    /**
     * @var RestResponseModel
     */
    private $restResponseModel;

    /**
     * @var bool
     */
    private $oneResponseSuccess = false;

    /**
     * @param ApiManager $apiM
     */
    public function __construct(ApiManager $apiM)
    {
        $this->apiM = $apiM;
        $this->errorApi = $this->apiM->errorManager();
        $this->errors = $this->errorApi->error()->errors();
        $this->restResponseModel = $this->apiM->modelRest()->response()->userWs()->version2()->userSearch()->findByApplication()->response();
        $this->userToken = $this->apiM->getToken();
    }

    /**
     * @param Request $request
     * @return RestView
     */
    public function init(Request $request)
    {
        // first: validate main if there is any error it is allowed to use the default error response
        $requestModelRest = $this->validateRestRequest($request);

        if ($this->errors->hasErrors()) {
            $this->errorApi->error()->errors()->setScriptSeconds($this->apiM->scriptTimeSeconds());

            return $this->errorApi->error()->view()->getErrorsView($this->errors->getErrors(), 416);
        }

        // second: send data
        $requestUserByApplication = $this->sendXmlRequest($requestModelRest);
        $this->restResponseModel->setUserByApplication($requestUserByApplication);

        // reset the main code status
        $this->restResponseModel->setCode($this->oneResponseSuccess ? 200 : 400);

        return $this->apiM->restView()->setData($this->restResponseModel)->setStatusCode($this->oneResponseSuccess ? 200 : 400);
    }

    /**
     * @param Request $request
     * @return RequestModel
     */
    private function validateRestRequest(Request $request)
    {
        $this->format = preg_match('/\.xml/i', $request->getUri()) ? 'xml' : 'json';

        $restRequestModel = $this->apiM->modelRest()->request()->userWs()->version2()->userSearch()->findByApplication();

        $UserByApplication = new UserByApplication();
        $userType = new UserType();
        $applicationType = new ApplicationType();

        $applicationType
            ->setName($request->query->get('applicationName'))
            ->setActive($request->query->get('applicationActive'));

        $userType
            ->setModified($request->query->get('modifiedAfter'))
            ->setApplicationType($applicationType);

        $UserByApplication->setUserType($userType);
        $restRequestModel->setUserByApplication($UserByApplication);

        $validationErrors = $this->apiM->validator()->validate($restRequestModel);

        if (count($validationErrors) > 0) {
            foreach ($validationErrors as $error) {
                $message = $error->getPropertyPath() . ": " . $error->getMessage();
                $this->errors->addError(400, $message, 'external')->setCode(400);
            }
        }

        return $restRequestModel;
    }

    /**
     * @param RequestModel $requestUserType
     * @return ResponseUserByApplication
     */
    private function sendXmlRequest(RequestModel $requestUserType)
    {
        $requestModel = $this->apiM->modelXml()->request()->userWs()->version2()->userSearch()->findByApplication()->request();

        // add credentials
        $credentialsModelXml = $this->apiM->modelXml()->request()->userWs()->version2()->userSearch()->findByApplication()->credentials();
        $credentialsModelXml->setCredentialsData(
            $this->userToken->servicetoken,
            'UserSearch',
            $this->userToken->useridentifier,
            $this->userToken->username,
            'en'
        );
        $requestModel->setCredentials($credentialsModelXml);

        // add applicationAttribute
        $objectModelXml = $this->apiM->modelXml()->request()->userWs()->version2()->userSearch()->findByApplication()->userSearch();
        $objectModelXml->setUserSearch(
            $requestUserType->getUserByApplication()->getUserType()->getApplicationType()->getName(),
            $requestUserType->getUserByApplication()->getUserType()->getApplicationType()->getActive(),
            $requestUserType->getUserByApplication()->getUserType()->getModified()
        );
        $requestModel->setUserSearch($objectModelXml);

        // send request
        try {
            $clientResponse = $this->apiM->client()->post(
                $this->apiM->configuration()->userWs()->getBaseUrl() . $this->apiM->configuration()->userWs()->getServiceAction(),
                $this->apiM->clientHelper()->options()->getOptions(2, 'UserSearch', 'FindByApplication',
                    $this->apiM->serializer()->serialize($requestModel, 'xml'))
            );
        } catch (RequestException $ex) {
            $errorResponse = $ex->getResponse()->getBody()->getContents();
            $this->apiM->logger()->error($errorResponse);

            return $this->buildItem([static::STATUS => false, 'xml' => $errorResponse]);
        }

        return $this->buildItem([static::STATUS => true, 'xml' => $clientResponse->getBody()->getContents()]);
    }

    /**
     * @param string $xmlResponse
     * @return ResponseUserByApplication
     */
    private function buildItem($xmlResponse)
    {
        $responseItem = $this->apiM->modelRest()->response()->userWs()->version2()->userSearch()->findByApplication()->UserByApplication();

        if (!$xmlResponse[static::STATUS]) {
            // build error response for a UserByApplication
            $view = $this->buildError($xmlResponse['xml'], $responseItem);
        } else {
            // fill the success response for a UserByApplication
            $view = $this->buildSuccess($xmlResponse['xml'], $responseItem);
        }

        return $view;
    }

    /**
     * @param string $xmlString
     * @param ResponseUserByApplication $responseItem
     * @return ResponseUserByApplication
     */
    private function buildSuccess($xmlString, ResponseUserByApplication $responseItem)
    {
        $this->successXmlEntity = $this->apiM->serializer()->deserialize($xmlString, SuccessXMLResponse::class, 'xml');

        if (!$this->successXmlEntity->getUserSearch()
            || ($this->successXmlEntity->getUserSearch()
                && $this->successXmlEntity->getUserSearch()->getFindByApplication()->getStatus() != 'Success')
        ) {
            return $this->buildError($xmlString, $responseItem);
        }

        $this->oneResponseSuccess = true;

        /**
         * @var ResponseUserByApplication $responseItem
         */
        $responseItem = $this->apiM->helper()->mapper()->modelInToModelOut(
            $this->successXmlEntity->getUserSearch()->getFindByApplication(),
            $responseItem
        );

        // Add additional fields to response
        $this->restResponseModel
            ->setCode(200)
            ->setTrackId($this->successXmlEntity->getTrackId())
            ->setDate($this->successXmlEntity->getDate())
            ->setScriptTimeSec($this->apiM->scriptTimeSeconds());

        // Add additional fields to success response
        foreach ($responseItem->getUserType() as $userType) {
            $userType->setCode(200);
        }

        return $responseItem;
    }

    /**
     * @param ResponseUserByApplication $responseItem
     * @param string $xmlString
     * @return ResponseUserByApplication
     */
    private function buildError($xmlString, ResponseUserByApplication $responseItem)
    {
        $this->errorXmlEntity = $this->apiM->serializer()->deserialize($xmlString, ErrorXMLResponse::class, 'xml');
        $errors = $this->errorXmlEntity->getErrors();
        $userTypes = new ArrayCollection();

        foreach ($errors as $error) {
            $errorItem = $this->apiM->modelRest()->response()->userWs()->version2()->userSearch()->findByApplication()->error();
            $message = 'ErrorCode: ' . $error->getErrorCode() . ' ErrorRef: ' . $error->getErrorRef() . ' ShortMessage: ' . $error->getShortMessage();
            $errorItem->setCode($this->errorXmlEntity->getCode());
            $errorItem->setMessage($message);
            $errorItem->setException(static::ERROR_EXTERNAL);

            $userType = new ResponseUserType();
            $userType->setCode($this->errorXmlEntity->getCode());
            $userType->setError($errorItem);
            $userTypes->add($userType);
        }

        // Add additional fields to response
        $this->restResponseModel
            ->setCode($this->errorXmlEntity->getCode())
            ->setTrackId($this->errorXmlEntity->getTrackId())
            ->setDate($this->errorXmlEntity->getDate())
            ->setScriptTimeSec($this->apiM->scriptTimeSeconds());

        // Add additional fields to error response
        $responseItem->setUserType($userTypes);

        return $responseItem;
    }
}
