<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2018 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\RestBundle\Worker\UserWs\Version2\UserSearch\FindByApplicationAttribute;

use Sso\RestBundle\ModelRest\Request\UserWs\Version2\UserSearch\FindByApplicationAttribute\ApplicationType;
use Sso\RestBundle\ModelRest\Request\UserWs\Version2\UserSearch\FindByApplicationAttribute\AttributeType;
use Sso\RestBundle\ModelRest\Request\UserWs\Version2\UserSearch\FindByApplicationAttribute\UserByAttribute;
use Sso\RestBundle\ModelRest\Request\UserWs\Version2\UserSearch\FindByApplicationAttribute\UserType;
use Sso\RestBundle\ModelRest\Response\UserWs\Version2\UserSearch\FindByApplicationAttribute\UserType as ResponseUserType;
use Doctrine\Common\Collections\ArrayCollection;
use FOS\RestBundle\View\View as RestView;
use Sso\RestBundle\Api\Manager as ApiManager;
use GuzzleHttp\Exception\RequestException;
use LifeStyle\Tools\RestErrorBundle\Api\Manager as ErrorApi;
use LifeStyle\Tools\RestErrorBundle\Api\Error\Errors\Index as Errors;
use LifeStyle\Tools\RestAuthBundle\Security\Authentication\Token\UidentifyToken;
use Symfony\Component\HttpFoundation\Request;
use Sso\RestBundle\ModelRest\Request\UserWs\Version2\UserSearch\FindByApplicationAttribute\Request as RequestModel;
use Sso\RestBundle\ModelRest\Response\UserWs\Version2\UserSearch\FindByApplicationAttribute\UserByAttribute as ResponseUserByAttribute;
use Sso\RestBundle\ModelRest\Response\UserWs\Version2\UserSearch\FindByApplicationAttribute\Response as RestResponseModel;
use Sso\RestBundle\ModelXml\Response\UserWs\Version2\UserSearch\FindByApplicationAttribute\Success\Response as SuccessXMLResponse;
use Sso\RestBundle\ModelXml\Response\UserWs\Version2\UserSearch\FindByApplicationAttribute\Error\Response as ErrorXMLResponse;

/**
 * Class Handler
 * @package Sso\RestBundle\Worker\UserWs\Version2\UserSearch\FindByApplicationAttribute
 */
class Handler
{
    const STATUS = "status";
    const ERROR_EXTERNAL = "external";

    /**
     * @var string
     */
    private $format = 'json';

    /**
     * @var UidentifyToken
     */
    private $userToken;

    /**
     * @var ApiManager
     */
    private $apiM;

    /**
     * @var ErrorApi
     */
    private $errorApi;

    /**
     * @var Errors
     */
    private $errors;

    /**
     * @var ErrorXMLResponse
     */
    private $errorXmlEntity;

    /**
     * @var SuccessXMLResponse
     */
    private $successXmlEntity;

    /**
     * @var RestResponseModel
     */
    private $restResponseModel;

    /**
     * @var bool
     */
    private $oneResponseSuccess = false;

    /**
     * @param ApiManager $apiM
     */
    public function __construct(ApiManager $apiM)
    {
        $this->apiM = $apiM;
        $this->errorApi = $this->apiM->errorManager();
        $this->errors = $this->errorApi->error()->errors();
        $this->restResponseModel = $this->apiM->modelRest()->response()->userWs()->version2()->userSearch()->findByApplicationAttribute()->response();
        $this->userToken = $this->apiM->getToken();
    }

    /**
     * @param Request $request
     * @return RestView
     */
    public function init(Request $request)
    {
        // first: validate main if there is any error it is allowed to use the default error response
        $requestModelRest = $this->validateRestRequest($request);

        if ($this->errors->hasErrors()) {
            $this->errorApi->error()->errors()->setScriptSeconds($this->apiM->scriptTimeSeconds());

            return $this->errorApi->error()->view()->getErrorsView($this->errors->getErrors(), 416);
        }

        // second: send data
        $requestUserByAttribute = $this->sendXmlRequest($requestModelRest);
        $this->restResponseModel->setUserByAttribute($requestUserByAttribute);

        // reset the main code status
        $this->restResponseModel->setCode($this->oneResponseSuccess ? 200 : 400);

        return $this->apiM->restView()->setData($this->restResponseModel)->setStatusCode($this->oneResponseSuccess ? 200 : 400);
    }

    /**
     * @param Request $request
     * @return RequestModel
     */
    private function validateRestRequest(Request $request)
    {
        $this->format = preg_match('/\.xml/i', $request->getUri()) ? 'xml' : 'json';

        $restRequestModel = $this->apiM->modelRest()->request()->userWs()->version2()->userSearch()->findByApplicationAttribute();

        $userByAttribute = new UserByAttribute();
        $userType = new UserType();
        $applicationType = new ApplicationType();
        $attributeType = new AttributeType();

        $applicationType
            ->setName($request->query->get('applicationName'))
            ->setActive($request->query->get('applicationActive'));

        $attributeType
            ->setName($request->query->get('attributeName'))
            ->setValue($request->query->get('attributeValue'));

        $userType
            ->setModified($request->query->get('modifiedAfter'))
            ->setApplicationType($applicationType)
            ->setAttributeType($attributeType);

        $userByAttribute->setUserType($userType);
        $restRequestModel->setUserByAttribute($userByAttribute);

        $validationErrors = $this->apiM->validator()->validate($restRequestModel);

        if (count($validationErrors) > 0) {
            foreach ($validationErrors as $error) {
                $message = $error->getPropertyPath() . ": " . $error->getMessage();
                $this->errors->addError(400, $message, 'external')->setCode(400);
            }
        }

        return $restRequestModel;
    }

    /**
     * @param RequestModel $requestUserType
     * @return ResponseUserByAttribute
     */
    private function sendXmlRequest(RequestModel $requestUserType)
    {
        $requestModel = $this->apiM->modelXml()->request()->userWs()->version2()->userSearch()->findByApplicationAttribute()->request();

        // add credentials
        $credentialsModelXml = $this->apiM->modelXml()->request()->userWs()->version2()->userSearch()->findByApplicationAttribute()->credentials();
        $credentialsModelXml->setCredentialsData(
            $this->userToken->servicetoken,
            'UserSearch',
            $this->userToken->useridentifier,
            $this->userToken->username,
            'en'
        );
        $requestModel->setCredentials($credentialsModelXml);

        // add applicationAttribute
        $objectModelXml = $this->apiM->modelXml()->request()->userWs()->version2()->userSearch()->findByApplicationAttribute()->userSearch();
        $objectModelXml->setUserSearch(
            $requestUserType->getUserByAttribute()->getUserType()->getApplicationType()->getName(),
            $requestUserType->getUserByAttribute()->getUserType()->getApplicationType()->getActive(),
            $requestUserType->getUserByAttribute()->getUserType()->getAttributeType()->getName(),
            $requestUserType->getUserByAttribute()->getUserType()->getAttributeType()->getValue(),
            $requestUserType->getUserByAttribute()->getUserType()->getModified()
        );
        $requestModel->setUserSearch($objectModelXml);

        // send request
        try {
            $clientResponse = $this->apiM->client()->post(
                $this->apiM->configuration()->userWs()->getBaseUrl() . $this->apiM->configuration()->userWs()->getServiceAction(),
                $this->apiM->clientHelper()->options()->getOptions(2, 'UserSearch', 'FindByApplicationAttribute',
                    $this->apiM->serializer()->serialize($requestModel, 'xml'))
            );
        } catch (RequestException $ex) {
            $errorResponse = $ex->getResponse()->getBody()->getContents();
            $this->apiM->logger()->error($errorResponse);

            return $this->buildItem([static::STATUS => false, 'xml' => $errorResponse]);
        }

        return $this->buildItem([static::STATUS => true, 'xml' => $clientResponse->getBody()->getContents()]);
    }

    /**
     * @param string $xmlResponse
     * @return ResponseUserByAttribute
     */
    private function buildItem($xmlResponse)
    {
        $responseItem = $this->apiM->modelRest()->response()->userWs()->version2()->userSearch()->findByApplicationAttribute()->userByAttribute();

        if (!$xmlResponse[static::STATUS]) {
            // build error response for a userByAttribute
            $view = $this->buildError($xmlResponse['xml'], $responseItem);
        } else {
            // fill the success response for a userByAttribute
            $view = $this->buildSuccess($xmlResponse['xml'], $responseItem);
        }

        return $view;
    }

    /**
     * @param string $xmlString
     * @param ResponseUserByAttribute $responseItem
     * @return ResponseUserByAttribute
     */
    private function buildSuccess($xmlString, ResponseUserByAttribute $responseItem)
    {
        $this->successXmlEntity = $this->apiM->serializer()->deserialize($xmlString, SuccessXMLResponse::class, 'xml');

        if (!$this->successXmlEntity->getUserSearch()
            || ($this->successXmlEntity->getUserSearch()
                && $this->successXmlEntity->getUserSearch()->getFindByApplicationAttribute()->getStatus() != 'Success')
        ) {
            return $this->buildError($xmlString, $responseItem);
        }

        $this->oneResponseSuccess = true;

        /**
         * @var ResponseUserByAttribute $responseItem
         */
        $responseItem = $this->apiM->helper()->mapper()->modelInToModelOut(
            $this->successXmlEntity->getUserSearch()->getFindByApplicationAttribute(),
            $responseItem
        );

        // Add additional fields to response
        $this->restResponseModel
            ->setCode(200)
            ->setTrackId($this->successXmlEntity->getTrackId())
            ->setDate($this->successXmlEntity->getDate())
            ->setScriptTimeSec($this->apiM->scriptTimeSeconds());

        // Add additional fields to success response
        foreach ($responseItem->getUserType() as $userType) {
            $userType->setCode(200);
        }

        return $responseItem;
    }

    /**
     * @param ResponseUserByAttribute $responseItem
     * @param string $xmlString
     * @return ResponseUserByAttribute
     */
    private function buildError($xmlString, ResponseUserByAttribute $responseItem)
    {
        $this->errorXmlEntity = $this->apiM->serializer()->deserialize($xmlString, ErrorXMLResponse::class, 'xml');
        $errors = $this->errorXmlEntity->getErrors();
        $userTypes = new ArrayCollection();

        foreach ($errors as $error) {
            $errorItem = $this->apiM->modelRest()->response()->userWs()->version2()->userSearch()->findByApplicationAttribute()->error();
            $message = 'ErrorCode: ' . $error->getErrorCode() . ' ErrorRef: ' . $error->getErrorRef() . ' ShortMessage: ' . $error->getShortMessage();
            $errorItem->setCode($this->errorXmlEntity->getCode());
            $errorItem->setMessage($message);
            $errorItem->setException(static::ERROR_EXTERNAL);

            $userType = new ResponseUserType();
            $userType->setCode($this->errorXmlEntity->getCode());
            $userType->setError($errorItem);
            $userTypes->add($userType);
        }

        // Add additional fields to response
        $this->restResponseModel
            ->setCode($this->errorXmlEntity->getCode())
            ->setTrackId($this->errorXmlEntity->getTrackId())
            ->setDate($this->errorXmlEntity->getDate())
            ->setScriptTimeSec($this->apiM->scriptTimeSeconds());

        // Add additional fields to error response
        $responseItem->setUserType($userTypes);

        return $responseItem;
    }
}
