<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2018 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\RestBundle\Controller\ObjectWs\Version1;

use FOS\RestBundle\Controller\Annotations\Post;
use FOS\RestBundle\View\View;
use JMS\SecurityExtraBundle\Annotation\Secure;
use Nelmio\ApiDocBundle\Annotation\ApiDoc;
use FOS\RestBundle\Controller\Annotations\NamePrefix;
use Symfony\Component\HttpFoundation\Request;
use FOS\RestBundle\Controller\FOSRestController;
use Sso\RestBundle\Worker\Worker as Worker;
use JMS\DiExtraBundle\Annotation as DI;
use FOS\RestBundle\Controller\Annotations\Delete;
use FOS\RestBundle\Controller\Annotations\Get;
use FOS\RestBundle\Controller\Annotations\Put;
use LifeStyle\Tools\CachingBundle\Annotations\LsCache;

/**
 * Class AttributeGroupController
 *
 * @NamePrefix("api_rest_")
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package    Sso\RestBundle\Controller
 */
class AttributeGroupController extends FOSRestController
{

    /**
     * @DI\Inject("rest_worker_manager")
     * @var Worker
     */
    protected $woM;

    /**
     *
     * This service is used to get all or a single attribute group by its id.
     * Returns a list of attribute groups with associated attributes included.
     *
     * #Request (header)
     *
     *      Key                 | Value               | Description
     *      ------------------- | ------------------- | --------------------------------------
     *      username            |  (string)           | your username
     *      useridentifier      |  (string)           | your useridentifier for given username
     *      servicetoken        |  (string)           | servicetoken from ws
     *
     * ##Request
     *
     * **Request example JSON**
     *
     *          /api/v1/attribute/group.json?id=YOURID&offset=YOUROFFSET&limit=YOURLIMIT
     *
     * **Request example XML**
     *
     *          /api/v1/attribute/group.xml?id=YOURID&offset=YOUROFFSET&limit=YOURLIMIT
     *
     * ##Response success
     *
     * **Example (header)**
     *
     *          200 OK
     *
     * **Example JSON (body)**
     *
     *          {
     *            "status": "Success",
     *            "trackid": "a3b0c061e3",
     *            "date": "2016-05-12T09:53:34+0200",
     *            "count": "3",
     *            "offset": "0",
     *            "limit": "3",
     *            "attributeGroups": [
     *              {
     *                "id": "1B9BADBF-0DD7-AF27-842A-CED56DD773B5",
     *                "name": "ProfileData",
     *                "attributes": [
     *                  {
     *                    "id": "CC2F3750-87C4-316B-AA12-8FE9CB5E8EF4",
     *                    "name": "FirstName"
     *                  }
     *                ]
     *              },
     *              {
     *                "id": "2CA2BA97-959A-9EEC-7BBB-1BBBA5ECA6B5",
     *                "name": "AdditionalData",
     *                "attributes": [
     *                  {
     *                    "id": "431B6B3D-A276-D925-25E4-3A5071C2070C",
     *                    "name": "Skills"
     *                  },
     *                  {
     *                    "id": "7EF984AF-9DD5-58D5-7991-B56AA1CA9CE2",
     *                    "name": "Strength"
     *                  }
     *                ]
     *              },
     *              {
     *                "id": "B3AB16CB-8E3D-36D7-33A6-B59A7CD7C49E",
     *                "name": "Permissions",
     *                "attributes": []
     *                ]
     *              }
     *            ]
     *          }
     *
     * **Example XML (body)**
     *
     *          <?xml version="1.0" encoding="UTF-8"?>
     *            <SsoResponse>
     *              <status><![CDATA[Success]]></status>
     *              <trackid><![CDATA[c0e6840c6b]]></trackid>
     *              <date><![CDATA[2016-05-12T10:00:11+0200]]></date>
     *              <count><![CDATA[3]]></count>
     *              <offset><![CDATA[0]]></offset>
     *              <limit><![CDATA[3]]></limit>
     *              <attributeGroups>
     *                <id><![CDATA[1B9BADBF-0DD7-AF27-842A-CED56DD773B5]]></id>
     *                <name><![CDATA[ProfileData]]></name>
     *                <attributes>
     *                  <attribute>
     *                    <id><![CDATA[CC2F3750-87C4-316B-AA12-8FE9CB5E8EF4]]></id>
     *                    <name><![CDATA[FirstName]]></name>
     *                  </attribute>
     *                </attributes>
     *              </attributeGroups>
     *              <attributeGroups>
     *                <id><![CDATA[2CA2BA97-959A-9EEC-7BBB-1BBBA5ECA6B5]]></id>
     *                <name><![CDATA[AdditionalData]]></name>
     *                <attributes>
     *                  <attribute>
     *                    <id><![CDATA[431B6B3D-A276-D925-25E4-3A5071C2070C]]></id>
     *                    <name><![CDATA[Skills]]></name>
     *                  </attribute>
     *                  <attribute>
     *                    <id><![CDATA[7EF984AF-9DD5-58D5-7991-B56AA1CA9CE2]]></id>
     *                    <name><![CDATA[Strength]]></name>
     *                  </attribute>
     *                </attributes>
     *              </attributeGroups>
     *              <attributeGroups>
     *                <id><![CDATA[B3AB16CB-8E3D-36D7-33A6-B59A7CD7C49E]]></id>
     *                <name><![CDATA[Permissions]]></name>
     *              </attributeGroups>
     *            </SsoResponse>
     *
     * ##Response error
     *
     * **Example (header)**
     *
     *          400 Bad Request
     *
     * **Example JSON (body)**
     *
     *          {
     *            "status": "400",
     *            "message": "Bad Request",
     *            "trackid": "b491280401",
     *            "count": "1",
     *            "errors": [
     *              {
     *                "code": "400",
     *                "message": "ErrorCode: No attribute groups found. ErrorRef: ags001 ShortMessage: No attribute groups found.",
     *                "exception": "external"
     *              }
     *            ]
     *          }
     *
     *
     * **Example XML (body)**
     *
     *          <?xml version="1.0" encoding="UTF-8"?>
     *            <SsoResponse>
     *              <status><![CDATA[400]]></status>
     *              <message><![CDATA[Bad Request]]></message>
     *              <trackid><![CDATA[7d0b555825]]></trackid>
     *              <count><![CDATA[1]]></count>
     *              <errors>
     *                <error>
     *                  <code><![CDATA[400]]></code>
     *                  <message><![CDATA[ErrorCode: No attribute groups found. ErrorRef: ags001 ShortMessage: No attribute groups found.]]></message>
     *                  <exception><![CDATA[external]]></exception>
     *                </error>
     *              </errors>
     *            </SsoResponse>
     *
     *
     *
     * @ApiDoc(
     *  resource="/api/{version}/sso-ws-object/attributeGroup",
     *  description="secured: Returns a list of attribute groups from sso-object-ws",
     *  headers={
     *      {"name"="username", "description"="sso username"},
     *      {"name"="useridentifier", "description"="sso useridentifier"},
     *      {"name"="servicetoken", "description"="servicetoken"}
     *  },
     *  statusCodes={
     *    200="OK",
     *    400="Bad Request",
     *    401="Unauthorized",
     *    403="Forbidden",
     *    404="Not Found",
     *    500="Internal Server Error"
     *  },
     *  responseMap={
     *      200 = "Sso\RestBundle\ModelRest\Response\IdpWs\ServiceUser\Get\UserIdentifier",
     *      400 = {
     *          "class"="Sso\RestBundle\ModelRest\Response\ObjectWs\Version1\AttributeGroup\Get\Error\Response"
     *      }
     *  },
     *  input="Sso\RestBundle\ModelRest\Request\ObjectWs\Version1\AttributeGroup\Show\Request",
     *  tags={"stable" = "#000"},
     *  views = { "sso-ws-object" }
     *  )
     *
     * @Get("/v1/sso-ws-object/attributeGroup/show", name="_sso-ws-object_v1", options={ "method_prefix" = true })
     * @Secure(roles="ROLE_API")
     * @LsCache(cacheTTL=300, keyHeader=true)
     * @param Request $request
     * @return View
     */
    public function getAttributeGroupAction(Request $request)
    {
        return $this->woM->objectWs()->version1()->attributeGroup()->show()->init($request);
    }

    /**
     *
     * *This service deletes single or multiple attribute groups from the object-webservice by its ids.
     * The service expects a JSON or XML in body, and a header authentication.*
     *
     * #Request (header)
     *
     *      Key                 | Value               | Description
     *      ------------------- | ------------------- | --------------------------------------
     *      username            |  (string)           | your username
     *      useridentifier      |  (string)           | your useridentifier for given username
     *      servicetoken        |  (string)           | servicetoken from ws
     *
     * #Request (body)
     *
     *
     * **Request Example JSON**
     *
     *       {
     *         "attributeGroup": [
     *           {
     *             "id": "42D9AB31-49E2-4AF3-B23B-81D3A344BE37"
     *           },
     *           {
     *             "id": "1B9BADBF-0DD7-AF27-842A-CED56DD773B5",
     *             "deleteReferences": true
     *           }
     *         ]
     *       }
     *
     * **Request Example XML**
     *
     *
     *       <?xml version="1.0"?>
     *         <attributeGroup>
     *           <item>
     *             <id>42D9AB31-49E2-4AF3-B23B-81D3A344BE37</id>
     *           </item>
     *           <item>
     *             <id>1B9BADBF-0DD7-AF27-842A-CED56DD773B5</id>
     *             <deleteReferences>false</deleteReferences>
     *           </item>
     *         </attributeGroup>
     *
     *
     * #Response success
     *
     * **Example (header)**
     *
     *     200 OK
     *
     *
     * **Example JSON (body)**
     *
     *      {
     *        "attributeGroup": [
     *          {
     *            "status": "200",
     *            "trackId": "b45d14ed3a",
     *            "id": "42D9AB31-49E2-4AF3-B23B-81D3A344BE37"
     *          },
     *          {
     *            "status": "200",
     *            "trackId": "6c4c00c486",
     *            "id": "B3AB16CB-8E3D-36D7-33A6-B59A7CD7C49E"
     *          }
     *        ]
     *      }
     *
     * **Example XML (body)**
     *
     *      <?xml version="1.0" encoding="UTF-8"?>
     *        <attributeGroup>
     *          <attributeGroup>
     *            <item>
     *              <status><![CDATA[200]]></status>
     *              <trackId><![CDATA[845908f592]]></trackId>
     *              <id><![CDATA[42D9AB31-49E2-4AF3-B23B-81D3A344BE37]]></id>
     *            </item>
     *            <item>
     *              <status><![CDATA[200]]></status>
     *              <trackId><![CDATA[e829e38460]]></trackId>
     *              <id><![CDATA[1B9BADBF-0DD7-AF27-842A-CED56DD773B5]]></id>
     *            </item>
     *          </attributeGroup>
     *        </attributeGroup>
     *
     *
     * #Response error
     *
     * **Example JSON (body)**
     *
     *      {
     *        "attributeGroup": [
     *          {
     *            "status": "200",
     *            "trackId": "75eb4274bb",
     *            "id": "42D9AB31-49E2-4AF3-B23B-81D3A344BE37"
     *          },
     *          {
     *            "status": "400",
     *            "id": "B3AB16CB-8E3D-36D7-33A6-B59A7CD7C49E",
     *            "error": {
     *              "status": 400,
     *              "message": "Bad Request",
     *              "trackid": "24e506b04c",
     *              "count": 1,
     *              "errors": [
     *                {
     *                  "code": 400,
     *                  "message": "ErrorCode: Can not delete AttributeGroup ErrorRef: agd002 ShortMessage: AttributeGroup has referenced Attributes!",
     *                  "exception": "external"
     *                }
     *              ]
     *            }
     *          },
     *          {
     *            "status": "400",
     *            "id": "UNKNOWNN-8E3D-36D7-33A6-B59A7CD7C49E",
     *            "error": {
     *              "status": 400,
     *              "message": "Bad Request",
     *              "trackid": "8ff721c4fa",
     *              "count": 1,
     *              "errors": [
     *                {
     *                  "code": 400,
     *                  "message": "ErrorCode: attributeGroup not found ErrorRef: agd001 ShortMessage: attributeGroup not found",
     *                  "exception": "external"
     *                }
     *              ]
     *            }
     *          }
     *        ]
     *      }
     *
     * **Example XML (body)**
     *
     *      <?xml version="1.0" encoding="UTF-8"?>
     *        <attributeGroup>
     *          <attributeGroup>
     *            <item>
     *              <status><![CDATA[200]]></status>
     *              <trackId><![CDATA[b2b0bdbf3a]]></trackId>
     *              <id><![CDATA[42D9AB31-49E2-4AF3-B23B-81D3A344BE37]]></id>
     *            </item>
     *            <item>
     *              <status><![CDATA[400]]></status>
     *              <id><![CDATA[B3AB16CB-8E3D-36D7-33A6-B59A7CD7C49E]]></id>
     *              <error>
     *                <status>400</status>
     *                <message><![CDATA[Bad Request]]></message>
     *                <trackid><![CDATA[d685982b1a]]></trackid>
     *                <count>1</count>
     *                <errors>
     *                  <error>
     *                    <code>400</code>
     *                    <message><![CDATA[ErrorCode: Can not delete AttributeGroup ErrorRef: agd002 ShortMessage: AttributeGroup has referenced Attributes!]]></message>
     *                    <exception><![CDATA[external]]></exception>
     *                  </error>
     *                </errors>
     *              </error>
     *            </item>
     *            <item>
     *              <status><![CDATA[400]]></status>
     *              <id><![CDATA[UNKNOWNN-8E3D-36D7-33A6-B59A7CD7C49E]]></id>
     *              <error>
     *                <status>400</status>
     *                <message><![CDATA[Bad Request]]></message>
     *                <trackid><![CDATA[ba3ab814b7]]></trackid>
     *                <count>1</count>
     *                <errors>
     *                  <error>
     *                    <code>400</code>
     *                    <message><![CDATA[ErrorCode: attributeGroup not found ErrorRef: agd001 ShortMessage: attributeGroup not found]]></message>
     *                    <exception><![CDATA[external]]></exception>
     *                  </error>
     *                </errors>
     *              </error>
     *            </item>
     *          </attributeGroup>
     *        </attributeGroup>
     *
     *
     * @ApiDoc(
     *  resource="/api/{version}/sso-ws-object/attributeGroup",
     *  description="secured: Deletes attribute groups from sso-object-ws",
     *  headers={
     *      {"name"="username", "description"="sso username", "required"=true},
     *      {"name"="useridentifier", "description"="sso useridentifier", "required"=true},
     *      {"name"="servicetoken", "description"="servicetoken", "required"=true}
     *  },
     *  statusCodes={
     *    200="OK",
     *    400="Bad Request",
     *    401="Unauthorized",
     *    403="Forbidden",
     *    404="Not Found",
     *    500="Internal Server Error"
     *  },
     *  tags={
     *      "stable" = "#000"
     *  },
     *  views = { "sso-ws-object" }
     *
     * )
     * @Delete("/v1/sso-ws-object/attributeGroup/delete", name="_sso-ws-object_v1", options={ "method_prefix" = true })
     * @Secure(roles="ROLE_API")
     *
     * @param Request $request
     * @return View
     */
    public function deleteAttributeGroupAction(Request $request)
    {
        return $this->woM->objectWs()->version1()->attributeGroup()->delete()->init($request);
    }

    /**
     *
     * *This service adds attribute groups in the sso-object-ws. The service expects a JSON or XML in body and a header authentication.*
     *
     * #Request (header)
     *
     *      Key                 | Value               | Description
     *      ------------------- | ------------------- | --------------------------------------
     *      username            |  (string)           | your username
     *      useridentifier      |  (string)           | your useridentifier for given username
     *      servicetoken        |  (string)           | servicetoken from ws
     *
     * ##Request
     *
     * **Request Example JSON**
     *
     *       {
     *           "groups":[
     *               {
     *                   "name":"testGroup",
     *                   "types":[
     *                       "113cc5d4-be89-4151-919d-1f290094075a",
     *                       "57eac39b-69ea-4c35-8ba4-5884a3f9efe5"
     *                   ]
     *               }
     *           ]
     *       }
     *
     * **Request Example XML**
     *
     *       <?xml version="1.0" encoding="UTF-8"?>
     *       <attributeGroupAdd>
     *           <groups>
     *               <attributeGroupType>
     *                   <name>testGroup</name>
     *                   <types>
     *                       <id>113cc5d4-be89-4151-919d-1f290094075a</id>
     *                       <id>57eac39b-69ea-4c35-8ba4-5884a3f9efe5</id>
     *                   </types>
     *               </attributeGroupType>
     *           </groups>
     *       </attributeGroupAdd>
     *
     * #Response success
     *
     * **Example JSON (body)**
     *
     *       {
     *           "code": 200,
     *           "status": "OK",
     *           "trackId": "db805109d0",
     *           "date": "2016-12-07T14:30:38+01:00",
     *           "scriptTimeSec": 2.0452,
     *           "attributeGroup": [
     *               {
     *                   "code": 200,
     *                   "status": "OK",
     *                   "id": "520f90c8-217a-47df-a40d-ef39a96c5252",
     *                   "name": "testGroup"
     *               }
     *           ]
     *       }
     *
     * **Example XML (body)**
     *
     *       <?xml version="1.0" encoding="UTF-8"?>
     *       <attributeGroupAdd>
     *           <code>200</code>
     *           <status><![CDATA[OK]]></status>
     *           <trackId><![CDATA[537caea8d7]]></trackId>
     *           <date><![CDATA[2016-12-07T14:16:58+01:00]]></date>
     *           <scriptTimeSec>4.8701</scriptTimeSec>
     *           <attributeGroup>
     *               <attributeType>
     *                   <code>200</code>
     *                   <status><![CDATA[OK]]></status>
     *                   <id><![CDATA[f7303f08-256c-4fca-91f7-33cc25aafb2c]]></id>
     *                   <name><![CDATA[testGroup]]></name>
     *               </attributeType>
     *           </attributeGroup>
     *       </attributeGroupAdd>
     *
     * #Response error
     *
     * **Example JSON (body)**
     *
     *       {
     *           "code": 400,
     *           "status": "Bad Request",
     *           "trackId": "db805109d0",
     *           "date": "2016-12-07T14:30:38+01:00",
     *           "scriptTimeSec": 2.0452,
     *           "attributeGroup": [
     *               {
     *                   "code": 400,
     *                   "status": "Bad Request",
     *                   "errors": [
     *                       {
     *                           "code": 400,
     *                           "message": "ErrorCode: crit1 ErrorRef: crit1 ShortMessage: Internal error",
     *                           "exception": "external"
     *                       }
     *                   ]
     *               }
     *           ]
     *       }
     *
     * **Example XML (body)**
     *
     *       <?xml version="1.0" encoding="UTF-8"?>
     *       <attributeGroupAdd>
     *           <code>400</code>
     *           <status><![CDATA[Bad Request]]></status>
     *           <trackId><![CDATA[537caea8d7]]></trackId>
     *           <date><![CDATA[2016-12-07T14:16:58+01:00]]></date>
     *           <scriptTimeSec>4.8701</scriptTimeSec>
     *           <attributeGroup>
     *               <attributeType>
     *                   <code>400</code>
     *                   <status><![CDATA[Bad Request]]></status>
     *                   <errors>
     *                       <error>
     *                           <code>400</code>
     *                           <message><![CDATA[ErrorCode: crit1 ErrorRef: crit1 ShortMessage: Internal error]]></message>
     *                           <exception><![CDATA[external]]></exception>
     *                       </error>
     *                   </errors>
     *               </attributeType>
     *           </attributeGroup>
     *       </attributeGroupAdd>
     *
     * ##Response error status 401, 416 or 500
     *
     * **Example JSON (body)**
     *
     *       {
     *       "code": 416,
     *       "status": "Not Found",
     *       "trackId": "2d008cc4c5",
     *       "scriptTimeSec": 1.521,
     *       "count": 1,
     *       "errors": [
     *          {
     *          "code": 416,
     *          "status": "Not Found",
     *          "message": "no valid json string",
     *          "type": "external",
     *          "exception": "InvalidRequest",
     *          "debug": "not set"
     *          }
     *         ]
     *       }
     *
     * **Example XML (body)**
     *
     *      <?xml version="1.0" encoding="UTF-8"?>
     *      <response>
     *          <code>416</code>
     *          <status><![CDATA[Not Found]]></status>
     *          <trackId><![CDATA[16fe523df4]]></trackId>
     *          <scriptTimeSec>2.4459</scriptTimeSec>
     *          <count>1</count>
     *          <errors>
     *              <error>
     *                <code>416</code>
     *                <status><![CDATA[Not Found]]></status>
     *                <message><![CDATA[expected '>']]></message>
     *                <type><![CDATA[external]]></type>
     *                <exception><![CDATA[InvalidRequest]]></exception>
     *                <debug><![CDATA[not set]]></debug>
     *              </error>
     *          </errors>
     *      </response>
     *
     * @ApiDoc(
     *  resource="/api/{version}/sso-ws-object/attributeGroup",
     *  description="secured: Adds multiple attribute groups in sso-object-ws",
     *  statusCodes={
     *         200="Returned when successful",
     *         204="Returned when no content was send",
     *         400="Returned when an error occurred",
     *         401="Unauthorized - Wrong credentials",
     *         403="Returned when the user is not authorized to do this request",
     *         416="Returned when the request is malformed",
     *         500="Returned on internal server errors"
     *  },
     *  headers={
     *      {
     *        "name"="username",
     *        "description"="sso username",
     *        "required"=true
     *       },
     *      {
     *        "name"="useridentifier",
     *        "description"="sso useridentifier",
     *        "required"=true
     *       },
     *      {
     *        "name"="servicetoken",
     *        "description"="servicetoken",
     *        "required"=true
     *       }
     *  },
     *  tags={
     *      "stable" = "#000"
     *  },
     *  output="Sso\RestBundle\ModelRest\Response\ObjectWs\Version1\AttributeGroup\Add\Response",
     *  views = { "sso-ws-object" }
     * )
     *
     * @Post("/v1/sso-ws-object/attributeGroup/add", name="_sso-ws_v1", options={ "method_prefix" = true })
     * @Secure(roles="ROLE_API")
     * @param Request $request
     * @return View
     */
    public function postAttributeGroupAction(Request $request)
    {
        return $this->woM->objectWs()->version1()->attributeGroup()->add()->init($request);
    }

    /**
     *
     * *This service updates attribute groups in the sso-object-ws. The service expects a JSON or XML in body and a header authentication.*
     *
     * #Request (header)
     *
     *      Key                 | Value               | Description
     *      ------------------- | ------------------- | --------------------------------------
     *      username            |  (string)           | your username
     *      useridentifier      |  (string)           | your useridentifier for given username
     *      servicetoken        |  (string)           | servicetoken from ws
     *
     * ##Request
     *
     * **Request Example JSON**
     *
     *       {
     *           "attributeGroups":[
     *               {
     *                   "id": "fc6b3de7-c24f-42ce-842e-cf1c8e136d0c",
     *                   "name":"testGroup",
     *                   "add":[ (optional)
     *                       "113cc5d4-be89-4151-919d-1f290094075a",
     *                       "57eac39b-69ea-4c35-8ba4-5884a3f9efe5"
     *                   ],
     *                   "delete":[ (optional)
     *                       "113cc5d4-be89-4151-919d-1f290094075a",
     *                       "57eac39b-69ea-4c35-8ba4-5884a3f9efe5"
     *                   ],
     *                   "deleteReferences": true (optional)
     *               }
     *           ]
     *       }
     *
     * **Request Example XML**
     *
     *       <?xml version="1.0" encoding="UTF-8"?>
     *       <attributeGroupUpdate>
     *           <attributeGroups>
     *               <attributeGroup>
     *                   <id>fc6b3de7-c24f-42ce-842e-cf1c8e136d0c</id>
     *                   <name>testGroup</name>
     *                   <add> (optional)
     *                       <id>113cc5d4-be89-4151-919d-1f290094075a</id>
     *                       <id>57eac39b-69ea-4c35-8ba4-5884a3f9efe5</id>
     *                   </add>
     *                   <delete> (optional)
     *                       <id>113cc5d4-be89-4151-919d-1f290094075a</id>
     *                       <id>57eac39b-69ea-4c35-8ba4-5884a3f9efe5</id>
     *                   </delete>
     *                   <deleteReferences>true (optional)</deleteReferences>
     *               </attributeGroup>
     *           </attributeGroups>
     *       </attributeGroupUpdate>
     *
     * #Response success
     *
     * **Example JSON (body)**
     *
     *       {
     *           "code": 200,
     *           "status": "OK",
     *           "trackId": "db805109d0",
     *           "date": "2016-12-07T14:30:38+01:00",
     *           "scriptTimeSec": 2.0452,
     *           "attributeGroups": [
     *               {
     *                   "code": 200,
     *                   "status": "OK",
     *                   "id": "520f90c8-217a-47df-a40d-ef39a96c5252",
     *                   "name": "testGroup",
     *                   "ids": [ (optional)
     *                       "229692eb-1848-4c6d-bb44-a87fe30b13e4",
     *                       "bd79944f-900d-4fa9-8acd-bef0a62988bf"
     *                   ]
     *               }
     *           ]
     *       }
     *
     * **Example XML (body)**
     *
     *       <?xml version="1.0" encoding="UTF-8"?>
     *       <attributeGroupUpdate>
     *           <code>200</code>
     *           <status><![CDATA[OK]]></status>
     *           <trackId><![CDATA[537caea8d7]]></trackId>
     *           <date><![CDATA[2016-12-07T14:16:58+01:00]]></date>
     *           <scriptTimeSec>4.8701</scriptTimeSec>
     *           <attributeGroup>
     *               <attributeType>
     *                   <code>200</code>
     *                   <status><![CDATA[OK]]></status>
     *                   <id><![CDATA[f7303f08-256c-4fca-91f7-33cc25aafb2c]]></id>
     *                   <name><![CDATA[testGroup]]></name>
     *                   <ids> (optional)
     *                       <id><![CDATA[229692eb-1848-4c6d-bb44-a87fe30b13e4]]></id>
     *                       <id><![CDATA[bd79944f-900d-4fa9-8acd-bef0a62988bf]]></id>
     *                   </ids>
     *               </attributeType>
     *           </attributeGroup>
     *       </attributeGroupUpdate>
     *
     * #Response error
     *
     * **Example JSON (body)**
     *
     *       {
     *           "code": 400,
     *           "status": "Bad Request",
     *           "trackId": "db805109d0",
     *           "date": "2016-12-07T14:30:38+01:00",
     *           "scriptTimeSec": 2.0452,
     *           "attributeGroup": [
     *               {
     *                   "code": 400,
     *                   "status": "Bad Request",
     *                   "errors": [
     *                       {
     *                           "code": 400,
     *                           "message": "ErrorCode: crit1 ErrorRef: crit1 ShortMessage: Internal error",
     *                           "exception": "external"
     *                       }
     *                   ]
     *               }
     *           ]
     *       }
     *
     * **Example XML (body)**
     *
     *       <?xml version="1.0" encoding="UTF-8"?>
     *       <attributeGroupUpdate>
     *           <code>400</code>
     *           <status><![CDATA[Bad Request]]></status>
     *           <trackId><![CDATA[537caea8d7]]></trackId>
     *           <date><![CDATA[2016-12-07T14:16:58+01:00]]></date>
     *           <scriptTimeSec>4.8701</scriptTimeSec>
     *           <attributeGroup>
     *               <attributeType>
     *                   <code>400</code>
     *                   <status><![CDATA[Bad Request]]></status>
     *                   <errors>
     *                       <error>
     *                           <code>400</code>
     *                           <message><![CDATA[ErrorCode: crit1 ErrorRef: crit1 ShortMessage: Internal error]]></message>
     *                           <exception><![CDATA[external]]></exception>
     *                       </error>
     *                   </errors>
     *               </attributeType>
     *           </attributeGroup>
     *       </attributeGroupUpdate>
     *
     * ##Response error status 401, 416 or 500
     *
     * **Example JSON (body)**
     *
     *       {
     *       "code": 416,
     *       "status": "Not Found",
     *       "trackId": "2d008cc4c5",
     *       "scriptTimeSec": 1.521,
     *       "count": 1,
     *       "errors": [
     *          {
     *          "code": 416,
     *          "status": "Not Found",
     *          "message": "no valid json string",
     *          "type": "external",
     *          "exception": "InvalidRequest",
     *          "debug": "not set"
     *          }
     *         ]
     *       }
     *
     * **Example XML (body)**
     *
     *      <?xml version="1.0" encoding="UTF-8"?>
     *      <response>
     *          <code>416</code>
     *          <status><![CDATA[Not Found]]></status>
     *          <trackId><![CDATA[16fe523df4]]></trackId>
     *          <scriptTimeSec>2.4459</scriptTimeSec>
     *          <count>1</count>
     *          <errors>
     *              <error>
     *                <code>416</code>
     *                <status><![CDATA[Not Found]]></status>
     *                <message><![CDATA[expected '>']]></message>
     *                <type><![CDATA[external]]></type>
     *                <exception><![CDATA[InvalidRequest]]></exception>
     *                <debug><![CDATA[not set]]></debug>
     *              </error>
     *          </errors>
     *      </response>
     *
     * @ApiDoc(
     *  resource="/api/{version}/sso-ws-object/attributeGroup",
     *  description="secured: Updates multiple attribute groups in sso-object-ws",
     *  statusCodes={
     *         200="Returned when successful",
     *         204="Returned when no content was send",
     *         400="Returned when an error occurred",
     *         401="Unauthorized - Wrong credentials",
     *         403="Returned when the user is not authorized to do this request",
     *         416="Returned when the request is malformed",
     *         500="Returned on internal server errors"
     *  },
     *  headers={
     *      {
     *        "name"="username",
     *        "description"="sso username",
     *        "required"=true
     *       },
     *      {
     *        "name"="useridentifier",
     *        "description"="sso useridentifier",
     *        "required"=true
     *       },
     *      {
     *        "name"="servicetoken",
     *        "description"="servicetoken",
     *        "required"=true
     *       }
     *  },
     *  tags={
     *      "stable" = "#000"
     *  },
     *  output="Sso\RestBundle\ModelRest\Response\ObjectWs\Version1\AttributeGroup\Update\Response",
     *  views = { "sso-ws-object" }
     * )
     *
     * @Put("/v1/sso-ws-object/attributeGroup/update", name="_sso-ws_v1", options={ "method_prefix" = true })
     * @Secure(roles="ROLE_API")
     * @param Request $request
     * @return View
     */
    public function putAttributeGroupAction(Request $request)
    {
        return $this->woM->objectWs()->version1()->attributeGroup()->update()->init($request);
    }
}
