<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2018 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\RestBundle\Controller\ObjectWs\Version1;

use FOS\RestBundle\View\View;
use JMS\SecurityExtraBundle\Annotation\Secure;
use Nelmio\ApiDocBundle\Annotation\ApiDoc;
use FOS\RestBundle\Controller\Annotations\NamePrefix;
use Symfony\Component\HttpFoundation\Request;
use FOS\RestBundle\Controller\FOSRestController;
use Sso\RestBundle\Worker\Worker as Worker;
use JMS\DiExtraBundle\Annotation as DI;
use FOS\RestBundle\Controller\Annotations\Get;
use LifeStyle\Tools\CachingBundle\Annotations\LsCache;

/**
 * Class LanguageController
 *
 * @NamePrefix("api_rest_")
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package    Sso\RestBundle\Controller
 */
class LanguageController extends FOSRestController
{

    /**
     * @DI\Inject("rest_worker_manager")
     * @var Worker
     */
    protected $woM;

    /**
     *
     * This service is used to get all or a single language by its iso code.
     * Returns a list of languages.
     *
     * #Request (header)
     *
     *      Key                 | Value               | Description
     *      ------------------- | ------------------- | --------------------------------------
     *      username            |  (string)           | your username
     *      useridentifier      |  (string)           | your useridentifier for given username
     *      servicetoken        |  (string)           | servicetoken from ws
     *
     * ##Request
     *
     * **Request example JSON**
     *
     *          /api/v1/language.json?code=YOURID
     *
     * **Request example XML**
     *
     *          /api/v1/language.xml?code=YOURID
     *
     * ##Response success
     *
     * **Example (header)**
     *
     *          200 OK
     *
     * **Example JSON (body)**
     *
     *          {
     *            "status": "Success",
     *            "trackid": "b9fdc7b443",
     *            "date": "2016-04-26T16:36:45+0200",
     *            "count": "344",
     *            "languages": [
     *              {
     *                "code": "aa",
     *                "name": "Afar"
     *              }
     *            ]
     *          }
     *
     * **Example XML (body)**
     *
     *          <?xml version="1.0" encoding="UTF-8"?>
     *            <SsoResponse>
     *              <status><![CDATA[Success]]></status>
     *              <trackid><![CDATA[dbf037ff6d]]></trackid>
     *              <date><![CDATA[2016-04-26T16:38:16+0200]]></date>
     *              <count><![CDATA[344]]></count>
     *              <languages>
     *                <code><![CDATA[aa]]></code>
     *                <name><![CDATA[Afar]]></name>
     *              </languages>
     *            </SsoResponse>
     *
     * ##Response error
     *
     * **Example (header)**
     *
     *          400 Bad Request
     *
     * **Example JSON (body)**
     *
     *          {
     *            "status": "400",
     *            "message": "Bad Request",
     *            "trackid": "b491280401",
     *            "count": "1",
     *            "errors": [
     *              {
     *                "code": "400",
     *                "message": "ErrorCode: No languages found. ErrorRef: os001 ShortMessage: No languages found.",
     *                "exception": "external"
     *              }
     *            ]
     *          }
     *
     *
     * **Example XML (body)**
     *
     *          <?xml version="1.0" encoding="UTF-8"?>
     *            <SsoResponse>
     *              <status><![CDATA[400]]></status>
     *              <message><![CDATA[Bad Request]]></message>
     *              <trackid><![CDATA[115ca5c2c1]]></trackid>
     *              <count><![CDATA[1]]></count>
     *              <errors>
     *                <error>
     *                  <code><![CDATA[400]]></code>
     *                  <message><![CDATA[ErrorCode: No languages found. ErrorRef: os001 ShortMessage: No languages found.]]></message>
     *                  <exception><![CDATA[external]]></exception>
     *                </error>
     *              </errors>
     *            </SsoResponse>
     *
     *
     *
     * @ApiDoc(
     *  resource="/api/{version}/sso-ws-object/language",
     *  description="secured: Returns a list of available languages from sso-object-ws",
     *  statusCodes={
     *         200="Returned when successful",
     *         400="Returned when an error occured",
     *         401="Unauthorized - Wrong credentials",
     *         403="Returned when the user is not authorized to do this request",
     *         500="Returned on internal server errors"
     *  },
     *  input="Sso\RestBundle\ModelRest\Request\ObjectWs\Version1\Language\Show\Request",
     *  tags={
     *      "stable" = "#000"
     *  },
     *  headers={
     *    {"name"="username", "description"="sso username", "required"=true},
     *    {"name"="useridentifier", "description"="sso useridentifier", "required"=true},
     *    {"name"="servicetoken", "description"="servicetoken", "required"=true}
     *   },
     *  statusCodes={
     *    200="OK",
     *    400="Bad Request",
     *    401="Unauthorized",
     *    403="Forbidden",
     *    404="Not Found",
     *    500="Internal Server Error"
     *  },
     *  views = { "sso-ws-object" }
     * )
     *
     * @Get("/v1/sso-ws-object/language/show", name="_sso-ws-object_v1", options={ "method_prefix" = true })
     * @Secure(roles="ROLE_API")
     * @LsCache(cacheTTL=300, keyHeader=true)
     * @param Request $request
     * @return View
     */
    public function getLanguageAction(Request $request)
    {
        return $this->woM->objectWs()->version1()->language()->show()->init($request);
    }
}
