<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2018 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\RestBundle\Controller\ObjectWs\Version1;

use FOS\RestBundle\View\View;
use JMS\SecurityExtraBundle\Annotation\Secure;
use Nelmio\ApiDocBundle\Annotation\ApiDoc;
use LifeStyle\Tools\RestAuthBundle\Security\Authentication\Token\UidentifyToken;
use FOS\RestBundle\Controller\Annotations\NamePrefix;
use Symfony\Component\HttpFoundation\Request;
use FOS\RestBundle\Controller\FOSRestController;
use Sso\RestBundle\Worker\Worker as Worker;
use JMS\DiExtraBundle\Annotation as DI;
use FOS\RestBundle\Controller\Annotations\Get;
use FOS\RestBundle\Controller\Annotations\Post;
use FOS\RestBundle\Controller\Annotations\Put;
use FOS\RestBundle\Controller\Annotations\Delete;
use LifeStyle\Tools\CachingBundle\Annotations\LsCache;

/**
 * Class ObjectController
 *
 * @NamePrefix("api_rest_")
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package    Sso\RestBundle\Controller
 */
class ObjectController extends FOSRestController
{

    /**
     * @DI\Inject("rest_worker_manager")
     * @var Worker
     */
    protected $woM;

    /**
     *
     * @var UidentifyToken
     */
    protected $userToken;

    /**
     * This service adds objects to the sso-object-ws, do not forget to add the object type to your request. The service expects a JSON in body, and a header authentication.
     *
     * #Request (header)
     *
     *      Key                 | Value               | Description
     *      ------------------- | ------------------- | --------------------------------------
     *      username            |  (string)           | your username
     *      useridentifier      |  (string)           | your useridentifier for given username
     *      servicetoken        |  (string)           | servicetoken from ws
     *
     * ##Request
     *
     * **Request Example JSON**
     *
     *       {
     *           "objects": [
     *               {
     *                   "referenceId": "S0000034",
     *                   "typeId": "E89FB00B-6873-0584-6F1E-7035C3B611E9",
     *                   "parentGuid": "60647a81-82f5-4c17-8da0-176f5906dc0d",
     *                   "name": "Test123",
     *                   "active": 1
     *               }
     *           ]
     *       }
     *
     * **Request Example XML**
     *
     *       <?xml version="1.0" encoding="UTF-8"?>
     *       <objectAdd>
     *           <objects>
     *               <object>
     *                   <referenceId>S0000034</referenceId>
     *                   <typeId>E89FB00B-6873-0584-6F1E-7035C3B611E9</typeId>
     *                   <parentGuid>60647a81-82f5-4c17-8da0-176f5906dc0d</parentGuid>
     *                   <name>Test123</name>
     *                   <active>1</active>
     *               </object>
     *           </objects>
     *       </objectAdd>
     *
     * #Response success
     *
     * **Example JSON (body)**
     *
     *       {
     *           "code": 200,
     *           "status": "OK",
     *           "trackId": "42345581de",
     *           "date": "2017-02-23T11:29:53+01:00",
     *           "scriptTimeSec": 20.5511,
     *           "objects": [
     *               {
     *                   "code": 200,
     *                   "status": "OK",
     *                   "guid": "b61a2a73-ce82-4f2c-835b-7d034f348255",
     *                   "referenceId": "S0010005",
     *                   "typeId": "113cc5d4-be89-4151-919d-1f290094075a",
     *                   "parentGuid": "60647a81-82f5-4c17-8da0-176f5906dc0d",
     *                   "name": "Test123",
     *                   "created": "2017-02-23T11:29:54+01:00",
     *                   "updated": "2017-02-23T11:29:54+01:00",
     *                   "active": 1
     *               }
     *           ]
     *       }
     *
     * **Example XML (body)**
     *
     *       <?xml version="1.0" encoding="UTF-8"?>
     *       <objectAdd>
     *           <code>200</code>
     *           <status><![CDATA[OK]]></status>
     *           <trackId><![CDATA[2ab7a8abeb]]></trackId>
     *           <date><![CDATA[2017-02-23T11:31:32+01:00]]></date>
     *           <scriptTimeSec>22.8348</scriptTimeSec>
     *           <objects>
     *               <object>
     *                   <code>200</code>
     *                   <status><![CDATA[OK]]></status>
     *                   <guid><![CDATA[60647a81-82f5-4c17-8da0-176f5906dc0d]]></guid>
     *                   <referenceId><![CDATA[S0010006]]></referenceId>
     *                   <typeId><![CDATA[113cc5d4-be89-4151-919d-1f290094075a]]></typeId>
     *                   <parentGuid><![CDATA[60647a81-82f5-4c17-8da0-176f5906dc0d]]></parentGuid>
     *                   <name><![CDATA[Test123]]></name>
     *                   <created><![CDATA[2017-02-23T11:31:33+01:00]]></created>
     *                   <updated><![CDATA[2017-02-23T11:31:33+01:00]]></updated>
     *                   <active>1</active>
     *               </object>
     *           </objects>
     *       </objectAdd>
     *
     * #Response error
     *
     * **Example JSON (body)**
     *
     *       {
     *           "code": 400,
     *           "status": "Bad Request",
     *           "trackId": "7c2465ba52",
     *           "date": "2017-02-23T11:01:39+01:00",
     *           "scriptTimeSec": 8.0223,
     *           "objects": [
     *               {
     *                   "code": 400,
     *                   "status": "Bad Request",
     *                   "errors": [
     *                       {
     *                           "code": 400,
     *                           "message": "ErrorCode: type not found ErrorRef: v001 ShortMessage: type not found",
     *                           "exception": "external"
     *                       }
     *                   ]
     *               }
     *           ]
     *       }
     *
     * **Example XML (body)**
     *
     *       <?xml version="1.0" encoding="UTF-8"?>
     *       <objectAdd>
     *           <code>400</code>
     *           <status><![CDATA[Bad Request]]></status>
     *           <trackId><![CDATA[463dd032cc]]></trackId>
     *           <date><![CDATA[2017-02-23T11:17:14+01:00]]></date>
     *           <scriptTimeSec>2.5947</scriptTimeSec>
     *           <objects>
     *               <object>
     *                   <code>400</code>
     *                   <status><![CDATA[Bad Request]]>
     *                   </status>
     *                   <errors>
     *                       <error>
     *                           <code>400</code>
     *                           <message><![CDATA[ErrorCode: type not found ErrorRef: v001 ShortMessage: type not found]]></message>
     *                           <exception><![CDATA[external]]></exception>
     *                       </error>
     *                   </errors>
     *               </object>
     *           </objects>
     *       </objectAdd>
     *
     * ##Response error status 401, 416 or 500
     *
     * **Example JSON (body)**
     *
     *       {
     *       "code": 416,
     *       "status": "Not Found",
     *       "trackId": "2d008cc4c5",
     *       "scriptTimeSec": 1.521,
     *       "count": 1,
     *       "errors": [
     *          {
     *          "code": 416,
     *          "status": "Not Found",
     *          "message": "no valid json string",
     *          "type": "external",
     *          "exception": "InvalidRequest",
     *          "debug": "not set"
     *          }
     *         ]
     *       }
     *
     * **Example XML (body)**
     *
     *      <?xml version="1.0" encoding="UTF-8"?>
     *      <response>
     *          <code>416</code>
     *          <status><![CDATA[Not Found]]></status>
     *          <trackId><![CDATA[16fe523df4]]></trackId>
     *          <scriptTimeSec>2.4459</scriptTimeSec>
     *          <count>1</count>
     *          <errors>
     *              <error>
     *                <code>416</code>
     *                <status><![CDATA[Not Found]]></status>
     *                <message><![CDATA[expected '>']]></message>
     *                <type><![CDATA[external]]></type>
     *                <exception><![CDATA[InvalidRequest]]></exception>
     *                <debug><![CDATA[not set]]></debug>
     *              </error>
     *          </errors>
     *      </response>
     *
     * @ApiDoc(
     *  resource="/api/{version}/sso-ws-object/object",
     *  description="secured: Adds multiple objects to sso-ws-object",
     *  statusCodes={
     *         200="Returned when successful",
     *         204="Returned when no content was send",
     *         400="Returned when an error occurred",
     *         401="Unauthorized - Wrong credentials",
     *         403="Returned when the user is not authorized to do this request",
     *         416="Returned when the request is malformed",
     *         500="Returned on internal server errors"
     *  },
     *  headers={
     *      {
     *        "name"="username",
     *        "description"="sso username",
     *        "required"=true
     *       },
     *      {
     *        "name"="useridentifier",
     *        "description"="sso useridentifier",
     *        "required"=true
     *       },
     *      {
     *        "name"="servicetoken",
     *        "description"="servicetoken",
     *        "required"=true
     *       }
     *  },
     *  tags={
     *      "stable" = "#000"
     *  },
     *  output="Sso\RestBundle\ModelRest\Response\ObjectWs\Version1\Object\Add\Response",
     *  views = { "sso-ws-object" }
     * )
     *
     * @Post("/v1/sso-ws-object/object/add", name="_sso-ws-object_v1", options={ "method_prefix" = true })
     * @Secure(roles="ROLE_API")
     * @param Request $request
     * @return View
     */
    public function postObjectAction(Request $request)
    {
        return $this->woM->objectWs()->version1()->object()->add()->init($request);
    }

    /**
     *
     * This service is used to get all or a single object by its guid without all children.
     * Returns a list of objects without children.
     *
     *#Request (header)
     *
     *      Key                 | Value               | Description
     *      ------------------- | ------------------- | --------------------------------------
     *      username            |  (string)           | your username
     *      useridentifier      |  (string)           | your useridentifier for given username
     *      servicetoken        |  (string)           | servicetoken from ws
     *
     * ##Request
     *
     * **Request example JSON**
     *
     *          /api/v1/object/single.json?guid=YOURGUID&referenceId=YOURREFID&offset=0&limit=15&languageCode=en
     *
     * **Request example XML**
     *
     *          /api/v1/object/single.xml?guid=YOURGUID&referenceId=YOURREFID&offset=0&limit=15&languageCode=en
     *
     * ##Response success
     *
     * **Example (header)**
     *
     *          200 OK
     *
     * **Example JSON (body)**
     *
     *          {
     *            "status": "Success",
     *            "trackid": "015a97a7ea",
     *            "date": "2016-04-11T13:51:33+0200",
     *            "count": "26",
     *            "objects": [
     *              {
     *                "typeId": "BB25734E-BFF8-878A-E4F7-B2F0F38142F7",
     *                "typeName": "TestTypeLocation",
     *                "guid": "16FA40FD-5F4B-4937-F53E-F598471AAFD8",
     *                "referenceId": "S001",
     *                "name": "ParentObject",
     *                "created": "2016-01-05T17:28:32+0100",
     *                "updated": "2016-04-11T13:51:34+0200",
     *                "attributes": []
     *              },
     *              {
     *                "typeId": "BB25734E-BFF8-878A-E4F7-B2F0F38142F7",
     *                "typeName": "TestTypeLocation",
     *                "guid": "1ABE53B9-1926-14E0-B77D-1F8FE316263E",
     *                "referenceId": "S002",
     *                "parentGuid": "69388074-D68A-5DCC-7B41-43826B7DF795",
     *                "name": "Object3",
     *                "created": "2016-02-08T09:53:18+0100",
     *                "updated": "2016-04-11T13:51:35+0200",
     *                "attributes": [
     *                  {
     *                    "groupName": "TestGroup",
     *                    "attributeName": "TestAttribute5",
     *                    "attributeValue": "testde",
     *                    "languageCode": "de"
     *                  }
     *                ]
     *              },
     *              {
     *                "typeId": "BB25734E-BFF8-878A-E4F7-B2F0F38142F7",
     *                "typeName": "TestTypeLocation",
     *                "guid": "344F1443-F8DF-EB28-5907-1DEA62C5A938",
     *                "referenceId": "S003",
     *                "parentGuid": "69388074-D68A-5DCC-7B41-43826B7DF795",
     *                "name": "Object9",
     *                "created": "2016-02-08T09:55:33+0100",
     *                "updated": "2016-04-11T13:51:35+0200",
     *                "attributes": []
     *              }
     *            ]
     *          }
     *
     * **Example XML (body)**
     *
     *          <?xml version="1.0" encoding="UTF-8"?>
     *            <SsoResponse>
     *              <status><![CDATA[Success]]></status>
     *              <trackid><![CDATA[e90a6ccdb3]]></trackid>
     *              <date><![CDATA[2016-04-11T13:46:33+0200]]></date>
     *              <count><![CDATA[26]]></count>
     *              <objects>
     *                <typeId><![CDATA[BB25734E-BFF8-878A-E4F7-B2F0F38142F7]]></typeId>
     *                <typeName><![CDATA[TestTypeLocation]]></typeName>
     *                <guid><![CDATA[16FA40FD-5F4B-4937-F53E-F598471AAFD8]]></guid>
     *                <referenceId><![CDATA[S001]]></referenceId>
     *                <name><![CDATA[ParentObject]]></name>
     *                <created><![CDATA[2016-01-05T17:28:32+0100]]></created>
     *                <updated><![CDATA[2016-04-11T13:46:34+0200]]></updated>
     *              </objects>
     *              <objects>
     *                <typeId><![CDATA[BB25734E-BFF8-878A-E4F7-B2F0F38142F7]]></typeId>
     *                <typeName><![CDATA[TestTypeLocation]]></typeName>
     *                <guid><![CDATA[1ABE53B9-1926-14E0-B77D-1F8FE316263E]]></guid>
     *                <referenceId><![CDATA[S002]]></referenceId>
     *                <parentGuid><![CDATA[69388074-D68A-5DCC-7B41-43826B7DF795]]></parentGuid>
     *                <name><![CDATA[Object3]]></name>
     *                <created><![CDATA[2016-02-08T09:53:18+0100]]></created>
     *                <updated><![CDATA[2016-04-11T13:46:34+0200]]></updated>
     *                <attributes>
     *                  <attribute>
     *                    <groupName><![CDATA[TestGroup]]></groupName>
     *                    <attributeName><![CDATA[TestAttribute5]]></attributeName>
     *                    <attributeValue><![CDATA[testde]]></attributeValue>
     *                    <languageCode><![CDATA[de]]></languageCode>
     *                  </attribute>
     *                </attributes>
     *              </objects>
     *              <objects>
     *                <typeId><![CDATA[BB25734E-BFF8-878A-E4F7-B2F0F38142F7]]></typeId>
     *                <typeName><![CDATA[TestTypeLocation]]></typeName>
     *                <guid><![CDATA[344F1443-F8DF-EB28-5907-1DEA62C5A938]]></guid>
     *                <referenceId><![CDATA[S003]]></referenceId>
     *                <parentGuid><![CDATA[69388074-D68A-5DCC-7B41-43826B7DF795]]></parentGuid>
     *                <name><![CDATA[Object9]]></name>
     *                <created><![CDATA[2016-02-08T09:55:33+0100]]></created>
     *                <updated><![CDATA[2016-04-11T13:46:34+0200]]></updated>
     *              </objects>
     *            </SsoResponse>
     *
     * ##Response error
     *
     * **Example (header)**
     *
     *          400 Bad Request
     *
     * **Example JSON (body)**
     *
     *          {
     *            "status": 400,
     *            "message": "Bad Request",
     *            "trackid": "3c3a14b6d2",
     *            "count": 1,
     *            "errors": [
     *              {
     *                "code": 400,
     *                "message": "ErrorCode: x000 ErrorRef: xh10 ShortMessage: Invalid xml",
     *                "exception": "external"
     *              }
     *            ]
     *          }
     *
     *
     * **Example XML (body)**
     *
     *          <?xml version="1.0" encoding="UTF-8"?>
     *          <SsoResponse>
     *            <status>400</status>
     *            <message><![CDATA[Bad Request]]></message>
     *            <trackid><![CDATA[de6a82e4db]]></trackid>
     *            <count>1</count>
     *            <errors>
     *              <error>
     *                <code>400</code>
     *                <message><![CDATA[ErrorCode: x000 ErrorRef: xh10 ShortMessage: Invalid xml]]></message>
     *                <exception><![CDATA[external]]></exception>
     *              </error>
     *            </errors>
     *          </SsoResponse>
     *
     *
     *
     * @ApiDoc(
     *  resource="/api/{version}/sso-ws-object/object",
     *  description="secured: Returns a list of objects from sso-object-ws",
     *  statusCodes={
     *         200="Returned when successful",
     *         400="Returned when an error occured",
     *         401="Unauthorized - Wrong credentials",
     *         403="Returned when the user is not authorized to do this request",
     *         500="Returned on internal server errors"
     *  },
     *  input="Sso\RestBundle\ModelRest\Request\ObjectWs\Version1\Object\Show\Single\Request",
     *  tags={
     *      "stable" = "#000"
     *  },
     *  headers={
     *      {"name"="username", "description"="sso username", "required"=true},
     *      {"name"="useridentifier", "description"="sso useridentifier", "required"=true},
     *      {"name"="servicetoken", "description"="servicetoken", "required"=true}
     *  },
     *  views = { "sso-ws-object" }
     * )
     *
     * @Secure(roles="ROLE_API")
     * @LsCache(cacheTTL=300, keyHeader=true)
     * @Get("/v1/sso-ws-object/object/single", name="_sso-ws-object_v1", options={ "method_prefix" = true })
     * @param Request $request
     * @return View
     */
    public function getObjectSingleAction(Request $request)
    {
        return $this->woM->objectWs()->version1()->object()->show()->single()->init($request);
    }

    /**
     *
     * This service is used to get all or a single object by its guid.
     * Returns a list of objects with all children objects included.
     *
     *#Request (header)
     *
     *      Key                 | Value               | Description
     *      ------------------- | ------------------- | --------------------------------------
     *      username            |  (string)           | your username
     *      useridentifier      |  (string)           | your useridentifier for given username
     *      servicetoken        |  (string)           | servicetoken from ws
     *
     * ##Request
     *
     * **Request example JSON**
     *
     *          /api/v1/object/multi.json?guid=YOURGUID&referenceId=YOURREFID&offset=0&limit=15&languageCode=en
     *
     * **Request example XML**
     *
     *          /api/v1/object/multi.xml?guid=YOURGUID&referenceId=YOURREFID&offset=0&limit=15&languageCode=en
     *
     * ##Response success
     *
     * **Example (header)**
     *
     *          200 OK
     *
     * **Example JSON (body)**
     *
     *          {
     *            "status": "Success",
     *            "trackid": "3c93cc6410",
     *            "date": "2016-04-11T13:25:58+0200",
     *            "count": "26",
     *            "objects": [
     *              {
     *                "typeId": "BB25734E-BFF8-878A-E4F7-B2F0F38142F7",
     *                "typeName": "TestTypeLocation",
     *                "guid": "16FA40FD-5F4B-4937-F53E-F598471AAFD8",
     *                "referenceId": "S001",
     *                "name": "ParentObject",
     *                "created": "2016-01-05T17:28:32+0100",
     *                "updated": "2016-04-11T13:26:00+0200",
     *                "attributes": [
     *                  {
     *                    "groupName": "TestGroup",
     *                    "attributeName": "New Name3",
     *                    "attributeValue": "asdsafsdfsaf",
     *                    "languageCode": "agq"
     *                  }
     *                ],
     *                "children": [
     *                  {
     *                    "typeId": "BB25734E-BFF8-878A-E4F7-B2F0F38142F7",
     *                    "typeName": "TestTypeLocation",
     *                    "guid": "69388074-D68A-5DCC-7B41-43826B7DF795",
     *                    "referenceId": "S008Updated",
     *                    "parentGuid": "16FA40FD-5F4B-4937-F53E-F598471AAFD8",
     *                    "name": "ParentObjectTest2",
     *                    "created": "2016-02-05T11:16:04+0100",
     *                    "updated": "2016-04-11T13:26:00+0200",
     *                    "attributes": [],
     *                    "children": [
     *                      {
     *                        "typeId": "BB25734E-BFF8-878A-E4F7-B2F0F38142F7",
     *                        "typeName": "TestTypeLocation",
     *                        "guid": "1ABE53B9-1926-14E0-B77D-1F8FE316263E",
     *                        "referenceId": "S002",
     *                        "parentGuid": "69388074-D68A-5DCC-7B41-43826B7DF795",
     *                        "name": "Object3",
     *                        "created": "2016-02-08T09:53:18+0100",
     *                        "updated": "2016-04-11T13:26:00+0200",
     *                        "attributes": [
     *                          {
     *                            "groupName": "TestGroup",
     *                            "attributeName": "TestAttribute5",
     *                            "attributeValue": "testde",
     *                            "languageCode": "de"
     *                          }
     *                        ]
     *                      }
     *                    ]
     *                  }
     *                ]
     *              }
     *            ]
     *          }
     *
     * **Example XML (body)**
     *
     *          <?xml version="1.0" encoding="UTF-8"?>
     *            <SsoResponse>
     *              <status><![CDATA[Success]]></status>
     *              <trackid><![CDATA[bb43badc1f]]></trackid>
     *              <date><![CDATA[2016-04-11T13:42:02+0200]]></date>
     *              <count><![CDATA[26]]></count>
     *              <objects>
     *                <typeId><![CDATA[BB25734E-BFF8-878A-E4F7-B2F0F38142F7]]></typeId>
     *                <typeName><![CDATA[TestTypeLocation]]></typeName>
     *                <guid><![CDATA[16FA40FD-5F4B-4937-F53E-F598471AAFD8]]></guid>
     *                <referenceId><![CDATA[S001]]></referenceId>
     *                <name><![CDATA[ParentObject]]></name>
     *                <created><![CDATA[2016-01-05T17:28:32+0100]]></created>
     *                <updated><![CDATA[2016-04-11T13:42:04+0200]]></updated>
     *                <attributes>
     *                  <attribute>
     *                    <groupName><![CDATA[TestGroup]]></groupName>
     *                    <attributeName><![CDATA[New Name3]]></attributeName>
     *                    <attributeValue><![CDATA[asdsafsdfsaf]]></attributeValue>
     *                    <languageCode><![CDATA[agq]]></languageCode>
     *                  </attribute>
     *                </attributes>
     *                <children>
     *                  <objects>
     *                    <typeId><![CDATA[BB25734E-BFF8-878A-E4F7-B2F0F38142F7]]></typeId>
     *                    <typeName><![CDATA[TestTypeLocation]]></typeName>
     *                    <guid><![CDATA[69388074-D68A-5DCC-7B41-43826B7DF795]]></guid>
     *                    <referenceId><![CDATA[S008Updated]]></referenceId>
     *                    <parentGuid><![CDATA[16FA40FD-5F4B-4937-F53E-F598471AAFD8]]></parentGuid>
     *                    <name><![CDATA[ParentObjectTest2]]></name>
     *                    <created><![CDATA[2016-02-05T11:16:04+0100]]></created>
     *                    <updated><![CDATA[2016-04-11T13:42:04+0200]]></updated>
     *                    <children>
     *                      <objects>
     *                        <typeId><![CDATA[BB25734E-BFF8-878A-E4F7-B2F0F38142F7]]></typeId>
     *                        <typeName><![CDATA[TestTypeLocation]]></typeName>
     *                        <guid><![CDATA[1ABE53B9-1926-14E0-B77D-1F8FE316263E]]></guid>
     *                        <referenceId><![CDATA[S002]]></referenceId>
     *                        <parentGuid><![CDATA[69388074-D68A-5DCC-7B41-43826B7DF795]]></parentGuid>
     *                        <name><![CDATA[Object3]]></name>
     *                        <created><![CDATA[2016-02-08T09:53:18+0100]]></created>
     *                        <updated><![CDATA[2016-04-11T13:42:04+0200]]></updated>
     *                        <attributes>
     *                          <attribute>
     *                            <groupName><![CDATA[TestGroup]]></groupName>
     *                            <attributeName><![CDATA[TestAttribute5]]></attributeName>
     *                            <attributeValue><![CDATA[testde]]></attributeValue>
     *                            <languageCode><![CDATA[de]]></languageCode>
     *                          </attribute>
     *                        </attributes>
     *                      </objects>
     *                    </children>
     *                  </objects>
     *                </children>
     *              </objects>
     *            </SsoResponse>
     *
     * ##Response error
     *
     * **Example (header)**
     *
     *          400 Bad Request
     *
     * **Example JSON (body)**
     *
     *          {
     *            "status": 400,
     *            "message": "Bad Request",
     *            "trackid": "3c3a14b6d2",
     *            "count": 1,
     *            "errors": [
     *              {
     *                "code": 400,
     *                "message": "ErrorCode: x000 ErrorRef: xh10 ShortMessage: Invalid xml",
     *                "exception": "external"
     *              }
     *            ]
     *          }
     *
     *
     * **Example XML (body)**
     *
     *          <?xml version="1.0" encoding="UTF-8"?>
     *          <SsoResponse>
     *            <status>400</status>
     *            <message><![CDATA[Bad Request]]></message>
     *            <trackid><![CDATA[de6a82e4db]]></trackid>
     *            <count>1</count>
     *            <errors>
     *              <error>
     *                <code>400</code>
     *                <message><![CDATA[ErrorCode: x000 ErrorRef: xh10 ShortMessage: Invalid xml]]></message>
     *                <exception><![CDATA[external]]></exception>
     *              </error>
     *            </errors>
     *          </SsoResponse>
     *
     *
     *
     * @ApiDoc(
     *  resource="/api/{version}/sso-ws-object/object",
     *  description="secured: Returns a list of objects from sso-object-ws with children included",
     *  statusCodes={
     *         200="Returned when successful",
     *         400="Returned when an error occured",
     *         401="Unauthorized - Wrong credentials",
     *         403="Returned when the user is not authorized to do this request",
     *         500="Returned on internal server errors"
     *  },
     *  input="Sso\RestBundle\ModelRest\Request\ObjectWs\Version1\Object\Show\Multi\Request",
     *  tags={
     *      "stable" = "#000"
     *  },
     *  headers={
     *      {"name"="username", "description"="sso username", "required"=true},
     *      {"name"="useridentifier", "description"="sso useridentifier", "required"=true},
     *      {"name"="servicetoken", "description"="servicetoken", "required"=true}
     *  },
     *  views = { "sso-ws-object" }
     * )
     *
     * @Secure(roles="ROLE_API")
     * @LsCache(cacheTTL=300, keyHeader=true)
     * @Get("/v1/sso-ws-object/object/multi", name="_sso-ws-object_v1", options={ "method_prefix" = true })
     * @param Request $request
     * @return View
     */
    public function getObjectMultiAction(Request $request)
    {
        return $this->woM->objectWs()->version1()->object()->show()->multi()->init($request);
    }

    /**
     * This service is used to get all or a single object by its guid for being displayed in a tree view.
     * Returns a flat list of all requested objects with a tree-id that can be used to build the tree.
     *
     *#Request (header)
     *
     *      Key                 | Value               | Description
     *      ------------------- | ------------------- | --------------------------------------
     *      username            |  (string)           | your username
     *      useridentifier      |  (string)           | your useridentifier for given username
     *      servicetoken        |  (string)           | servicetoken from ws
     *
     * ##Request
     *
     * **Request example JSON**
     *
     *          /api/v1/object/tree.json?guid=OBJECTGUID&referenceId=OBJECTREFID&offset=0&limit=15
     *
     * **Request example XML**
     *
     *          /api/v1/object/tree.xml?guid=OBJECTGUID&referenceId=OBJECTREFID&offset=0&limit=15
     *
     * ##Response success
     *
     * **Example (header)**
     *
     *          200 OK
     *
     * **Example JSON (body)**
     *
     *         {
     *           "code": 200,
     *           "status": "OK",
     *           "scriptTime": "1.907",
     *           "trackid": "4024ea91a1",
     *           "date": "2016-12-27T10:25:49+01:00",
     *           "count": "853",
     *           "offset": "10",
     *           "limit": "2",
     *           "objects": [
     *             {
     *               "typeName": "DistrictOrUnit",
     *               "guid": "71b4c655-e3db-4cf3-b650-c521e0498452",
     *               "referenceId": "S0000856",
     *               "name": "North East",
     *               "treeId": "1.01.01.02"
     *             },
     *             {
     *               "typeName": "LanguageCenter",
     *               "guid": "06291dfe-1374-4109-a2d4-e62fbfa0ad4c",
     *               "referenceId": "S0000344",
     *               "name": "Dresden",
     *               "treeId": "1.01.01.02.01"
     *             }
     *           ]
     *         }
     *
     *
     * **Example XML (body)**
     *
     *       <?xml version="1.0" encoding="UTF-8"?>
     *       <objectTreeView>
     *           <code>200</code>
     *           <status><![CDATA[OK]]></status>
     *           <trackId><![CDATA[d1c0174096]]></trackId>
     *           <date><![CDATA[2016-12-27T11:49:16+01:00]]></date>
     *           <scriptTimeSec>1.603</scriptTimeSec>
     *           <count><![CDATA[853]]></count>
     *           <offset><![CDATA[0]]></offset>
     *           <limit><![CDATA[2]]></limit>
     *           <objects>
     *               <typeName><![CDATA[Root]]></typeName>
     *               <guid><![CDATA[d106d322-0b49-4cc5-806f-2638e8e19800]]></guid>
     *               <referenceId><![CDATA[S0000001]]></referenceId>
     *               <name><![CDATA[All Sso]]></name>
     *               <treeId><![CDATA[1]]></treeId>
     *           </objects>
     *           <objects>
     *               <typeName><![CDATA[Division]]></typeName>
     *               <guid><![CDATA[37a21be4-5c6a-4cbd-88bf-de8b6783c561]]></guid>
     *               <referenceId><![CDATA[S0000309]]></referenceId>
     *               <name><![CDATA[United Europe]]></name>
     *               <treeId><![CDATA[1.01]]></treeId>
     *           </objects>
     *       </objectTreeView>
     *
     * ##Response error
     *
     * **Example (header)**
     *
     *          400 Bad Request
     *
     * **Example JSON (body)**
     *
     *         {
     *           "code": 400,
     *           "status": "Bad Request",
     *           "trackId": "186ab69a2b",
     *           "count": 1,
     *           "errors": [
     *             {
     *               "code": 400,
     *               "status": "Bad Request",
     *               "message": "guid: This value should have exactly 36 characters.",
     *               "type": "external",
     *               "exception": "not set",
     *               "debug": "not set"
     *             }
     *           ]
     *         }
     *
     *
     * **Example XML (body)**
     *
     *         <?xml version="1.0" encoding="UTF-8"?>
     *         <response>
     *           <code>400</code>
     *           <status><![CDATA[Bad Request]]></status>
     *           <trackId><![CDATA[201b0814b8]]></trackId>
     *           <count>1</count>
     *           <errors>
     *             <error>
     *               <code>400</code>
     *               <status><![CDATA[Bad Request]]></status>
     *               <message><![CDATA[guid: This value should have exactly 36 characters.]]></message>
     *               <type><![CDATA[external]]></type>
     *               <exception><![CDATA[not set]]></exception>
     *               <debug><![CDATA[not set]]></debug>
     *             </error>
     *           </errors>
     *         </response>
     *
     *
     * @ApiDoc(
     *  resource="/api/{version}/sso-ws-object/object",
     *  description="secured: Returns a list of object suited to be displayed in a tree view",
     *  statusCodes={
     *         200="Returned when successful",
     *         400="Returned when an error occured",
     *         401="Unauthorized - Wrong credentials",
     *         403="Returned when the user is not authorized to do this request",
     *         500="Returned on internal server errors"
     *  },
     *  parameters={
     *      {"name"="guid", "dataType"="string", "required"=false, "description"="Object GUID"},
     *      {"name"="referenceId", "dataType"="string", "required"=false, "description"="Object reference ID"},
     *      {"name"="offset", "dataType"="integer", "required"=false, "description"="Offest for getting objects"},
     *      {"name"="limit", "dataType"="integer", "required"=false, "description"="Limit for getting objects"}
     *  },
     *  tags={
     *      "stable" = "#000"
     *  },
     *  headers={
     *      {"name"="username", "description"="sso username", "required"=true},
     *      {"name"="useridentifier", "description"="sso useridentifier", "required"=true},
     *      {"name"="servicetoken", "description"="servicetoken", "required"=true}
     *  },
     *  views = { "sso-ws-object" }
     * )
     *
     * @Secure(roles="ROLE_API")
     * @Get("/v1/sso-ws-object/object/tree", name="_sso-ws-object_v1", options={ "method_prefix" = true })
     * @param Request $request
     * @return View
     */
    public function getObjectTreeAction(Request $request)
    {
        return $this->woM->objectWs()->version1()->object()->show()->tree()->init($request);
    }

    /**
     *
     * This service updates objects in the sso-object-ws. The service expects a JSON or XML in body, and a header authentication.
     *
     * #Request (header)
     *
     *      Key                 | Value               | Description
     *      ------------------- | ------------------- | --------------------------------------
     *      username            |  (string)           | your username
     *      useridentifier      |  (string)           | your useridentifier for given username
     *      servicetoken        |  (string)           | servicetoken from ws
     *
     * ##Request
     *
     * **Request Example JSON**
     *
     *       {
     *           "objects": [
     *               {
     *                   "guid": "60647a81-82f5-4c17-8da0-176f5906dc0d",
     *                   "referenceId": "S0000034",
     *                   "typeId": "E89FB00B-6873-0584-6F1E-7035C3B611E9",
     *                   "parentGuid": "60647a81-82f5-4c17-8da0-176f5906dc0d",
     *                   "name": "Test123",
     *                   "active": 1
     *               }
     *           ]
     *       }
     *
     * **Request Example XML**
     *
     *       <?xml version="1.0" encoding="UTF-8"?>
     *       <objectUpdate>
     *           <objects>
     *               <object>
     *                   <guid>60647a81-82f5-4c17-8da0-176f5906dc0d</guid>
     *                   <referenceId>S0000034</referenceId>
     *                   <typeId>E89FB00B-6873-0584-6F1E-7035C3B611E9</typeId>
     *                   <parentGuid>60647a81-82f5-4c17-8da0-176f5906dc0d</parentGuid>
     *                   <name>Test123</name>
     *                   <active>1</active>
     *               </object>
     *           </objects>
     *       </objectUpdate>
     *
     * #Response success
     *
     * **Example JSON (body)**
     *
     *       {
     *           "code": 200,
     *           "status": "OK",
     *           "trackId": "42345581de",
     *           "date": "2017-02-23T11:29:53+01:00",
     *           "scriptTimeSec": 20.5511,
     *           "objects": [
     *               {
     *                   "code": 200,
     *                   "status": "OK",
     *                   "guid": "b61a2a73-ce82-4f2c-835b-7d034f348255",
     *                   "referenceId": "S0010005",
     *                   "typeId": "113cc5d4-be89-4151-919d-1f290094075a",
     *                   "parentGuid": "60647a81-82f5-4c17-8da0-176f5906dc0d",
     *                   "name": "Test123",
     *                   "created": "2017-02-23T11:29:54+01:00",
     *                   "updated": "2017-02-23T11:29:54+01:00",
     *                   "active": 1
     *               }
     *           ]
     *       }
     *
     * **Example XML (body)**
     *
     *       <?xml version="1.0" encoding="UTF-8"?>
     *       <objectUpdate>
     *           <code>200</code>
     *           <status><![CDATA[OK]]></status>
     *           <trackId><![CDATA[2ab7a8abeb]]></trackId>
     *           <date><![CDATA[2017-02-23T11:31:32+01:00]]></date>
     *           <scriptTimeSec>22.8348</scriptTimeSec>
     *           <objects>
     *               <object>
     *                   <code>200</code>
     *                   <status><![CDATA[OK]]></status>
     *                   <guid><![CDATA[60647a81-82f5-4c17-8da0-176f5906dc0d]]></guid>
     *                   <referenceId><![CDATA[S0010006]]></referenceId>
     *                   <typeId><![CDATA[113cc5d4-be89-4151-919d-1f290094075a]]></typeId>
     *                   <parentGuid><![CDATA[60647a81-82f5-4c17-8da0-176f5906dc0d]]></parentGuid>
     *                   <name><![CDATA[Test123]]></name>
     *                   <created><![CDATA[2017-02-23T11:31:33+01:00]]></created>
     *                   <updated><![CDATA[2017-02-23T11:31:33+01:00]]></updated>
     *                   <active>1</active>
     *               </object>
     *           </objects>
     *       </objectUpdate>
     *
     * #Response error
     *
     * **Example JSON (body)**
     *
     *       {
     *           "code": 400,
     *           "status": "Bad Request",
     *           "trackId": "7c2465ba52",
     *           "date": "2017-02-23T11:01:39+01:00",
     *           "scriptTimeSec": 8.0223,
     *           "objects": [
     *               {
     *                   "code": 400,
     *                   "status": "Bad Request",
     *                   "errors": [
     *                       {
     *                           "code": 400,
     *                           "message": "ErrorCode: objectUpdate01 ErrorRef: v001 ShortMessage: Invalid TypeId",
     *                           "exception": "external"
     *                       }
     *                   ]
     *               }
     *           ]
     *       }
     *
     * **Example XML (body)**
     *
     *       <?xml version="1.0" encoding="UTF-8"?>
     *       <objectUpdate>
     *           <code>400</code>
     *           <status><![CDATA[Bad Request]]></status>
     *           <trackId><![CDATA[463dd032cc]]></trackId>
     *           <date><![CDATA[2017-02-23T11:17:14+01:00]]></date>
     *           <scriptTimeSec>2.5947</scriptTimeSec>
     *           <objects>
     *               <object>
     *                   <code>400</code>
     *                   <status><![CDATA[Bad Request]]>
     *                   </status>
     *                   <errors>
     *                       <error>
     *                           <code>400</code>
     *                           <message><![CDATA[ErrorCode: objectUpdate01 ErrorRef: v001 ShortMessage: Invalid TypeId]]></message>
     *                           <exception><![CDATA[external]]></exception>
     *                       </error>
     *                   </errors>
     *               </object>
     *           </objects>
     *       </objectUpdate>
     *
     * ##Response error status 401, 416 or 500
     *
     * **Example JSON (body)**
     *
     *       {
     *       "code": 416,
     *       "status": "Not Found",
     *       "trackId": "2d008cc4c5",
     *       "scriptTimeSec": 1.521,
     *       "count": 1,
     *       "errors": [
     *          {
     *          "code": 416,
     *          "status": "Not Found",
     *          "message": "no valid json string",
     *          "type": "external",
     *          "exception": "InvalidRequest",
     *          "debug": "not set"
     *          }
     *         ]
     *       }
     *
     * **Example XML (body)**
     *
     *      <?xml version="1.0" encoding="UTF-8"?>
     *      <response>
     *          <code>416</code>
     *          <status><![CDATA[Not Found]]></status>
     *          <trackId><![CDATA[16fe523df4]]></trackId>
     *          <scriptTimeSec>2.4459</scriptTimeSec>
     *          <count>1</count>
     *          <errors>
     *              <error>
     *                <code>416</code>
     *                <status><![CDATA[Not Found]]></status>
     *                <message><![CDATA[expected '>']]></message>
     *                <type><![CDATA[external]]></type>
     *                <exception><![CDATA[InvalidRequest]]></exception>
     *                <debug><![CDATA[not set]]></debug>
     *              </error>
     *          </errors>
     *      </response>
     *
     * @ApiDoc(
     *  resource="/api/{version}/sso-ws-object/object",
     *  description="secured: Updates multiple objects in a sso-ws-object",
     *  statusCodes={
     *         200="Returned when successful",
     *         204="Returned when no content was send",
     *         400="Returned when an error occurred",
     *         401="Unauthorized - Wrong credentials",
     *         403="Returned when the user is not authorized to do this request",
     *         416="Returned when the request is malformed",
     *         500="Returned on internal server errors"
     *  },
     *  headers={
     *      {
     *        "name"="username",
     *        "description"="sso username",
     *        "required"=true
     *       },
     *      {
     *        "name"="useridentifier",
     *        "description"="sso useridentifier",
     *        "required"=true
     *       },
     *      {
     *        "name"="servicetoken",
     *        "description"="servicetoken",
     *        "required"=true
     *       }
     *  },
     *  tags={
     *      "stable" = "#000"
     *  },
     *  output="Sso\RestBundle\ModelRest\Response\ObjectWs\Version1\Object\Update\Response",
     *  views = { "sso-ws-object" }
     * )
     *
     * @Put("/v1/sso-ws-object/object/update", name="_sso-ws-object_v1", options={ "method_prefix" = true })
     * @Secure(roles="ROLE_API")
     * @param Request $request
     * @return View
     */
    public function putObjectAction(Request $request)
    {
        return $this->woM->objectWs()->version1()->object()->update()->init($request);
    }

    /**
     *
     *
     * *This service deletes single or multiple objects from the object-webservice by its guid or by its referenceId.
     * You have to specify guid or referenceId as the identifier - If not, an exception will be thrown.
     * The service expects a JSON or XML in body, and a header authentication.*
     *
     * #Request (header)
     *
     *      Key                 | Value               | Description
     *      ------------------- | ------------------- | --------------------------------------
     *      username            |  (string)           | your username
     *      useridentifier      |  (string)           | your useridentifier for given username
     *      servicetoken        |  (string)           | servicetoken from ws
     *
     * #Request (body)
     *
     *
     * **Request Example JSON**
     *
     *       {
     *         "object": [
     *           {
     *             "guid": "DD216B2D-5F93-48C0-E00F-479CDF36BE49"
     *           },
     *           {
     *             "referenceId": "S012345"
     *           }
     *         ]
     *       }
     *
     * **Request Example XML**
     *
     *
     *       <?xml version="1.0"?>
     *         <object>
     *           <item>
     *             <guid>DD216B2D-5F93-48C0-E00F-479CDF36BE49</guid>
     *           </item>
     *           <item>
     *             <referenceId>S012345</referenceId>
     *           </item>
     *         </object>
     *
     *
     * #Response success
     *
     * **Example (header)**
     *
     *     200 OK
     *
     *
     * **Example JSON (body)**
     *
     *      {
     *        "object": [
     *          {
     *            "status": "200",
     *            "trackId": "1350de63c4",
     *            "guid": "9ED921EA-99D7-E2AA-8589-167661796339"
     *          },
     *          {
     *            "status": "200",
     *            "trackId": "1350de63c4",
     *            "guid": "D400DE84-97B3-C940-06F0-2562AB9DC35B"
     *          }
     *        ]
     *      }
     *
     * **Example XML (body)**
     *
     *      <?xml version="1.0" encoding="UTF-8"?>
     *        <object>
     *          <object>
     *            <item>
     *              <status><![CDATA[200]]></status>
     *              <trackId><![CDATA[760d913ebf]]></trackId>
     *              <guid><![CDATA[9ED921EA-99D7-E2AA-8589-167661796339]]></guid>
     *            </item>
     *            <item>
     *              <status><![CDATA[200]]></status>
     *              <trackId><![CDATA[f2165de107]]></trackId>
     *              <guid><![CDATA[D400DE84-97B3-C940-06F0-2562AB9DC35B]]></guid>
     *            </item>
     *          </object>
     *        </object>
     *
     *
     * #Response error
     *
     * **Example JSON (body)**
     *
     *      {
     *          "object": [
     *            {
     *              "status": "400",
     *              "guid": "BD2DF42A-D7C8-F926-EF78-04783389CB5E",
     *              "attributes": [],
     *              "error": {
     *                "status": 400,
     *                "message": "Bad Request",
     *                "trackid": "922618f61b",
     *                "count": 1,
     *                "errors": [
     *                  {
     *                    "code": 400,
     *                    "message": "ErrorCode: objectDelete01 ErrorRef: v001 ShortMessage: Invalid guid",
     *                    "exception": "external"
     *                  }
     *                ]
     *              }
     *            },
     *            {
     *              "status": "400",
     *              "guid": "D400DE84-97B3-C940-06F0-2562AB9DC35B",
     *              "attributes": [],
     *              "error": {
     *                "status": 400,
     *                "message": "Bad Request",
     *                "trackid": "5500e175ce",
     *                "count": 1,
     *                "errors": [
     *                  {
     *                    "code": 400,
     *                    "message": "ErrorCode: objectDelete01 ErrorRef: v001 ShortMessage: Invalid guid",
     *                    "exception": "external"
     *                  }
     *                ]
     *              }
     *            }
     *          ]
     *        }
     *
     * **Example XML (body)**
     *
     *      <?xml version="1.0" encoding="UTF-8"?>
     *        <object>
     *          <object>
     *            <item>
     *              <status><![CDATA[400]]></status>
     *              <guid><![CDATA[BD2DF42A-D7C8-F926-EF78-04783389CB5E]]></guid>
     *              <error>
     *                <status>400</status>
     *                <message><![CDATA[Bad Request]]></message>
     *                <trackid><![CDATA[bb56855651]]></trackid>
     *                <count>1</count>
     *                <errors>
     *                  <error>
     *                    <code>400</code>
     *                    <message><![CDATA[ErrorCode: object not found ErrorRef: o001 ShortMessage: object not found]]></message>
     *                    <exception><![CDATA[external]]></exception>
     *                  </error>
     *                </errors>
     *              </error>
     *            </item>
     *            <item>
     *              <status><![CDATA[400]]></status>
     *              <guid><![CDATA[D400DE84-97B3-C940-06F0-2562AB9DC35B]]></guid>
     *              <error>
     *                <status>400</status>
     *                <message><![CDATA[Bad Request]]></message>
     *                <trackid><![CDATA[5500e175ce]]></trackid>
     *                <count>1</count>
     *                <errors>
     *                  <error>
     *                    <code>400</code>
     *                    <message><![CDATA[ErrorCode: Object not found ErrorRef: o001 ShortMessage: Object not found]]></message>
     *                    <exception><![CDATA[external]]></exception>
     *                  </error>
     *                </errors>
     *              </error>
     *            </item>
     *          </object>
     *        </object>
     *
     *
     * @ApiDoc(
     *  resource="/api/{version}/sso-ws-object/object",
     *  description="secured: Deletes objects from sso-object-ws",
     *  statusCodes={
     *         200="Returned when successful",
     *         204="Returned when no content was send",
     *         400="Returned when an error occured",
     *         401="Unauthorized - Wrong credentials",
     *         403="Returned when the user is not authorized to do this request",
     *         500="Returned on internal server errors"
     *  },
     *  headers={
     *      {"name"="username", "description"="sso username"},
     *      {"name"="useridentifier", "description"="sso useridentifier"},
     *      {"name"="servicetoken", "description"="servicetoken"}
     *  },
     *  tags={
     *      "stable" = "#000"
     *  },
     *  views = { "sso-ws-object" }
     *
     * )
     *
     *
     * @Secure(roles="ROLE_API")
     * @Delete("/v1/sso-ws-object/object/delete", name="_sso-ws-object_v1", options={ "method_prefix" = true })
     * @param Request $request
     * @return View
     */
    public function deleteObjectAction(Request $request)
    {
        return $this->woM->objectWs()->version1()->object()->deleteObject()->init($request);
    }
}
