<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2018 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\RestBundle\Controller\UserWs\Version1;

use FOS\RestBundle\Controller\Annotations\Delete;
use FOS\RestBundle\Controller\Annotations\Get;
use FOS\RestBundle\Controller\Annotations\Post;
use FOS\RestBundle\View\View;
use JMS\SecurityExtraBundle\Annotation\Secure;
use Nelmio\ApiDocBundle\Annotation\ApiDoc;
use FOS\RestBundle\Controller\Annotations\NamePrefix;
use Symfony\Component\HttpFoundation\Request;
use FOS\RestBundle\Controller\FOSRestController;
use Sso\RestBundle\Worker\Worker as Worker;
use JMS\DiExtraBundle\Annotation as DI;
use LifeStyle\Tools\CachingBundle\Annotations\LsCache;

/**
 * @NamePrefix("api_rest_")
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package    Sso\RestBundle\Controller
 */
class ApplicationAttributeController extends FOSRestController
{
    /**
     * @DI\Inject("rest_worker_manager")
     * @var Worker
     */
    private $woM;

    /**
     *
     * *This service adds multiple attributes to the sso-ws application. The service expects a JSON or XML in body and a header authentication.*
     *
     * #Request (header)
     *
     *      Key                 | Value               | Description
     *      ------------------- | ------------------- | --------------------------------------
     *      username            |  (string)           | your username
     *      useridentifier      |  (string)           | your useridentifier for given username
     *      servicetoken        |  (string)           | servicetoken from ws
     *
     * ##Request
     *
     * **Request Example JSON**
     *
     *       {
     *           "attributes": [
     *               {
     *                   "name": "TestAttribute",
     *                   "type": "one",
     *                   "applicationType": {
     *                       "name": "TestApplication"
     *                   }
     *               }
     *           ]
     *       }
     *
     * **Request Example XML**
     *
     *       <?xml version="1.0" encoding="UTF-8"?>
     *       <applicationAttributeAdd>
     *           <attributes>
     *               <attributeType>
     *                   <name>TestAttribute</name>
     *                   <type>one</type>
     *                   <applicationType>
     *                       <name>TestApplication</name>
     *                   </applicationType>
     *               </attributeType>
     *           </attributes>
     *       </applicationAttributeAdd>
     *
     * #Response success
     *
     * **Example JSON (body)**
     *
     *       {
     *           "code": 200,
     *           "status": "OK",
     *           "trackId": "ee366791a0",
     *           "date": "2016-11-10T10:55:22+01:00",
     *           "scriptTimeSec": 1.1373,
     *           "attributes": [
     *               {
     *                   "code": 200,
     *                   "status": "OK"
     *               }
     *           ]
     *       }
     *
     * **Example XML (body)**
     *
     *       <?xml version="1.0" encoding="UTF-8"?>
     *       <applicationAttributeAdd>
     *           <code>200</code>
     *           <status><![CDATA[OK]]></status>
     *           <trackId><![CDATA[3ddc720b2b]]></trackId>
     *           <date><![CDATA[2016-11-10T10:55:45+01:00]]></date>
     *           <scriptTimeSec>1.1847</scriptTimeSec>
     *           <attributes>
     *               <attributeType>
     *                   <code>200</code>
     *                   <status><![CDATA[OK]]></status>
     *               </attributeType>
     *           </attributes>
     *       </applicationAttributeAdd>
     *
     * #Response error
     *
     * **Example JSON (body)**
     *
     *       {
     *           "code": 400,
     *           "status": "Bad Request",
     *           "trackId": "ee366791a0",
     *           "date": "2016-11-10T10:55:22+01:00",
     *           "scriptTimeSec": 1.1373,
     *           "attributes": [
     *               {
     *                   "code": 400,
     *                   "status": "Bad Request",
     *                   "errors": [
     *                       {
     *                           "code": 400,
     *                           "message": "ErrorCode: at001 ErrorRef: at002 ShortMessage: Invalid attribute",
     *                           "exception": "external"
     *                       }
     *                   ]
     *               }
     *           ]
     *       }
     *
     * **Example XML (body)**
     *
     *       <?xml version="1.0" encoding="UTF-8"?>
     *       <applicationAttributeAdd>
     *           <code>400</code>
     *           <status><![CDATA[Bad Request]]></status>
     *           <trackId><![CDATA[3ddc720b2b]]></trackId>
     *           <date><![CDATA[2016-11-10T10:55:45+01:00]]></date>
     *           <scriptTimeSec>1.1847</scriptTimeSec>
     *           <attributes>
     *               <attributeType>
     *                   <code>400</code>
     *                   <status><![CDATA[Bad Request]]></status>
     *                   <errors>
     *                       <error>
     *                           <code>400</code>
     *                           <message><![CDATA[ErrorCode: at001 ErrorRef: at002 ShortMessage: Invalid attribute]]></message>
     *                           <exception><![CDATA[external]]></exception>
     *                       </error>
     *                   </errors>
     *               </attributeType>
     *           </attributes>
     *       </applicationAttributeAdd>
     *
     * ##Response error status 401, 416 or 500
     *
     * **Example JSON (body)**
     *
     *       {
     *       "code": 416,
     *       "status": "Not Found",
     *       "trackId": "2d008cc4c5",
     *       "scriptTimeSec": 1.521,
     *       "count": 1,
     *       "errors": [
     *          {
     *          "code": 416,
     *          "status": "Not Found",
     *          "message": "no valid json string",
     *          "type": "external",
     *          "exception": "InvalidRequest",
     *          "debug": "not set"
     *          }
     *         ]
     *       }
     *
     * **Example XML (body)**
     *
     *      <?xml version="1.0" encoding="UTF-8"?>
     *      <response>
     *          <code>416</code>
     *          <status><![CDATA[Not Found]]></status>
     *          <trackId><![CDATA[16fe523df4]]></trackId>
     *          <scriptTimeSec>2.4459</scriptTimeSec>
     *          <count>1</count>
     *          <errors>
     *              <error>
     *                <code>416</code>
     *                <status><![CDATA[Not Found]]></status>
     *                <message><![CDATA[expected '>']]></message>
     *                <type><![CDATA[external]]></type>
     *                <exception><![CDATA[InvalidRequest]]></exception>
     *                <debug><![CDATA[not set]]></debug>
     *              </error>
     *          </errors>
     *      </response>
     *
     * @ApiDoc(
     *  resource="/api/{version}/sso-ws/application/attribute",
     *  description="secured: Adds multiple attributes to sso-ws application",
     *  statusCodes={
     *         200="Returned when successful",
     *         204="Returned when no content was send",
     *         400="Returned when an error occurred",
     *         401="Unauthorized - Wrong credentials",
     *         403="Returned when the user is not authorized to do this request",
     *         416="Returned when the request is malformed",
     *         500="Returned on internal server errors"
     *  },
     *  headers={
     *      {
     *        "name"="username",
     *        "description"="sso username",
     *        "required"=true
     *       },
     *      {
     *        "name"="useridentifier",
     *        "description"="sso useridentifier",
     *        "required"=true
     *       },
     *      {
     *        "name"="servicetoken",
     *        "description"="servicetoken",
     *        "required"=true
     *       }
     *   },
     *  tags={
     *      "stable" = "#000"
     *  },
     *  output="Sso\RestBundle\ModelRest\Response\UserWs\Version1\ApplicationAttribute\Add\Response",
     *  views = { "sso-ws" }
     * )
     *
     * @Post("/v1/sso-ws/application/attribute/add", name="_sso-ws_v1", options={ "method_prefix" = true })
     * @Secure(roles="ROLE_API")
     * @param Request $request
     * @return View
     */
    public function postApplicationAttributeAction(Request $request)
    {
        return $this->woM->userWs()->version1()->applicationAttribute()->add()->init($request);
    }

    /**
     *
     * *This service returns attribute informations. The service expects a JSON or XML in body and a header authentication.*
     *
     * #Request (header)
     *
     *      Key                 | Value               | Description
     *      ------------------- | ------------------- | --------------------------------------
     *      username            |  (string)           | your username
     *      useridentifier      |  (string)           | your useridentifier for given username
     *      servicetoken        |  (string)           | servicetoken from ws
     *
     * #Response success
     *
     * **Example JSON (body)**
     *
     *       {
     *           "code": 200,
     *           "status": "OK",
     *           "trackId": "a8580dc2ce",
     *           "date": "2016-11-14T14:33:59+01:00",
     *           "scriptTimeSec": 4.6347,
     *           "attributes": [
     *               {
     *                   "code": 200,
     *                   "status": "OK",
     *                   "name": "TestAttribute",
     *                   "type": "one"
     *               }
     *           ]
     *       }
     *
     * **Example XML (body)**
     *
     *       <?xml version="1.0" encoding="UTF-8"?>
     *       <applicationAttribute>
     *           <code>200</code>
     *           <status><![CDATA[OK]]></status>
     *           <trackId><![CDATA[5be83f8da3]]></trackId>
     *           <date><![CDATA[2016-11-14T14:35:05+01:00]]></date>
     *           <scriptTimeSec>1.1464</scriptTimeSec>
     *           <attributes>
     *               <attributeType>
     *                   <code>200</code>
     *                   <status><![CDATA[OK]]></status>
     *                   <name><![CDATA[TestAttribute]]></name>
     *                   <type><![CDATA[one]]></type>
     *               </attributeType>
     *           </attributes>
     *       </applicationAttribute>
     *
     * #Response error
     *
     * **Example JSON (body)**
     *
     *       {
     *           "code": 400,
     *           "status": "Bad Request",
     *           "trackId": "f29e5fcde2",
     *           "date": "2016-11-14T14:36:21+01:00",
     *           "scriptTimeSec": 5.0705,
     *           "attributes": [
     *               {
     *                   "code": 400,
     *                   "status": "Bad Request",
     *                   "errors": [
     *                       {
     *                           "code": 400,
     *                           "message": "ErrorCode: a001 ErrorRef: at201 ShortMessage: Invalid application",
     *                           "exception": "external"
     *                       }
     *                   ]
     *               }
     *           ]
     *       }
     *
     * **Example XML (body)**
     *
     *       <?xml version="1.0" encoding="UTF-8"?>
     *       <applicationAttribute>
     *           <code>400</code>
     *           <status><![CDATA[Bad Request]]></status>
     *           <trackId><![CDATA[4a85908d20]]></trackId>
     *           <date><![CDATA[2016-11-14T14:37:15+01:00]]></date>
     *           <scriptTimeSec>4.4376</scriptTimeSec>
     *           <attributes>
     *               <attributeType>
     *                   <code>400</code>
     *                   <status><![CDATA[Bad Request]]></status>
     *                   <errors>
     *                       <error>
     *                           <code>400</code>
     *                           <message><![CDATA[ErrorCode: a001 ErrorRef: at201 ShortMessage: Invalid application]]></message>
     *                           <exception><![CDATA[external]]></exception>
     *                       </error>
     *                   </errors>
     *               </attributeType>
     *           </attributes>
     *       </applicationAttribute>
     *
     * ##Response error status 401, 416 or 500
     *
     * **Example JSON (body)**
     *
     *       {
     *       "code": 416,
     *       "status": "Not Found",
     *       "trackId": "2d008cc4c5",
     *       "scriptTimeSec": 1.521,
     *       "count": 1,
     *       "errors": [
     *          {
     *          "code": 416,
     *          "status": "Not Found",
     *          "message": "no valid json string",
     *          "type": "external",
     *          "exception": "InvalidRequest",
     *          "debug": "not set"
     *          }
     *         ]
     *       }
     *
     * **Example XML (body)**
     *
     *      <?xml version="1.0" encoding="UTF-8"?>
     *      <response>
     *          <code>416</code>
     *          <status><![CDATA[Not Found]]></status>
     *          <trackId><![CDATA[16fe523df4]]></trackId>
     *          <scriptTimeSec>2.4459</scriptTimeSec>
     *          <count>1</count>
     *          <errors>
     *              <error>
     *                <code>416</code>
     *                <status><![CDATA[Not Found]]></status>
     *                <message><![CDATA[expected '>']]></message>
     *                <type><![CDATA[external]]></type>
     *                <exception><![CDATA[InvalidRequest]]></exception>
     *                <debug><![CDATA[not set]]></debug>
     *              </error>
     *          </errors>
     *      </response>
     *
     * @ApiDoc(
     *  resource="/api/{version}/sso-ws/application/attribute",
     *  description="secured: Gets attribute informations from sso-ws application",
     *  statusCodes={
     *         200="Returned when successful",
     *         204="Returned when no content was send",
     *         400="Returned when an error occurred",
     *         401="Unauthorized - Wrong credentials",
     *         403="Returned when the user is not authorized to do this request",
     *         416="Returned when the request is malformed",
     *         500="Returned on internal server errors"
     *  },
     *  headers={
     *      {
     *        "name"="username",
     *        "description"="sso username",
     *        "required"=true
     *       },
     *      {
     *        "name"="useridentifier",
     *        "description"="sso useridentifier",
     *        "required"=true
     *       },
     *      {
     *        "name"="servicetoken",
     *        "description"="servicetoken",
     *        "required"=true
     *       }
     *  },
     *  parameters={
     *      {"name"="attributeName", "dataType"="string", "required"=true, "description"="Attribute name"},
     *      {"name"="applicationName", "dataType"="string", "required"=true, "description"="Application name"}
     *  },
     *  tags={
     *      "stable" = "#000"
     *  },
     *  output="Sso\RestBundle\ModelRest\Response\UserWs\Version1\ApplicationAttribute\Show\Response",
     *  views = { "sso-ws" }
     * )
     *
     * @Get("/v1/sso-ws/application/attribute", name="_sso-ws_v1", options={ "method_prefix" = true })
     * @Secure(roles="ROLE_API")
     * @LsCache(cacheTTL=300, keyHeader=true)
     * @param Request $request
     * @return View
     */
    public function getApplicationAttributeAction(Request $request)
    {
        return $this->woM->userWs()->version1()->applicationAttribute()->show()->init($request);
    }

    /**
     *
     * *This service deletes attributes. The service expects a JSON or XML in body and a header authentication.*
     *
     * #Request (header)
     *
     *      Key                 | Value               | Description
     *      ------------------- | ------------------- | --------------------------------------
     *      username            |  (string)           | your username
     *      useridentifier      |  (string)           | your useridentifier for given username
     *      servicetoken        |  (string)           | servicetoken from ws
     *
     * ##Request
     *
     * **Request Example JSON**
     *
     *       {
     *           "attributes": [
     *               {
     *                   "name": "testAttribute",
     *                   "applicationType": {
     *                       "name": "testApplication"
     *                   }
     *               }
     *           ]
     *       }
     *
     * **Request Example XML**
     *
     *       <?xml version="1.0" encoding="UTF-8"?>
     *       <applicationAttributeDelete>
     *           <attributes>
     *               <attributeType>
     *                   <name>testAttribute</name>
     *                   <applicationType>
     *                       <name>testApplication</name>
     *                   </applicationType>
     *               </attributeType>
     *           </attributes>
     *       </applicationAttributeDelete>
     *
     * #Response success
     *
     * **Example JSON (body)**
     *
     *       {
     *           "code": 200,
     *           "status": "OK",
     *           "trackId": "8b5b8e49ea",
     *           "date": "2016-11-16T09:08:15+01:00",
     *           "scriptTimeSec": 5.7169,
     *           "attributes": [
     *               {
     *                   "code": 200,
     *                   "status": "OK",
     *                   "message": "Attribute \"testAttribute\" deleted"
     *               }
     *           ]
     *       }
     *
     * **Example XML (body)**
     *
     *       <?xml version="1.0" encoding="UTF-8"?>
     *       <applicationAttributeDelete>
     *           <code>200</code>
     *           <status><![CDATA[OK]]></status>
     *           <trackId><![CDATA[b1138e8562]]></trackId>
     *           <date><![CDATA[2016-11-16T09:09:17+01:00]]></date>
     *           <scriptTimeSec>4.6424</scriptTimeSec>
     *           <attributes>
     *               <attributeType>
     *                   <code>200</code>
     *                   <status><![CDATA[OK]]></status>
     *                   <message><![CDATA[Attribute "testAttribute" deleted]]></message>
     *               </attributeType>
     *           </attributes>
     *       </applicationAttributeDelete>
     *
     * #Response error
     *
     * **Example JSON (body)**
     *
     *       {
     *           "code": 400,
     *           "status": "Bad Request",
     *           "trackId": "c043714ecf",
     *           "date": "2016-11-16T09:10:17+01:00",
     *           "scriptTimeSec": 5.0844,
     *           "attributes": [
     *               {
     *                   "code": 400,
     *                   "status": "Bad Request",
     *                   "errors": [
     *                       {
     *                           "code": 400,
     *                           "message": "ErrorCode: at002 ErrorRef: at102 ShortMessage: Invalid attribute",
     *                           "exception": "external"
     *                       }
     *                   ]
     *               }
     *           ]
     *       }
     *
     * **Example XML (body)**
     *
     *       <?xml version="1.0" encoding="UTF-8"?>
     *       <applicationAttributeDelete>
     *           <code>400</code>
     *           <status><![CDATA[Bad Request]]></status>
     *           <trackId><![CDATA[5aa722e5830]]></trackId>
     *           <date><![CDATA[2016-11-16T09:10:55+01:00]]></date>
     *           <scriptTimeSec>4.6388</scriptTimeSec>
     *           <attributes>
     *               <attributeType>
     *                   <code>400</code>
     *                   <status><![CDATA[Bad Request]]></status>
     *                   <errors>
     *                       <error>
     *                           <code>400</code>
     *                           <message><![CDATA[ErrorCode: at002 ErrorRef: at102 ShortMessage: Invalid attribute]]></message>
     *                           <exception><![CDATA[external]]></exception>
     *                       </error>
     *                   </errors>
     *               </attributeType>
     *           </attributes>
     *       </applicationAttributeDelete>
     *
     * ##Response error status 401, 416 or 500
     *
     * **Example JSON (body)**
     *
     *       {
     *       "code": 416,
     *       "status": "Not Found",
     *       "trackId": "2d008cc4c5",
     *       "scriptTimeSec": 1.521,
     *       "count": 1,
     *       "errors": [
     *          {
     *          "code": 416,
     *          "status": "Not Found",
     *          "message": "no valid json string",
     *          "type": "external",
     *          "exception": "InvalidRequest",
     *          "debug": "not set"
     *          }
     *         ]
     *       }
     *
     * **Example XML (body)**
     *
     *      <?xml version="1.0" encoding="UTF-8"?>
     *      <response>
     *          <code>416</code>
     *          <status><![CDATA[Not Found]]></status>
     *          <trackId><![CDATA[16fe523df4]]></trackId>
     *          <scriptTimeSec>2.4459</scriptTimeSec>
     *          <count>1</count>
     *          <errors>
     *              <error>
     *                <code>416</code>
     *                <status><![CDATA[Not Found]]></status>
     *                <message><![CDATA[expected '>']]></message>
     *                <type><![CDATA[external]]></type>
     *                <exception><![CDATA[InvalidRequest]]></exception>
     *                <debug><![CDATA[not set]]></debug>
     *              </error>
     *          </errors>
     *      </response>
     *
     * @ApiDoc(
     *  resource="/api/{version}/sso-ws/application/attribute",
     *  description="secured: Deletes given attributes",
     *  statusCodes={
     *         200="Returned when successful",
     *         204="Returned when no content was send",
     *         400="Returned when an error occurred",
     *         401="Unauthorized - Wrong credentials",
     *         403="Returned when the user is not authorized to do this request",
     *         416="Returned when the request is malformed",
     *         500="Returned on internal server errors"
     *  },
     *  headers={
     *      {
     *        "name"="username",
     *        "description"="sso username",
     *        "required"=true
     *       },
     *      {
     *        "name"="useridentifier",
     *        "description"="sso useridentifier",
     *        "required"=true
     *       },
     *      {
     *        "name"="servicetoken",
     *        "description"="servicetoken",
     *        "required"=true
     *       }
     *  },
     *  tags={
     *      "stable" = "#000"
     *  },
     *  output="Sso\RestBundle\ModelRest\Response\UserWs\Version1\ApplicationAttribute\Delete\Response",
     *  views = { "sso-ws" }
     * )
     *
     * @Delete("/v1/sso-ws/application/attribute/delete", name="_sso-ws_v1", options={ "method_prefix" = true })
     * @Secure(roles="ROLE_API")
     * @param Request $request
     * @return View
     */
    public function deleteApplicationAttributeAction(Request $request)
    {
        return $this->woM->userWs()->version1()->applicationAttribute()->delete()->init($request);
    }
}
