<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2018 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\RestBundle\Controller\UserWs\Version1;

use FOS\RestBundle\Controller\Annotations\Get;
use FOS\RestBundle\Controller\Annotations\Post;
use FOS\RestBundle\View\View;
use JMS\SecurityExtraBundle\Annotation\Secure;
use Nelmio\ApiDocBundle\Annotation\ApiDoc;
use FOS\RestBundle\Controller\Annotations\NamePrefix;
use Symfony\Component\HttpFoundation\Request;
use FOS\RestBundle\Controller\FOSRestController;
use Sso\RestBundle\Worker\Worker as Worker;
use JMS\DiExtraBundle\Annotation as DI;
use FOS\RestBundle\Controller\Annotations\Delete;
use FOS\RestBundle\Controller\Annotations\Put;
use LifeStyle\Tools\CachingBundle\Annotations\LsCache;

/**
 * Class ApplicationController
 *
 * @NamePrefix("api_rest_")
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package    Sso\RestBundle\Controller
 */
class ApplicationController extends FOSRestController
{
    /**
     * @DI\Inject("rest_worker_manager")
     * @var Worker
     */
    private $woM;

    /**
     *
     * *This service adds multiple applications to the sso-ws. The service expects a JSON or XML in body and a header authentication.*
     *
     * #Request (header)
     *
     *      Key                 | Value               | Description
     *      ------------------- | ------------------- | --------------------------------------
     *      username            |  (string)           | your username
     *      useridentifier      |  (string)           | your useridentifier for given username
     *      servicetoken        |  (string)           | servicetoken from ws
     *
     * ##Request
     *
     * **Request Example JSON**
     *
     *       {
     *           "applications": [
     *               {
     *                   "name": "testApplication",
     *                   "description": "Test description"
     *               }
     *           ]
     *       }
     *
     * **Request Example XML**
     *
     *       <applicationAdd>
     *           <applications>
     *               <application>
     *                   <name>testApplication</name>
     *                   <description>Test description</description>
     *               </application>
     *           </applications>
     *       </applicationAdd>
     *
     * #Response success
     *
     * **Example JSON (body)**
     *
     *       {
     *           "code": 200,
     *           "status": "OK",
     *           "trackId": "eb5bf0b149",
     *           "date": "2016-11-15T12:40:43+01:00",
     *           "scriptTimeSec": 4.3627,
     *           "applications": [
     *               {
     *                   "code": 200,
     *                   "status": "OK",
     *                   "message": "Application added"
     *               }
     *           ]
     *       }
     *
     * **Example XML (body)**
     *
     *       <?xml version="1.0" encoding="UTF-8"?>
     *       <applicationAdd>
     *           <code>200</code>
     *           <status><![CDATA[OK]]></status>
     *           <trackId><![CDATA[14fc29cbdc]]></trackId>
     *           <date><![CDATA[2016-11-15T12:39:27+01:00]]></date>
     *           <scriptTimeSec>4.972</scriptTimeSec>
     *           <applications>
     *               <application>
     *                   <code>200</code>
     *                   <status><![CDATA[OK]]></status>
     *                   <message><![CDATA[Application added]]></message>
     *               </application>
     *           </applications>
     *       </applicationAdd>
     *
     * #Response error
     *
     * **Example JSON (body)**
     *
     *       {
     *           "code": 400,
     *           "status": "Bad Request",
     *           "trackId": "7c39d356e0",
     *           "date": "2016-11-15T12:30:31+01:00",
     *           "scriptTimeSec": 1.638,
     *           "applications": [
     *               {
     *                   "code": 400,
     *                   "status": "Bad Request",
     *                   "errors": [
     *                       {
     *                           "code": 400,
     *                           "message": "ErrorCode: a10 ErrorRef: v001 ShortMessage: Invalid application-name",
     *                           "exception": "external"
     *                       }
     *                   ]
     *               }
     *           ]
     *       }
     *
     * **Example XML (body)**
     *
     *       <?xml version="1.0" encoding="UTF-8"?>
     *       <applicationAdd>
     *           <code>400</code>
     *           <status><![CDATA[Bad Request]]></status>
     *           <trackId><![CDATA[5a7f343b34]]></trackId>
     *           <date><![CDATA[2016-11-15T12:28:46+01:00]]></date>
     *           <scriptTimeSec>1.8849</scriptTimeSec>
     *           <applications>
     *               <application>
     *                   <code>400</code>
     *                   <status><![CDATA[Bad Request]]></status>
     *                   <errors>
     *                       <error>
     *                           <code>400</code>
     *                           <message><![CDATA[ErrorCode: a10 ErrorRef: v001 ShortMessage: Invalid application-name]]></message>
     *                           <exception><![CDATA[external]]></exception>
     *                       </error>
     *                   </errors>
     *               </application>
     *           </applications>
     *       </applicationAdd>
     *
     * ##Response error status 401, 416 or 500
     *
     * **Example JSON (body)**
     *
     *       {
     *       "code": 416,
     *       "status": "Not Found",
     *       "trackId": "2d008cc4c5",
     *       "scriptTimeSec": 1.521,
     *       "count": 1,
     *       "errors": [
     *          {
     *          "code": 416,
     *          "status": "Not Found",
     *          "message": "no valid json string",
     *          "type": "external",
     *          "exception": "InvalidRequest",
     *          "debug": "not set"
     *          }
     *         ]
     *       }
     *
     * **Example XML (body)**
     *
     *      <?xml version="1.0" encoding="UTF-8"?>
     *      <response>
     *          <code>416</code>
     *          <status><![CDATA[Not Found]]></status>
     *          <trackId><![CDATA[16fe523df4]]></trackId>
     *          <scriptTimeSec>2.4459</scriptTimeSec>
     *          <count>1</count>
     *          <errors>
     *              <error>
     *                <code>416</code>
     *                <status><![CDATA[Not Found]]></status>
     *                <message><![CDATA[expected '>']]></message>
     *                <type><![CDATA[external]]></type>
     *                <exception><![CDATA[InvalidRequest]]></exception>
     *                <debug><![CDATA[not set]]></debug>
     *              </error>
     *          </errors>
     *      </response>
     *
     * @ApiDoc(
     *  resource="/api/{version}/sso-ws/application",
     *  description="secured: Adds multiple applications to sso-ws",
     *  statusCodes={
     *         200="Returned when successful",
     *         204="Returned when no content was send",
     *         400="Returned when an error occurred",
     *         401="Unauthorized - Wrong credentials",
     *         403="Returned when the user is not authorized to do this request",
     *         416="Returned when the request is malformed",
     *         500="Returned on internal server errors"
     *  },
     *  headers={
     *      {
     *        "name"="username",
     *        "description"="sso username",
     *        "required"=true
     *       },
     *      {
     *        "name"="useridentifier",
     *        "description"="sso useridentifier",
     *        "required"=true
     *       },
     *      {
     *        "name"="servicetoken",
     *        "description"="servicetoken",
     *        "required"=true
     *       }
     *  },
     *  tags={
     *      "stable" = "#000"
     *  },
     *  output="Sso\RestBundle\ModelRest\Response\UserWs\Version1\Application\Add\Response",
     *  views = { "sso-ws" }
     * )
     *
     * @Post("/v1/sso-ws/application/add", name="_sso-ws_v1", options={ "method_prefix" = true })
     * @Secure(roles="ROLE_API")
     * @param Request $request
     * @return View
     */
    public function postApplicationAction(Request $request)
    {
        return $this->woM->userWs()->version1()->application()->add()->init($request);
    }

    /**
     * *This service gets an application details from a sso-ws. The service expects a JSON or XML in body and a header authentication.*
     *
     * #Request (header)
     *
     *      Key                 | Value               | Description
     *      ------------------- | ------------------- | --------------------------------------
     *      username            |  (string)           | your username
     *      useridentifier      |  (string)           | your useridentifier for given username
     *      servicetoken        |  (string)           | servicetoken from ws
     *
     * ##Request
     *
     * **Request example JSON**
     *
     *       /api/v1/application/show.json?name=AppName
     *
     * **Request example XML**
     *
     *       /api/v1/application/show.xml?name=AppName
     *
     * ##Response success
     *
     * **Example JSON (body)**
     *
     *       {
     *           "code": 200,
     *           "status": "OK",
     *           "trackId": "73550392cf",
     *           "date": "2016-12-13T10:29:44+01:00",
     *           "scriptTimeSec": 9.9368,
     *           "application": {
     *               "code": 200,
     *               "status": "OK",
     *               "name": "testApplication",
     *               "description": "test description"
     *           }
     *       }
     *
     * **Example XML (body)**
     *
     *       <?xml version="1.0" encoding="UTF-8"?>
     *       <application>
     *           <code>200</code>
     *           <status><![CDATA[OK]]></status>
     *           <trackId><![CDATA[e6435e635c]]></trackId>
     *           <date><![CDATA[2016-12-13T10:56:38+01:00]]></date>
     *           <scriptTimeSec>1.2614</scriptTimeSec>
     *           <application>
     *               <code>200</code>
     *               <status><![CDATA[OK]]></status>
     *               <name><![CDATA[testApplication]]></name>
     *               <description><![CDATA[test description]]></description>
     *           </application>
     *       </application>
     *
     * ##Response error status 400
     *
     * **Example JSON (body)**
     *
     *       {
     *           "code": 400,
     *           "status": "Bad Request",
     *           "trackId": "2000b8780d",
     *           "date": "2016-12-13T10:59:21+01:00",
     *           "scriptTimeSec": 5.903,
     *           "application": {
     *               "code": 400,
     *               "status": "Bad Request",
     *               "errors": [
     *                   {
     *                       "code": 400,
     *                       "message": "ErrorCode: a001 ErrorRef: a011 ShortMessage: Invalid application",
     *                       "exception": "external"
     *                   }
     *               ]
     *           }
     *       }
     *
     * **Example XML (body)**
     *
     *       <?xml version="1.0" encoding="UTF-8"?>
     *       <application>
     *           <code>400</code>
     *           <status><![CDATA[Bad Request]]></status>
     *           <trackId><![CDATA[ea7c70fe1d]]></trackId>
     *           <date><![CDATA[2016-12-13T10:59:23+01:00]]></date>
     *           <scriptTimeSec>5.0225</scriptTimeSec>
     *           <application>
     *               <code>400</code>
     *               <status><![CDATA[Bad Request]]></status>
     *               <errors>
     *                   <error>
     *                       <code>400</code>
     *                       <message><![CDATA[ErrorCode: a001 ErrorRef: a011 ShortMessage: Invalid application]]></message>
     *                       <exception><![CDATA[external]]></exception>
     *                   </error>
     *               </errors>
     *           </application>
     *       </application>
     *
     * ##Response error status 401, 416 or 500
     *
     * **Example JSON (body)**
     *
     *       {
     *       "code": 416,
     *       "status": "Not Found",
     *       "trackId": "2d008cc4c5",
     *       "scriptTimeSec": 1.521,
     *       "count": 1,
     *       "errors": [
     *          {
     *          "code": 416,
     *          "status": "Not Found",
     *          "message": "no valid json string",
     *          "type": "external",
     *          "exception": "InvalidRequest",
     *          "debug": "not set"
     *          }
     *         ]
     *       }
     *
     * **Example XML (body)**
     *
     *      <?xml version="1.0" encoding="UTF-8"?>
     *      <response>
     *          <code>416</code>
     *          <status><![CDATA[Not Found]]></status>
     *          <trackId><![CDATA[16fe523df4]]></trackId>
     *          <scriptTimeSec>2.4459</scriptTimeSec>
     *          <count>1</count>
     *          <errors>
     *              <error>
     *                <code>416</code>
     *                <status><![CDATA[Not Found]]></status>
     *                <message><![CDATA[expected '>']]></message>
     *                <type><![CDATA[external]]></type>
     *                <exception><![CDATA[InvalidRequest]]></exception>
     *                <debug><![CDATA[not set]]></debug>
     *              </error>
     *          </errors>
     *      </response>
     *
     * @ApiDoc(
     *  resource="/api/{version}/sso-ws/application",
     *  description="secured: Shows an application details from a sso-ws",
     *  headers={
     *      {
     *        "name"="username",
     *        "description"="sso username",
     *        "required"=true
     *       },
     *      {
     *        "name"="useridentifier",
     *        "description"="sso useridentifier",
     *        "required"=true
     *       },
     *      {
     *        "name"="servicetoken",
     *        "description"="servicetoken",
     *        "required"=true
     *       }
     *   },
     *  statusCodes={
     *         200="Returned when successful",
     *         204="Returned when no content was send",
     *         400="Returned when an error occurred",
     *         401="Unauthorized - Wrong credentials",
     *         403="Returned when the user is not authorized to do this request",
     *         416="Returned when the request is malformed",
     *         500="Returned on internal server errors"
     *  },
     *  tags={
     *      "stable" = "#000"
     *  },
     *  parameters={
     *      {"name"="name", "dataType"="string", "required"=true, "description"="Application Name"}
     *  },
     *  output="Sso\RestBundle\ModelRest\Response\UserWs\Version1\Application\Show\Response",
     *  views = { "sso-ws" }
     * )
     * @Get("/v1/sso-ws/application/show", name="_sso-ws_v1", options={ "method_prefix" = true })
     * @param Request $request
     * @Secure(roles="ROLE_API")
     * @LsCache(cacheTTL=300, keyHeader=true)
     * @return View
     */
    public function getApplicationAction(Request $request)
    {
        return $this->woM->userWs()->version1()->application()->show()->init($request);
    }

    /**
     *
     * *This service deletes applications. The service expects a JSON or XML in body and a header authentication.*
     *
     * #Request (header)
     *
     *      Key                 | Value               | Description
     *      ------------------- | ------------------- | --------------------------------------
     *      username            |  (string)           | your username
     *      useridentifier      |  (string)           | your useridentifier for given username
     *      servicetoken        |  (string)           | servicetoken from ws
     *
     * ##Request
     *
     * **Request Example JSON**
     *
     *       {
     *           "applications": [
     *               {
     *                   "name": "testApplication"
     *               }
     *           ]
     *       }
     *
     * **Request Example XML**
     *
     *       <applicationDelete>
     *           <applications>
     *               <application>
     *                   <name>testApplication</name>
     *               </application>
     *           </applications>
     *       </applicationDelete>
     *
     * #Response success
     *
     * **Example JSON (body)**
     *
     *       {
     *           "code": 200,
     *           "status": "OK",
     *           "trackId": "eb5bf0b149",
     *           "date": "2016-11-15T12:40:43+01:00",
     *           "scriptTimeSec": 4.3627,
     *           "applications": [
     *               {
     *                   "code": 200,
     *                   "status": "OK",
     *                   "message": Application \"testApplication\" deleted
     *               }
     *           ]
     *       }
     *
     * **Example XML (body)**
     *
     *       <?xml version="1.0" encoding="UTF-8"?>
     *       <applicationDelete>
     *           <code>200</code>
     *           <status><![CDATA[OK]]></status>
     *           <trackId><![CDATA[14fc29cbdc]]></trackId>
     *           <date><![CDATA[2016-11-15T12:39:27+01:00]]></date>
     *           <scriptTimeSec>4.972</scriptTimeSec>
     *           <applications>
     *               <application>
     *                   <code>200</code>
     *                   <status><![CDATA[OK]]></status>
     *                   <message><![CDATA[Application "testApplication" deleted]]></message>
     *               </application>
     *           </applications>
     *       </applicationDelete>
     *
     * #Response error
     *
     * **Example JSON (body)**
     *
     *       {
     *           "code": 400,
     *           "status": "Bad Request",
     *           "trackId": "7c39d356e0",
     *           "date": "2016-11-15T12:30:31+01:00",
     *           "scriptTimeSec": 1.638,
     *           "applications": [
     *               {
     *                   "code": 400,
     *                   "status": "Bad Request",
     *                   "errors": [
     *                       {
     *                           "code": 400,
     *                           "message": "ErrorCode: a001 ErrorRef: a021 ShortMessage: Invalid application",
     *                           "exception": "external"
     *                       }
     *                   ]
     *               }
     *           ]
     *       }
     *
     * **Example XML (body)**
     *
     *       <?xml version="1.0" encoding="UTF-8"?>
     *       <applicationDelete>
     *           <code>400</code>
     *           <status><![CDATA[Bad Request]]></status>
     *           <trackId><![CDATA[5a7f343b34]]></trackId>
     *           <date><![CDATA[2016-11-15T12:28:46+01:00]]></date>
     *           <scriptTimeSec>1.8849</scriptTimeSec>
     *           <applications>
     *               <application>
     *                   <code>400</code>
     *                   <status><![CDATA[Bad Request]]></status>
     *                   <errors>
     *                       <error>
     *                           <code>400</code>
     *                           <message><![CDATA[ErrorCode: a001 ErrorRef: a021 ShortMessage: Invalid application]]></message>
     *                           <exception><![CDATA[external]]></exception>
     *                       </error>
     *                   </errors>
     *               </application>
     *           </applications>
     *       </applicationDelete>
     *
     * ##Response error status 401, 416 or 500
     *
     * **Example JSON (body)**
     *
     *       {
     *       "code": 416,
     *       "status": "Not Found",
     *       "trackId": "2d008cc4c5",
     *       "scriptTimeSec": 1.521,
     *       "count": 1,
     *       "errors": [
     *          {
     *          "code": 416,
     *          "status": "Not Found",
     *          "message": "no valid json string",
     *          "type": "external",
     *          "exception": "InvalidRequest",
     *          "debug": "not set"
     *          }
     *         ]
     *       }
     *
     * **Example XML (body)**
     *
     *      <?xml version="1.0" encoding="UTF-8"?>
     *      <response>
     *          <code>416</code>
     *          <status><![CDATA[Not Found]]></status>
     *          <trackId><![CDATA[16fe523df4]]></trackId>
     *          <scriptTimeSec>2.4459</scriptTimeSec>
     *          <count>1</count>
     *          <errors>
     *              <error>
     *                <code>416</code>
     *                <status><![CDATA[Not Found]]></status>
     *                <message><![CDATA[expected '>']]></message>
     *                <type><![CDATA[external]]></type>
     *                <exception><![CDATA[InvalidRequest]]></exception>
     *                <debug><![CDATA[not set]]></debug>
     *              </error>
     *          </errors>
     *      </response>
     *
     * @ApiDoc(
     *  resource="/api/{version}/sso-ws/application",
     *  description="secured: Deletes given applications",
     *  statusCodes={
     *         200="Returned when successful",
     *         204="Returned when no content was send",
     *         400="Returned when an error occurred",
     *         401="Unauthorized - Wrong credentials",
     *         403="Returned when the user is not authorized to do this request",
     *         416="Returned when the request is malformed",
     *         500="Returned on internal server errors"
     *  },
     *  headers={
     *      {
     *        "name"="username",
     *        "description"="sso username",
     *        "required"=true
     *       },
     *      {
     *        "name"="useridentifier",
     *        "description"="sso useridentifier",
     *        "required"=true
     *       },
     *      {
     *        "name"="servicetoken",
     *        "description"="servicetoken",
     *        "required"=true
     *       }
     *  },
     *  tags={
     *      "stable" = "#000"
     *  },
     *  output="Sso\RestBundle\ModelRest\Response\UserWs\Version1\Application\Delete\Response",
     *  views = { "sso-ws" }
     * )
     *
     * @Delete("/v1/sso-ws/application/delete", name="_sso-ws_v1", options={ "method_prefix" = true })
     * @Secure(roles="ROLE_API")
     * @param Request $request
     * @return View
     */
    public function deleteApplicationAction(Request $request)
    {
        return $this->woM->userWs()->version1()->application()->delete()->init($request);
    }
}
