<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2018 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\RestBundle\Controller\UserWs\Version1;

use Sso\RestBundle\Worker\Worker as Worker;
use FOS\RestBundle\Controller\Annotations\Delete;
use FOS\RestBundle\Controller\Annotations\Get;
use FOS\RestBundle\Controller\Annotations\NamePrefix;
use FOS\RestBundle\Controller\Annotations\Post;
use FOS\RestBundle\Controller\FOSRestController;
use FOS\RestBundle\View\View;
use JMS\DiExtraBundle\Annotation as DI;
use JMS\SecurityExtraBundle\Annotation\Secure;
use LifeStyle\Tools\RestAuthBundle\Security\Authentication\Token\UidentifyToken;
use Nelmio\ApiDocBundle\Annotation\ApiDoc;
use Symfony\Component\HttpFoundation\Request;
use LifeStyle\Tools\CachingBundle\Annotations\LsCache;

/**
 * @NamePrefix("api_rest_")
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package    Sso\RestBundle\Controller
 */
class UserController extends FOSRestController
{
    /**
     * @DI\Inject("rest_worker_manager")
     * @var Worker
     */
    protected $woM;

    /**
     * @var UidentifyToken
     */
    protected $userToken;

    /**
     * *This service adds a new users to the sso-ws. The service expects a JSON or XML in body and a header authentication.*
     *
     * #Request (header)
     *
     *      Key                 | Value               | Description
     *      ------------------- | ------------------- | --------------------------------------
     *      username            |  (string)           | your username
     *      useridentifier      |  (string)           | your useridentifier for given username
     *      servicetoken        |  (string)           | servicetoken from ws
     *
     * ##Request
     *
     * **Request example JSON**
     *
     *       {
     *           "users" : [
     *               {
     *                   "email" : "user@domain.tld",
     *                   "username" : "testuser",
     *                   "firstname" : "Tester",
     *                   "lastname" : "Testuser",
     *                   "password" : "xxxxxx",
     *                   "active" : 1,
     *                   "authId" : "24446666688888889",
     *                   "ldapSearchAttribute" : "",
     *                   "ldapSearchValue" : ""
     *               }
     *           ]
     *       }
     *
     * **Request example XML**
     *
     *       <userAdd>
     *         <users>
     *           <userType>
     *             <email><![CDATA[user@domain.tld]]></email>
     *             <username><![CDATA[testuser]]></username>
     *             <firstname><![CDATA[Tester]]></firstname>
     *             <lastname><![CDATA[Testuser]]></lastname>
     *             <password><![CDATA[xxxxxx]]></password>
     *             <active>1</active>
     *             <authId><![CDATA[24446666688888889]]></authId>
     *             <ldapSearchAttribute><![CDATA[]]></ldapSearchAttribute>
     *             <ldapSearchValue><![CDATA[]]></ldapSearchValue>
     *           </userType>
     *         </users>
     *       </userAdd>
     *
     * ##Response success
     *
     * **Example JSON (body)**
     *
     *       {
     *           "code": 200,
     *           "status": "Success",
     *           "trackId": "b5d53cc66a",
     *           "date": "2016-10-31T14:02:48+01:00",
     *           "scriptTimeSec": 2.0267,
     *           "users": [
     *               {
     *                   "code": 200,
     *                   "status": "OK",
     *                   "guid": "9f6b9dec64d06ba7ee7a4efa0e6b8b09",
     *                   "username": "testuser",
     *                   "email": "user@domain.tld",
     *                   "firstname": "Tester",
     *                   "lastname": "Testuser",
     *                   "active": 1,
     *                   "authId": "24446666688888889"
     *               }
     *           ]
     *       }
     *
     * **Example XML (body)**
     *
     *       <?xml version="1.0" encoding="UTF-8"?>
     *       <userAdd>
     *           <code>200</code>
     *           <status><![CDATA[OK]]></status>
     *           <trackId><![CDATA[0ad1c06d00]]></trackId>
     *           <date><![CDATA[2016-10-31T14:35:56+01:00]]></date>
     *           <scriptTimeSec>1.8142</scriptTimeSec>
     *           <users>
     *               <userType>
     *                   <code>200</code>
     *                   <status><![CDATA[OK]]></status>
     *                   <guid><![CDATA[02bc06f6a2d31f6683de904267872144]]></guid>
     *                   <username><![CDATA[testuser]]></username>
     *                   <email><![CDATA[user@domain.tld]]></email>
     *                   <firstname><![CDATA[Tester]]></firstname>
     *                   <lastname><![CDATA[Testuser]]></lastname>
     *                   <active>1</active>
     *                   <authId><![CDATA[24446666688888889]]></authId>
     *               </userType>
     *           </users>
     *       </userAdd>
     *
     * ##Response error status 400
     *
     * **Example JSON (body)**
     *
     *       {
     *           "code": 400,
     *           "status": "Bad Request",
     *           "trackId": "3113f4788d",
     *           "date": "2016-10-31T14:38:47+01:00",
     *           "scriptTimeSec": 1.6056,
     *           "users": [
     *               {
     *                   "code": 400,
     *                   "status": "Bad Request",
     *                   "errors": [
     *                       {
     *                       "code": 400,
     *                       "message": "ErrorCode: u10 ErrorRef: v001 ShortMessage: Invalid email-address",
     *                       "exception": "external"
     *                       }
     *                   ]
     *               }
     *           ]
     *       }
     *
     * **Example XML (body)**
     *
     *       <?xml version="1.0" encoding="UTF-8"?>
     *       <userAdd>
     *           <code>400</code>
     *           <status><![CDATA[Bad Request]]></status>
     *           <trackId><![CDATA[dd454a1d10]]></trackId>
     *           <date><![CDATA[2016-10-31T14:40:30+01:00]]></date>
     *           <scriptTimeSec>1.9725</scriptTimeSec>
     *           <users>
     *               <userType>
     *                   <code>400</code>
     *                   <status><![CDATA[Bad Request]]></status>
     *                   <errors>
     *                       <error>
     *                           <code>400</code>
     *                           <message><![CDATA[ErrorCode: u10 ErrorRef: v001 ShortMessage: Invalid email-address]]></message>
     *                           <exception><![CDATA[external]]></exception>
     *                       </error>
     *                   </errors>
     *               </userType>
     *           </users>
     *       </userAdd>
     *
     * ##Response error status 401, 416 or 500
     *
     * **Example JSON (body)**
     *
     *       {
     *       "code": 416,
     *       "status": "Not Found",
     *       "trackId": "2d008cc4c5",
     *       "scriptTimeSec": 1.521,
     *       "count": 1,
     *       "errors": [
     *          {
     *          "code": 416,
     *          "status": "Not Found",
     *          "message": "no valid json string",
     *          "type": "external",
     *          "exception": "InvalidRequest",
     *          "debug": "not set"
     *          }
     *         ]
     *       }
     *
     * **Example XML (body)**
     *
     *      <?xml version="1.0" encoding="UTF-8"?>
     *      <response>
     *          <code>416</code>
     *          <status><![CDATA[Not Found]]></status>
     *          <trackId><![CDATA[16fe523df4]]></trackId>
     *          <scriptTimeSec>2.4459</scriptTimeSec>
     *          <count>1</count>
     *          <errors>
     *              <error>
     *                <code>416</code>
     *                <status><![CDATA[Not Found]]></status>
     *                <message><![CDATA[expected '>']]></message>
     *                <type><![CDATA[external]]></type>
     *                <exception><![CDATA[InvalidRequest]]></exception>
     *                <debug><![CDATA[not set]]></debug>
     *              </error>
     *          </errors>
     *      </response>
     *
     * @ApiDoc(
     *  resource="/api/{version}/sso-ws/user",
     *  description="secured: Adds an user to a sso-ws",
     *  headers={
     *      {
     *        "name"="username",
     *        "description"="sso username",
     *        "required"=true
     *       },
     *      {
     *        "name"="useridentifier",
     *        "description"="sso useridentifier",
     *        "required"=true
     *       },
     *      {
     *        "name"="servicetoken",
     *        "description"="servicetoken",
     *        "required"=true
     *       }
     *   },
     *  statusCodes={
     *         200="Returned when successful",
     *         204="Returned when no content was send",
     *         400="Returned when an error occurred",
     *         401="Unauthorized - Wrong credentials",
     *         403="Returned when the user is not authorized to do this request",
     *         416="Returned when the request is malformed",
     *         500="Returned on internal server errors"
     *  },
     *  tags={
     *      "stable" = "#000"
     *  },
     *  output="Sso\RestBundle\ModelRest\Response\UserWs\Version1\User\Add\Response",
     *  views = { "sso-ws" }
     * )
     * @Post("/v1/sso-ws/user/add", name="_sso-ws_v1", options={ "method_prefix" = true })
     * @param Request $request
     * @Secure(roles="ROLE_API")
     * @return View
     */
    public function userAddAction(Request $request)
    {
        return $this->woM->userWs()->version1()->user()->add()->init($request);
    }

    /**
     * *This service gets a user from the user ws. The services requires a header authentication!*
     *
     * #Request (header)
     *
     *      Key                 | Value               | Description
     *      ------------------- | ------------------- | --------------------------------------
     *      username            |  (string)           | your username
     *      useridentifier      |  (string)           | your useridentifier for given username
     *      servicetoken        |  (string)           | servicetoken from ws
     *
     * ##Request
     *
     * **Request example JSON**
     *
     *      GET /api/v1/sso-ws/user/show.json?identifier[0]=UserToCheck1&identifier[1]=UserToCheck2
     *
     * **Request example XML**
     *
     *      GET /api/v1/sso-ws/user/show.xml?identifier[0]=UserToCheck1&identifier[1]=UserToCheck2
     *
     * ##Response success
     *
     * **Example JSON (body)**
     *
     *      {
     *          "code" : 200,
     *          "status" : "OK",
     *          "trackId" : "b7e6d36424",
     *          "date" : "2016-12-12T12:17:10+01:00",
     *          "scriptTimeSec" : 4.1672,
     *          "users" : [
     *              {
     *                  "code" : 200,
     *                  "status" : "OK",
     *                  "guid" : "sdfsdfsdfsdfsdfsdfsdfsdfsdfsf",
     *                  "username" : "sb@life-style.de",
     *                  "email" : "sb@life-style.de",
     *                  "firstname" : "John",
     *                  "lastname" : "Doe",
     *                  "active" : "1",
     *                  "authId" : "",
     *                  "ldapSearchAttributes" : "",
     *                  "ldapSearchValue" : ""
     *              }
     *          ]
     *      }
     *
     * **Example XML (body)**
     *
     *      <?xml version="1.0" encoding="UTF-8"?>
     *      <userShow>
     *          <code>200</code>
     *          <status><![CDATA[OK]]></status>
     *          <trackId><![CDATA[bc4a4b60e1]]></trackId>
     *          <date><![CDATA[2016-12-12T12:18:27+01:00]]></date>
     *          <scriptTimeSec>3.5844</scriptTimeSec>
     *          <users>
     *              <userType>
     *                  <code>200</code>
     *                  <status><![CDATA[OK]]></status>
     *                  <guid><![CDATA[asdfsdfsdfsdfsdfsdf]]></guid>
     *                  <username><![CDATA[sb@life-style.de]]></username>
     *                  <email><![CDATA[sb@life-style.de]]></email>
     *                  <firstname><![CDATA[John]]></firstname>
     *                  <lastname><![CDATA[Doe]]></lastname>
     *                  <active><![CDATA[1]]></active>
     *                  <authId><![CDATA[]]></authId>
     *                  <ldapSearchAttributes><![CDATA[]]></ldapSearchAttributes>
     *                  <ldapSearchValue><![CDATA[]]></ldapSearchValue>
     *              </userType>
     *          </users>
     *      </userShow>
     *
     * ##Response error status 400
     *
     * **Example JSON (body)**
     *
     *       {
     *           "code": 400,
     *           "status": "Bad Request",
     *           "trackId": "3113f4788d",
     *           "date": "2016-10-31T14:38:47+01:00",
     *           "scriptTimeSec": 1.6056,
     *           "users": [
     *               {
     *                   "code": 400,
     *                   "status": "Bad Request",
     *                   "errors": [
     *                       {
     *                       "code": 400,
     *                       "message": "ErrorCode: u001 ErrorRef: u011 ShortMessage: Invalid user",
     *                       "exception": "external"
     *                       }
     *                   ]
     *               }
     *           ]
     *       }
     *
     * **Example XML (body)**
     *
     *       <?xml version="1.0" encoding="UTF-8"?>
     *       <userShow>
     *           <code>400</code>
     *           <status><![CDATA[Bad Request]]></status>
     *           <trackId><![CDATA[dd454a1d10]]></trackId>
     *           <date><![CDATA[2016-10-31T14:40:30+01:00]]></date>
     *           <scriptTimeSec>1.9725</scriptTimeSec>
     *           <users>
     *               <userType>
     *                   <code>400</code>
     *                   <status><![CDATA[Bad Request]]></status>
     *                   <errors>
     *                       <error>
     *                           <code>400</code>
     *                           <message><![CDATA[ErrorCode: u001 ErrorRef: u011 ShortMessage: Invalid user]]></message>
     *                           <exception><![CDATA[external]]></exception>
     *                       </error>
     *                   </errors>
     *               </userType>
     *           </users>
     *       </userShow>
     *
     *
     * @ApiDoc(
     *  resource="/api/{version}/sso-ws/user",
     *  description="secured: get user scope from sso-ws",
     *  statusCodes={
     *         200="Returned when successful",
     *         400="Returned when an error occured",
     *         500="Returned on internal server errors"
     *  },
     *  headers={
     *      {"name"="username", "description"="sso username"},
     *      {"name"="useridentifier", "description"="sso useridentifier"},
     *      {"name"="servicetoken", "description"="servicetoken"}
     *  },
     *  tags={
     *      "stable" = "#000"
     *  },
     *  input="Sso\RestBundle\ModelRest\Request\UserWs\Version1\User\Show\Request",
     *  views = { "sso-ws" }
     * )
     *
     * @Get("/v1/sso-ws/user/show", name="_sso-ws_v1", options={ "method_prefix" = true })
     * @Secure(roles="ROLE_API")
     * @LsCache(cacheTTL=300, keyHeader=true)
     * @param Request $request
     * @return View
     */
    public function showUserAction(Request $request)
    {
        return $this->woM->userWs()->version1()->user()->show()->init($request);
    }

    /**
     * *This service deletes user from sso. The service expects a JSON or XML in body and a header authentication.*
     *
     * #Request (header)
     *
     *      Key                 | Value               | Description
     *      ------------------- | ------------------- | --------------------------------------
     *      username            |  (string)           | your username
     *      useridentifier      |  (string)           | your useridentifier for given username
     *      servicetoken        |  (string)           | servicetoken from ws
     *
     * ##Request
     *
     * **Request example JSON**
     *
     *      {
     *          "users": [
     *              { "identifier": "testUserIdentifier" },
     *              { "identifier": "testAnotherUserIdentifier" }
     *          ]
     *      }
     *
     * **Request example XML**
     *
     *      <userDelete>
     *          <users>
     *              <userType>
     *                  <identifier><![CDATA[testUserIdentifier]]></identifier>
     *              </userType>
     *              <userType>
     *                  <identifier><![CDATA[testAnotherUserIdentifier]]></identifier>
     *              </userType>
     *          </users>
     *      </userDelete>
     *
     * ##Response success
     *
     * **Example JSON (body)**
     *
     *      {
     *          "code": 200,
     *          "status": "OK",
     *          "trackId": "78ec6c2872",
     *          "date": "2016-11-25T15:09:07+01:00",
     *          "scriptTimeSec": 1.4164,
     *          "users": [
     *              {
     *                  "code": 200,
     *                  "status": "OK",
     *                  "message": "User \"testUserIdentifier\" deleted"
     *              }
     *          ]
     *      }
     *
     * **Example XML (body)**
     *
     *      <?xml version="1.0" encoding="UTF-8"?>
     *      <userDelete>
     *          <code>200</code>
     *          <status><![CDATA[OK]]></status>
     *          <trackId><![CDATA[1c46e01940]]></trackId>
     *          <date><![CDATA[2016-11-25T10:10:56+01:00]]></date>
     *          <scriptTimeSec>2.1386</scriptTimeSec>
     *          <users>
     *              <userType>
     *                  <code>200</code>
     *                  <status><![CDATA[OK]]></status>
     *                  <message><![CDATA[User "testUserIdentifier" deleted]]></message>
     *              </userType>
     *          </users>
     *      </userDelete>
     *
     * ##Response error status 400
     *
     * **Example JSON (body)**
     *
     *       {
     *           "code": 400,
     *           "status": "Bad Request",
     *           "trackId": "3113f4788d",
     *           "date": "2016-10-31T14:38:47+01:00",
     *           "scriptTimeSec": 1.6056,
     *           "users": [
     *               {
     *                   "code": 400,
     *                   "status": "Bad Request",
     *                   "errors": [
     *                       {
     *                          "code": 400,
     *                          "message": "ErrorCode: u10 ErrorRef: v001 ShortMessage: Invalid user",
     *                          "exception": "external"
     *                       }
     *                   ]
     *               }
     *           ]
     *       }
     *
     * **Example XML (body)**
     *
     *      <?xml version="1.0" encoding="UTF-8"?>
     *      <userDelete>
     *          <code>400</code>
     *          <status><![CDATA[Bad request]]></status>
     *          <trackId><![CDATA[1c46e01940]]></trackId>
     *          <date><![CDATA[2016-11-25T10:10:56+01:00]]></date>
     *          <scriptTimeSec>2.1386</scriptTimeSec>
     *          <users>
     *              <userType>
     *                  <code>400</code>
     *                  <status><![CDATA[Bad Request]]></status>
     *                  <errors>
     *                      <error>
     *                          <code>400</code>
     *                          <message><![CDATA[ErrorCode: u001 ErrorRef: u041 ShortMessage: Invalid user]]></message>
     *                          <exception><![CDATA[external]]></exception>
     *                      </error>
     *                  </errors>
     *              </userType>
     *          </users>
     *      </userDelete>
     *
     * ##Response error status 401, 416 or 500
     *
     * **Example JSON (body)**
     *
     *       {
     *       "code": 416,
     *       "status": "Not Found",
     *       "trackId": "2d008cc4c5",
     *       "scriptTimeSec": 1.521,
     *       "count": 1,
     *       "errors": [
     *          {
     *          "code": 416,
     *          "status": "Not Found",
     *          "message": "no valid json string",
     *          "type": "external",
     *          "exception": "InvalidRequest",
     *          "debug": "not set"
     *          }
     *         ]
     *       }
     *
     * **Example XML (body)**
     *
     *      <?xml version="1.0" encoding="UTF-8"?>
     *      <response>
     *          <code>416</code>
     *          <status><![CDATA[Not Found]]></status>
     *          <trackId><![CDATA[16fe523df4]]></trackId>
     *          <scriptTimeSec>2.4459</scriptTimeSec>
     *          <count>1</count>
     *          <errors>
     *              <error>
     *                <code>416</code>
     *                <status><![CDATA[Not Found]]></status>
     *                <message><![CDATA[expected '>']]></message>
     *                <type><![CDATA[external]]></type>
     *                <exception><![CDATA[InvalidRequest]]></exception>
     *                <debug><![CDATA[not set]]></debug>
     *              </error>
     *          </errors>
     *      </response>
     *
     * @ApiDoc(
     *  resource="/api/{version}/sso-ws/user",
     *  description="secured: deletes user from sso-ws",
     *  statusCodes={
     *         200="Returned when successful",
     *         204="Returned when no content was send",
     *         400="Returned when an error occurred",
     *         401="Unauthorized - Wrong credentials",
     *         403="Returned when the user is not authorized to do this request",
     *         416="Returned when the request is malformed",
     *         500="Returned on internal server errors"
     *  },
     *  headers={
     *      {
     *        "name"="username",
     *        "description"="sso username",
     *        "required"=true
     *       },
     *      {
     *        "name"="useridentifier",
     *        "description"="sso useridentifier",
     *        "required"=true
     *       },
     *      {
     *        "name"="servicetoken",
     *        "description"="servicetoken",
     *        "required"=true
     *       }
     *   },
     *  tags={
     *      "stable" = "#000"
     *  },
     *  output="Sso\RestBundle\ModelRest\Response\UserWs\Version1\User\Delete\Response",
     *  views = { "sso-ws" }
     * )
     *
     * @Delete("/v1/sso-ws/user/delete", name="_sso-ws_v1", options={ "method_prefix" = true })
     * @Secure(roles="ROLE_API")
     * @param Request $request
     * @return View
     */
    public function deleteUserAction(Request $request)
    {
        return $this->woM->userWs()->version1()->user()->delete()->init($request);
    }
}
