<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2018 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\RestBundle\Controller\UserWs\Version2;

use Sso\RestBundle\Worker\Worker as Worker;
use FOS\RestBundle\Controller\Annotations\Get;
use FOS\RestBundle\Controller\Annotations\NamePrefix;
use FOS\RestBundle\Controller\Annotations\QueryParam;
use FOS\RestBundle\View\View;
use JMS\DiExtraBundle\Annotation as DI;
use JMS\SecurityExtraBundle\Annotation\Secure;
use Nelmio\ApiDocBundle\Annotation\ApiDoc;
use Symfony\Component\HttpFoundation\Request;
use LifeStyle\Tools\CachingBundle\Annotations\LsCache;

/**
 * @NamePrefix("api_rest_")
 *
 * Class UserApplicationRoleController
 * @package Sso\RestBundle\Controller\UserWs\Version2
 */
class UserApplicationRoleController
{
    /**
     * @DI\Inject("rest_worker_manager")
     * @var Worker
     */
    protected $woM;

    /**
     * *This service shows user application role from sso-ws.*
     *
     * #Request (header)
     *
     *      Key                 | Value               | Description
     *      ------------------- | ------------------- | --------------------------------------
     *      username            |  (string)           | your username
     *      useridentifier      |  (string)           | your useridentifier for given username
     *      servicetoken        |  (string)           | your servicetoken
     *
     * ##Request
     *
     * **Request example JSON**
     *
     *          /api/v2/user/application/role/show.json?identifier=someUser&applicationName=someApplication
     *
     * **Request example XML**
     *
     *          /api/v2/user/application/role/show.xml?identifier=someUser&applicationName=someApplication
     *
     * ##Response success
     *
     * **Example (header)**
     *
     *          200 OK
     *
     * **Example JSON (body)**
     *
     *      {
     *          "code": 200,
     *          "status": "OK",
     *          "trackId": "d363f0905b",
     *          "date": "2016-12-02T09:40:49+01:00",
     *          "scriptTimeSec": 1.0973,
     *          "userTypes": [
     *              {
     *                  "code": 200,
     *                  "status": "Success",
     *                  "guid": "7c1388f802d85f7fabdba84b4dff823f",
     *                  "username": "john@doe.com",
     *                  "email": "john@doe.com",
     *                  "firstname": "John",
     *                  "lastname": "Doe",
     *                  "active": "1",
     *                  "deleted": "0",
     *                  "createdAt": "2016-09-06T14:06:45+02:00",
     *                  "lastLoginAt": "2016-12-02T09:22:43+01:00",
     *                  "updatedAt": "2016-11-07T10:12:56+01:00",
     *                  "authId": "",
     *                  "ldapSearchAttributes": "",
     *                  "ldapSearchValue": "",
     *                  "mfaEnabled": "0",
     *                  "applicationType": {
     *                  "name": "SsoAdmin",
     *                  "active": "0",
     *                  "roleType": [
     *                      {
     *                      "name": "HrCoreRegisterUser",
     *                      "active": "1"
     *                      },
     *                      {
     *                      "name": "WorkLocations",
     *                      "active": "1"
     *                      }
     *                    ]
     *                  }
     *              }
     *          ]
     *      }
     *
     * **Example XML (body)**
     *
     *       <?xml version="1.0" encoding="UTF-8"?>
     *       <userRoleShow>
     *          <code>200</code>
     *          <status><![CDATA[OK]]></status>
     *          <trackId><![CDATA[3d0055da3b]]></trackId>
     *          <date><![CDATA[2016-12-02T09:42:46+01:00]]></date>
     *          <scriptTimeSec>2.5696</scriptTimeSec>
     *          <userTypes>
     *            <userType>
     *              <code>200</code>
     *              <status><![CDATA[Success]]></status>
     *              <guid><![CDATA[7c1388f802d85f7fabdba84b4dff823f]]></guid>
     *              <username><![CDATA[john@doe.com]></username>
     *              <email><![CDATA[john@doe.com]]></email>
     *              <firstname><![CDATA[John]]></firstname>
     *              <lastname><![CDATA[Doe]]></lastname>
     *              <active><![CDATA[1]]></active>
     *              <deleted><![CDATA[0]]></deleted>
     *              <createdAt><![CDATA[2016-09-06T14:06:45+02:00]]></createdAt>
     *              <lastLoginAt><![CDATA[2016-12-02T09:22:43+01:00]]></lastLoginAt>
     *              <updatedAt><![CDATA[2016-11-07T10:12:56+01:00]]></updatedAt>
     *              <authId><![CDATA[]]></authId>
     *              <ldapSearchAttributes><![CDATA[]]></ldapSearchAttributes>
     *              <ldapSearchValue><![CDATA[]]></ldapSearchValue>
     *              <mfaEnabled><![CDATA[0]]></mfaEnabled>
     *              <applicationType>
     *                  <name><![CDATA[SsoAdmin]]></name>
     *                  <active><![CDATA[0]]></active>
     *                  <roleType>
     *                    <name><![CDATA[HrCoreRegisterUser]]></name>
     *                    <active><![CDATA[1]]></active>
     *                  </roleType>
     *                  <roleType>
     *                    <name><![CDATA[WorkLocations]]></name>
     *                    <active><![CDATA[1]]></active>
     *                  </roleType>
     *              </applicationType>
     *            </userType>
     *          </userTypes>
     *       </userRoleShow>
     *
     * ##Response error
     *
     * **Example (header)**
     *
     *          400 Bad Request
     *
     * **Example JSON (body)**
     *
     *          {
     *              "code": 400,
     *              "status": "Bad Request",
     *              "trackId": "43f92f152d",
     *              "date": "2016-11-23T12:17:37+01:00",
     *              "scriptTimeSec": 1.5656,
     *              "userRoles": [
     *                  {
     *                      "code": 404,
     *                      "status": "Not Found",
     *                      "errors": [
     *                          {
     *                              "code": 400,
     *                              "message": "ErrorCode: u001 ErrorRef: u2016 ShortMessage: Invalid user",
     *                              "exception": "external"
     *                          }
     *                      ]
     *                  }
     *              ]
     *          }
     *
     * **Example XML (body)**
     *
     *          <?xml version="1.0" encoding="UTF-8"?>
     *          <userRoleShow>
     *              <code>400</code>
     *              <status>
     *                  <![CDATA[Bad Request]]>
     *              </status>
     *              <trackId>
     *                  <![CDATA[49162b1753]]>
     *              </trackId>
     *              <date>
     *                  <![CDATA[2016-11-23T12:16:32+01:00]]>
     *              </date>
     *              <scriptTimeSec>5.2922</scriptTimeSec>
     *              <userRoles>
     *                  <userType>
     *                      <code>404</code>
     *                      <status>
     *                          <![CDATA[Not Found]]>
     *                      </status>
     *                      <errors>
     *                          <error>
     *                              <code>400</code>
     *                              <message>
     *                                  <![CDATA[ErrorCode: u001 ErrorRef: u2016 ShortMessage: Invalid user]]>
     *                              </message>
     *                              <exception>
     *                                  <![CDATA[external]]>
     *                              </exception>
     *                          </error>
     *                      </errors>
     *                  </userType>
     *              </userRoles>
     *          </userRoleShow>
     *
     * @ApiDoc(
     *  resource = "/api/{version}/sso-ws/userApplicationRole",
     *  description = "secured: show user application role from sso-ws",
     *  statusCodes = {
     *      200="OK",
     *      400="Bad Request",
     *      401="Unauthorized",
     *      403="Forbidden",
     *      404="Not Found",
     *      500="Internal Server Error"
     *  },
     *  headers = {
     *      {
     *          "name" = "username",
     *          "description" = "sso username",
     *          "required" = "true"
     *      },
     *      {
     *          "name" = "useridentifier",
     *          "description" = "sso useridentifier",
     *          "required" = "true"
     *      },
     *      {
     *          "name" = "servicetoken",
     *          "description" = "servicetoken",
     *          "required" = "true"
     *      }
     *  },
     *  tags = {
     *      "untested"
     *  },
     *  views = {"sso-ws"}
     * )
     *
     * @Get("/v2/sso-ws/userApplicationRole/show", name="_sso-ws_v2", options={ "method_prefix" = true })
     * @QueryParam(name="identifier", nullable=false, strict=true, description="any identifier of an user (e-mail, guid, username, ...")
     * @QueryParam(name="applicationName", nullable=false, strict=true, description="application name")
     *
     * @Secure(roles="ROLE_API")
     * @LsCache(cacheTTL=300, keyHeader=true)
     * @param Request $request
     * @return View
     */
    public function getUserApplicationRoleAction(Request $request)
    {
        return $this->woM->userWs()->version2()->userApplicationRole()->show()->init($request);
    }
}
